#' Information-based generalized Robinson-Foulds distance between two trees
#'
#' Functions reporting the distances or similarities between pairs of trees,
#' based on information-theoretic concepts.
#' 
#'
#' Each partition in a tree can be viewed either as 
#' - (a) a statement that the 'true' tree is one of those that splits the 
#'   taxa as specified;
#' - (b) a statement that the taxa are subdivided into the two groups specified.
#' 
#' The former concept corresponds to the concept of phylogenetic 
#' information, and views the information content of a pair of partitions
#' as relating to the proportion of phylogenetic trees that are consistent
#' with both partitions, giving rise to the
#' Mutual Arboreal Information similarity measure (`MutualArborealInfo`), and 
#' the complementary Variation of Arboreal Information distance metric
#' (`VariationOfArborealInfo`).
#' 
#' The latter sees the information content of a pair of partitions as relating
#' to the proportion of all possible pairings that are at least as similar
#' (measured using Meila's (2007) the variation of information) as the pairing
#' in question, giving rise to the Mutual Clustering Information similarity
#' measure (`MutualClusterInfo`).
#' 
#' A tree similarity measure is generated by finding an optimal matching that
#' maximises the total information in common between a partition on one tree
#' and its pair on a second, considering all possible ways to pair partitions 
#' between trees (including leaving a partition unpaired).
#' 
#' The returned tree similarity measures state the amount of information, 
#' in bits, that the partitions in two trees hold in common 
#' when they are optimally matched, following Smith (forthcoming).  
#' The complementary tree distance measures state how much information is 
#' different in the partitions of two trees, under an optimal matching.
#' 
#' @param tree1,tree2 Trees of class `phylo`, with tips labelled identically,
#' or lists of such trees to undergo pairwise comparison.
#' 
#' @param reportMatching Logical specifying whether to return the clade
#' matchings as an attribute of the score.
#'
#' @return If `reportMatching = FALSE`, the functions return a numeric 
#' vector specifying the requested similarities or differences.
#' 
#' If `reportMatching = TRUE`, the functions additionally return 
#'  
#' @examples {
#'   tree1 <- ape::read.tree(text='((((a, b), c), d), (e, (f, (g, h))));')
#'   tree2 <- ape::read.tree(text='(((a, b), (c, d)), ((e, f), (g, h)));')
#'   tree3 <- ape::read.tree(text='((((h, b), c), d), (e, (f, (g, a))));')
#'   
#'   # Best possible score is obtained by matching a tree with itself
#'   VariationOfArborealInfo(tree1, tree1) # 0, by definition
#'   MutualArborealInfo(tree1, tree1)
#'   
#'   # Best possible score is a function of tree shape; the partitions within
#'   # balanced trees are more independent and thus contain less information
#'   MutualArborealInfo(tree2, tree2)
#'   
#'   # How similar are two trees?
#'   MutualArborealInfo(tree1, tree2)
#'   VariationOfArborealInfo(tree1, tree2)
#'   VariationOfArborealInfo(tree2, tree1) # Identical, by symmetry
#'   
#'   
#'   # Maximum possible score for Cluster information is independent
#'   # of tree shape, as every possible pairing is considered
#'   MutualClusterInfo(tree1, tree1)
#'   MutualClusterInfo(tree2, tree2)
#'   
#'   # It is thus easier to interpret the value of
#'   MutualClusterInfo(tree1, tree2)
#'   # Although it may not be possible to find a tree pair with zero mutual
#'   # cluster info.
#'   
#'   # Every partition in tree1 is contradicted by every partition in tree3
#'   # Non-arboreal matches contain clustering, but not phylogenetic, information
#'   MutualArborealInfo(tree1, tree3) # = 0
#'   MutualClusterInfo(tree1, tree3) # > 0
#'   
#' }
#' 
#' @references {
#'  * \insertRef{Meila2007}{TreeSearch}
#'  * \insertRef{SmithDist}{TreeSearch}
#'  * \insertRef{Vinh2010}{TreeSearch}
#' }
#' 
#' @author Martin R. Smith
#' 
#' @concept Tree distance
#' @importFrom clue solve_LSAP
#' @export
MutualArborealInfo <- function (tree1, tree2, reportMatching = FALSE) {
  CalculateTreeDistance(MutualArborealInfoSplits, tree1, tree2, 
                        reportMatching=reportMatching)
}

#' @describeIn MutualArborealInfo Variation of phylogenetic information between two trees
#' @export
VariationOfArborealInfo <- function (tree1, tree2, reportMatching = FALSE) {
  CalculateTreeDistance(VariationOfArborealInfoSplits, tree1, tree2, 
                        reportMatching=reportMatching)
}

#' @describeIn MutualArborealInfo Mutual clustering information between two trees
#' @export
MutualClusterInfo <- function (tree1, tree2,
                                  reportMatching = FALSE,
                                  bestMatchOnly = TRUE) {
  CalculateTreeDistance(MutualClusterInfoSplits, tree1, tree2, 
                        reportMatching=reportMatching, bestMatchOnly=bestMatchOnly)
}

#' Nye et al. (2006) tree comparison
#' 
#' Implements the tree comparison metric of Nye _et al_. (2006).
#' 
#' @inheritParams MutualArborealInfo
#' 
#' @references \insertRef{Nye2006}{TreeSearch}
#' @concept Tree distance
#' 
#' @author Martin R. Smith
#' @export
NyeTreeSimilarity <- function (tree1, tree2,
                             reportMatching = FALSE) {
  CalculateTreeDistance(NyeSplitSimilarity, tree1, tree2, reportMatching)
}

#' Matching Split Distance
#' 
#' Implements the Matching Split Distance for unrooted binary phylogenetic 
#' trees of Bogdanowicz and Giaro (2012).
#' 
#' @inheritParams MutualArborealInfo
#' 
#' @references \insertRef{Bogdanowicz2012}{TreeSearch}
#' @concept Tree distance
#' 
#' @author Martin R. Smith
#' @export
MatchingSplitDistance <- function (tree1, tree2,
                             reportMatching = FALSE) {
  CalculateTreeDistance(MatchingSplitDistanceSplits, tree1, tree2, reportMatching)
}

#' Wrapper for tree distance calculations
#' 
#' Calls tree distance functions from trees or lists of trees
#' 
#' @inheritParams MutualArborealInfo
#' @param Func Tree distance function.
#' @param \dots Additional arguments to `Func`.
#' 
#' @author Martin R. Smith
#' @keywords internal
#' @export
CalculateTreeDistance <- function (Func, tree1, tree2, reportMatching, ...) {
  if (class(tree1) == 'phylo') {
    if (class(tree2) == 'phylo') {
      if (length(setdiff(tree1$tip.label, tree2$tip.label)) > 0) {
        stop("Tree tips must bear identical labels")
      }
      
      Func(Tree2Splits(tree1), Tree2Splits(tree2), reportMatching, ...)
    } else {
      splits1 <- Tree2Splits(tree1)
      vapply(tree2, 
             function (tr2) Func(splits1, Tree2Splits(tr2), ...),
             double(1))
    }
  } else {
    if (class(tree2) == 'phylo') {
      splits1 <- Tree2Splits(tree2)
      vapply(tree1, 
             function (tr2) Func(splits1, Tree2Splits(tr2), ...),
             double(1))
    } else {
      splits1 <- lapply(tree1, Tree2Splits)
      splits2 <- lapply(tree2, Tree2Splits)
      matrix(mapply(Func, rep(splits1, each=length(splits2)), splits2),
             length(splits2), length(splits1),
             dimnames = list(names(tree2), names(tree1)), ...)
    }
  }
}


#' @describeIn MutualArborealInfo Takes splits instead of trees
#' @export
MutualArborealInfoSplits <- function (splits1, splits2, reportMatching = FALSE) {
  
  dimSplits1 <- dim(splits1)
  dimSplits2 <- dim(splits2)
  nTerminals <- dimSplits1[1]
  if (dimSplits2[1] != nTerminals) {
    stop("Split rows must bear identical labels")
  }
  lnUnrootedN <- LnUnrooted.int(nTerminals)
  
  if (dimSplits1[2] < dimSplits2[2]) {
    tmp <- splits1
    splits1 <- splits2
    splits2 <- tmp
    
    tmp <- dimSplits1
    dimSplits1 <- dimSplits2
    dimSplits2 <- tmp
  }
  
  taxonNames <- rownames(splits1) 
  
  if (!is.null(taxonNames)) {
    splits2 <- unname(splits2[rownames(splits1), , drop=FALSE])
    splits1 <- unname(splits1) # split1[split2] faster without names
  }
  
  nSplits1 <- dimSplits1[2]
  nSplits2 <- dimSplits2[2]
  inSplit1 <- colSums(splits1)
  inSplit2 <- colSums(splits2)
  notInSplit1 <- nTerminals - inSplit1 # TODO delete, and remove where used below?
  notInSplit2 <- nTerminals - inSplit2
  
  OneOverlap <- function(A1, A2) {
    if (A1 == A2) {
      # Return:
      LnRooted.int(A1) + LnRooted.int(nTerminals - A2)
    } else {
      if (A1 < A2) {
        tmp <- A2
        A2 <- A1
        A1 <- tmp
      }
      # Return:
      LnRooted.int(A1) + LnRooted.int(nTerminals - A2) - LnRooted.int(A1 - A2 + 1L) 
    }
  }
  
  pairScores <- matrix((mapply(function(i, j) {
    split1 <- splits1[, i]
    split2 <- splits2[, j]
    
    if (all(oneAndTwo <- split1[split2]) ||
        all(notOneNotTwo <- !split1[!split2])) {
      OneOverlap(inSplit1[i], inSplit2[j])
      
    } else if (all(notOneAndTwo <- !split1[split2]) ||
               all(oneNotTwo <- split1[!split2])) {
      OneOverlap(inSplit1[i], notInSplit2[j])
      
    } else {
      #in1 <- inSplit1[i]
      #out1 <- notInSplit1[i]
      #in2 <- inSplit2[j]
      #out2 <- notInSplit2[j]
      #splitTwoMoreEven <- min(in1, out1) < min(in2, out2)
      #contradictions <- c(min(sum(oneAndTwo), sum(notOneNotTwo)), 
      #                    min(sum(notOneAndTwo), sum(oneNotTwo)))
      #contradictionToFix <- which.min(contradictions)
      #
      #infoGainedBySwap <- if (contradictionToFix == 1L) {
      #  OneOverlap(in1, out2)
      #} else {
      #  OneOverlap(in1, in2)
      #}
      #
      #contradictionSize <- contradictions[contradictionToFix]
      #infoLostBySwap <- if (splitTwoMoreEven) {
      #  lchoose(in1, contradictionSize) + lchoose(out1, contradictionSize)
      #} else {
      #  lchoose(in2, contradictionSize) + lchoose(out2, contradictionSize)
      #}
      #c(SwapGain = infoGainedBySwap, SwapLoss = infoLostBySwap,
      #  SwapNet = infoGainedBySwap - infoLostBySwap)
      #
      #
      #contradictions <- c(sum(oneAndTwo), sum(notOneNotTwo), 
      #                    sum(notOneAndTwo), sum(oneNotTwo))
      lnUnrootedN
    }
  }, rep(seq_len(nSplits1), each=nSplits2), seq_len(nSplits2)
  ) - lnUnrootedN) / -log(2), nSplits2, nSplits1)
  
  if (nSplits2 == 1) {
    max(pairScores)
  } else {
    optimalMatching <- solve_LSAP(pairScores, TRUE)
    
    # Return:
    ret <- sum(pairScores[matrix(c(seq_along(optimalMatching), optimalMatching), ncol=2L)])
    if (reportMatching) {
      attr(ret, 'matching') <- optimalMatching
      attr(ret, 'pairScores') <- pairScores
      ret
    } else {
      ret
    }
  }
}


#' @describeIn MutualArborealInfo Calculate variation of arboreal information from splits
#' @export
VariationOfArborealInfoSplits <- function (splits1, splits2, reportMatching = FALSE) {
  
  dimSplits1 <- dim(splits1)
  dimSplits2 <- dim(splits2)
  nTerminals <- dimSplits1[1]
  if (dimSplits2[1] != nTerminals) {
    stop("Split rows must bear identical labels")
  }
  lnUnrootedN <- LnUnrooted.int(nTerminals)
  
  if (dimSplits1[2] < dimSplits2[2]) {
    tmp <- splits1
    splits1 <- splits2
    splits2 <- tmp
    
    tmp <- dimSplits1
    dimSplits1 <- dimSplits2
    dimSplits2 <- tmp
  }
  
  taxonNames <- rownames(splits1) 
  
  if (!is.null(taxonNames)) {
    splits2 <- unname(splits2[rownames(splits1), , drop=FALSE])
    splits1 <- unname(splits1) # split1[split2] faster without names
  }
  
  nSplits1 <- dimSplits1[2]
  nSplits2 <- dimSplits2[2]
  inSplit1 <- colSums(splits1)
  inSplit2 <- colSums(splits2)
  notInSplit1 <- nTerminals - inSplit1
  notInSplit2 <- nTerminals - inSplit2
  
  logTrees1 <- LnRooted(inSplit1) + LnRooted(notInSplit1)
  logTrees2 <- LnRooted(inSplit2) + LnRooted(notInSplit2)
  
  OneOverlap <- function(A1, A2) {
    if (A1 == A2) {
      # Return:
      LnRooted.int(A1) + LnRooted.int(nTerminals - A2)
    } else {
      if (A1 < A2) {
        tmp <- A2
        A2 <- A1
        A1 <- tmp
      }
      # Return:
      LnRooted.int(A1) + LnRooted.int(nTerminals - A2) - LnRooted.int(A1 - A2 + 1L) 
    }
  }
  
  pairScores <- matrix((mapply(function(i, j) {
    split1 <- splits1[, i]
    split2 <- splits2[, j]
    
    logMutualTrees <- 
      if (all(oneAndTwo <- split1[split2]) ||
          all(notOneNotTwo <- !split1[!split2])) {
        OneOverlap(inSplit1[i], inSplit2[j])
        
      } else if (all(notOneAndTwo <- !split1[split2]) ||
                 all(oneNotTwo <- split1[!split2])) {
        OneOverlap(inSplit1[i], notInSplit2[j])
        
      } else {
        lnUnrootedN
      }
    logTrees1[i] + logTrees2[j] - logMutualTrees - logMutualTrees
  }, rep(seq_len(nSplits1), each=nSplits2), seq_len(nSplits2)
  )) / -log(2), nSplits2, nSplits1)
  
  
  
  if (nSplits2 == 1) {
    min(pairScores)
  } else {
    optimalMatching <- solve_LSAP(pairScores, FALSE)
    
    # Return:
    ret <- sum(pairScores[matrix(c(seq_along(optimalMatching), optimalMatching), ncol=2L)])
    if (reportMatching) {
      attr(ret, 'matching') <- optimalMatching
      attr(ret, 'pairScores') <- pairScores
      ret
    } else {
      ret
    }
  }
}

#' @describeIn NyeTreeSimilarity Takes splits instead of trees
#' @inheritParams MutualArborealInfoSplits
#' @export
NyeSplitSimilarity <- function (splits1, splits2, reportMatching = FALSE) {
  
  dimSplits1 <- dim(splits1)
  dimSplits2 <- dim(splits2)
  nTerminals <- dimSplits1[1]
  if (dimSplits2[1] != nTerminals) {
    stop("Split rows must bear identical labels")
  }
  
  if (dimSplits1[2] < dimSplits2[2]) {
    tmp <- splits1
    splits1 <- splits2
    splits2 <- tmp
    
    tmp <- dimSplits1
    dimSplits1 <- dimSplits2
    dimSplits2 <- tmp
  }
  
  taxonNames <- rownames(splits1) 
  
  if (!is.null(taxonNames)) {
    splits2 <- unname(splits2[rownames(splits1), , drop=FALSE])
    splits1 <- unname(splits1) # split1[split2] faster without names
  }
  
  nSplits1 <- dimSplits1[2]
  nSplits2 <- dimSplits2[2]
  
  Ars <- function (pir, pjs) {
    sum(pir[pjs]) / sum(pir | pjs)
  }
  
  pairScores <- matrix((mapply(function(i, j) {
    splitI0 <- splits1[, i]
    splitJ0 <- splits2[, j]
    splitI1 <- !splitI0
    splitJ1 <- !splitJ0
    
    max(
      min(Ars(splitI0, splitJ0), Ars(splitI1, splitJ1)),
      min(Ars(splitI0, splitJ1), Ars(splitI1, splitJ0))
    )
      
  }, rep(seq_len(nSplits1), each=nSplits2), seq_len(nSplits2)
  )), nSplits2, nSplits1)
  
  if (nSplits2 == 1) {
    min(pairScores)
  } else {
    optimalMatching <- solve_LSAP(pairScores, TRUE)
    
    # Return:
    ret <- sum(pairScores[matrix(c(seq_along(optimalMatching), optimalMatching), ncol=2L)])
    if (reportMatching) {
      attr(ret, 'matching') <- optimalMatching
      attr(ret, 'pairScores') <- pairScores
      ret
    } else {
      ret
    }
  }
}

#' @describeIn MatchingSplitDistance Takes splits instead of trees
#' @inheritParams MutualArborealInfoSplits
#' @export
MatchingSplitDistanceSplits <- function (splits1, splits2, reportMatching = FALSE) {
  
  dimSplits1 <- dim(splits1)
  dimSplits2 <- dim(splits2)
  nTerminals <- dimSplits1[1]
  if (dimSplits2[1] != nTerminals) {
    stop("Split rows must bear identical labels")
  }
  
  if (dimSplits1[2] < dimSplits2[2]) {
    tmp <- splits1
    splits1 <- splits2
    splits2 <- tmp
    
    tmp <- dimSplits1
    dimSplits1 <- dimSplits2
    dimSplits2 <- tmp
  }
  
  taxonNames <- rownames(splits1) 
  
  if (!is.null(taxonNames)) {
    splits2 <- unname(splits2[rownames(splits1), , drop=FALSE])
    splits1 <- unname(splits1) # split1[split2] faster without names
  }
  
  nSplits1 <- dimSplits1[2]
  nSplits2 <- dimSplits2[2]
  
  SymmetricDifference <- function (A, B) {
    (A & !B) | (!A & B)
    #TODO: Assert: These will always be the same size
    #Therefore: Improve efficiency by only calculating one, and not dividing by 2
  }
  
  pairScores <- matrix((mapply(function(i, j) {
    A1 <- splits1[, i]
    A2 <- splits2[, j]
    B1 <- !A1
    B2 <- !A2
    
    min(
      sum(SymmetricDifference(A1, A2), SymmetricDifference(B1, B2)),
      sum(SymmetricDifference(A1, A2), SymmetricDifference(B1, B2))
    ) / 2L
      
  }, rep(seq_len(nSplits1), each=nSplits2), seq_len(nSplits2)
  )), nSplits2, nSplits1)
  
  if (nSplits2 == 1) {
    min(pairScores)
  } else {
    optimalMatching <- solve_LSAP(pairScores, FALSE)
    
    # Return:
    ret <- sum(pairScores[matrix(c(seq_along(optimalMatching), optimalMatching), ncol=2L)])
    if (reportMatching) {
      attr(ret, 'matching') <- optimalMatching
      attr(ret, 'pairScores') <- pairScores
      ret
    } else {
      ret
    }
  }
}

#' @describeIn MutualArborealInfo Takes splits instead of trees
#' 
#' @param partitionQualityIndex Output of [`SplitPairingInformationIndex`] for
#' `n` taxa; calculated automatically if not specified, but passing a cached
#' value may improve performance.
#' @param bestMatchOnly Logical specifying whether to return how informative
#'  each split is about its best match only (`TRUE`) 
#'  or how informative each split is about each other split (`FALSE`).
#' @template splits12params
#' 
#' @export
MutualClusterInfoSplits <- function (
    splits1, splits2, reportMatching = FALSE,
    bestMatchOnly = TRUE,
    partitionQualityIndex = SplitPairingInformationIndex(dim(splits1)[1])
  ) {
  
  dimSplits1 <- dim(splits1)
  dimSplits2 <- dim(splits2)
  nTerminals <- dimSplits1[1]
  if (dimSplits2[1] != nTerminals) {
    stop("Split rows must bear identical labels")
  }
  lnUnrootedN <- LnUnrooted.int(nTerminals)
  
  if (dimSplits1[2] < dimSplits2[2]) {
    tmp <- splits1
    splits1 <- splits2
    splits2 <- tmp
    
    tmp <- dimSplits1
    dimSplits1 <- dimSplits2
    dimSplits2 <- tmp
  }
  
  taxonNames <- rownames(splits1) 
  
  if (!is.null(taxonNames)) {
    splits2 <- unname(splits2[rownames(splits1), , drop=FALSE])
    splits1 <- unname(splits1) # split1[split2] faster without names
  }
  
  nSplits1 <- dimSplits1[2]
  nSplits2 <- dimSplits2[2]
  inSplit1 <- colSums(splits1)
  inSplit2 <- colSums(splits2)
  notInSplit1 <- nTerminals - inSplit1
  notInSplit2 <- nTerminals - inSplit2
  
  logTrees1 <- LnRooted(inSplit1) + LnRooted(notInSplit1)
  logTrees2 <- LnRooted(inSplit2) + LnRooted(notInSplit2)
  
  pairScores <- matrix((mapply(function(i, j) {
    split1 <- splits1[, i]
    split2 <- splits2[, j]
    
    variationOfInfo <- SplitEntropy(split1, split2)['vI']
    partitionQualityIndex[as.character(round(variationOfInfo, 6L))]
  }, rep(seq_len(nSplits1), each=nSplits2), seq_len(nSplits2)
  )), nSplits2, nSplits1)
  
  if (bestMatchOnly) {
    if (nSplits2 == 1) {
      min(pairScores)
    } else {
      optimalMatching <- solve_LSAP(pairScores, TRUE)
      
      # Return:
      ret <- sum(pairScores[matrix(c(seq_along(optimalMatching), optimalMatching), ncol=2L)])
      if (reportMatching) {
        attr(ret, 'matching') <- optimalMatching
        attr(ret, 'pairScores') <- pairScores
        ret
      } else {
        ret
      }
    }
  } else {
    # Return:
    sum(pairScores)
  }
}

#' Are splits compatible?
#' 
#' Splits are compatible if they are concave; i.e. they can both be true
#' simultaneously.
#' 
#' @template split12params
#' @return `SplitsCompatible` returns a logical specifying whether the splits
#' provided are compatible with one another.
#' 
#' @author Martin R. Smith
#' @export
SplitsCompatible <- function (split1, split2) {
  # Return:
  (
    all (split1[split2]) ||
    all(split1[!split2]) ||
    all(!split1[split2]) ||
    all(!split1[!split2])
  )
}

#' Visualise a matching
#' 
#' Depicts the bipartitions that are matched between two trees using a 
#' specified Generalized Robinson Foulds tree distance measure.
#' 
#' @param Func Function used to construct tree similarity.
#' @param tree1,tree2 Trees of class `phylo`, with tips labelled identically.
#' @param setPar Logical specifying whether graphical parameters should be 
#' set to display trees side by side.
#' @param Plot Function to use to plot trees.
#' @param \dots Additional parameters to send to `Plot`.
#' 
#' @author Martin R. Smith
#' @importFrom ape nodelabels edgelabels plot.phylo
#' @importFrom colorspace qualitative_hcl
#' @importFrom graphics par
#' @export
VisualizeMatching <- function(Func, tree1, tree2, setPar = TRUE, Plot = plot.phylo, ...) {
  
  splits1 <- Tree2Splits(tree1)
  edge1 <- tree1$edge
  child1 <- edge1[, 2]
  partitionEdges1 <- vapply(colnames(splits1), 
                            function (node) which(child1 == node), integer(1))
  
  splits2 <- Tree2Splits(tree2)
  edge2 <- tree2$edge
  child2 <- edge2[, 2]
  partitionEdges2 <- vapply(colnames(splits2), 
                            function (node) which(child2 == node), integer(1))
  
  matching <- Func(tree1, tree2, reportMatching = TRUE)
  pairings <- attr(matching, 'matching')
  scores <- attr(matching, 'pairScores')
  pairScores <- signif(mapply(function (i, j) scores[i, j], seq_along(pairings), pairings), 3)
  
  nSplits <- length(pairings)
  palette <- qualitative_hcl(nSplits, c=42, l=88)
  adjNo <- c(0.5, -0.2)
  adjVal <- c(0.5, 1.1)
  
  if (setPar) origPar <- par(mfrow=c(2, 1), mar=rep(0.5, 4))
  
  Plot(tree1)#, ...)
  edgelabels(seq_along(pairings), partitionEdges1[pairings], bg=palette, adj=adjNo)
  #edgelabels(seq_along(pairings), partitionEdges1, cex=0.8, font=2, frame='n', adj=adjVal)
  edgelabels(pairScores, partitionEdges1[pairings], frame='n', adj=adjVal, cex=0.8)
  
  Plot(tree2)#, ...)
  edgelabels(seq_along(pairings), partitionEdges2, bg=palette, adj=adjNo)
  #edgelabels(seq_along(pairings), partitionEdges2, cex=0.8, font=2, frame='n', adj=adjVal)
  edgelabels(pairScores, partitionEdges2, frame='n', adj=adjVal, cex=0.8)
  
  if (setPar) par(origPar)
}