% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameters.R
\name{parameters}
\alias{parameters}
\title{Converting TL-moments to distribution parameters}
\usage{
parameters(tlm, distr, ...)
}
\arguments{
\item{tlm}{object returned by TLMoments}

\item{distr}{character object defining the distribution. Supported types are
"gev", "gumbel", and "gpd"}

\item{...}{additional arguments}
}
\value{
Numeric vector, matrix, list, or data.frame of parameter estimates with
class \code{parameters}. The object contains the following attributes: \itemize{
 \item \code{distribution}: a character indicating the used distribution
 \item \code{source}: a list with background information (used function, data, n, formula, trimmings; mainly for internal purposes)
}
The attributes are hidden in the print-function for a clearer presentation.
}
\description{
Converts TL-moments to distribution parameters. By now, conversions only for GEV, GUMBEL, and GPD
are available. Important trimming options are calculated through known formulas (see references for
some of them), other options are calculated through a numerical optimization. In this case there's a
warning informing about the experimental nature of this feature.
}
\examples{
xmat <- matrix(evd::rgev(100, shape = .2), nc = 4)
xvec <- xmat[, 3]
xlist <- lapply(1L:ncol(xmat), function(i) xmat[, i])
xdat <- data.frame(
 station = rep(letters[1:2], each = 50),
 season = rep(c("S", "W"), 50),
 hq = as.vector(xmat)
)

tlm <- TLMoments(xvec, leftrim = 0, rightrim = 0)
parameters(tlm, "gev")

tlm <- TLMoments(xmat, leftrim = 1, rightrim = 1)
parameters(tlm, "gumbel")

tlm <- TLMoments(xlist)
parameters(tlm, "gpd")

tlm <- TLMoments(xdat, hq ~ station, leftrim = 0, rightrim = 2)
parameters(tlm, "gev")

tlm <- TLMoments(xdat, hq ~ station + season, leftrim = 0, rightrim = 2)
parameters(tlm, "gev")

library(magrittr)
TLMoments(evd::rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 0) \%>\%
 parameters("gpd")

TLMoments(evd::rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 0) \%>\%
 parameters("gpd", u = 10)

TLMoments(evd::rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 1) \%>\%
 parameters("gpd")

TLMoments(evd::rgpd(500, loc = 10, scale = 3, shape = .3), rightrim = 2) \%>\%
 parameters("gpd")

# Numerical calculation
tlm <- TLMoments(evd::rgumbel(1000, loc = 5, scale = 2), leftrim = 1, rightrim = 4)
parameters(tlm, "gumbel")

tlm <- TLMoments(evd::rgumbel(1000, loc = 5000, scale = 2000), leftrim = 1, rightrim = 4)
parameters(tlm, "gumbel")

tlm <- TLMoments(evd::rgev(1000, loc = 10, scale = 5, shape = .4), leftrim = 0, rightrim = 5)
parameters(tlm, "gev")

# Its A LOT slower:
# system.time(replicate(500,
#   parameters(TLMoments(evd::rgumbel(100, loc = 5, scale = 2), 1, 1), "gumbel")
# ))[3]
# system.time(replicate(500,
#   parameters(TLMoments(evd::rgumbel(100, loc = 5, scale = 2), 1, 2), "gumbel")
# ))[3]

}
\references{
Elamir, E. A. H. (2010). Optimal choices for trimming in trimmed L-moment method. Applied Mathematical Sciences, 4(58), 2881-2890.

Fischer, S., Fried, R., & Schumann, A. (2015). Examination for robustness of parametric estimators for flood statistics in the context of extraordinary extreme events. Hydrology and Earth System Sciences Discussions, 12, 8553-8576.

Hosking, J. R. (1990). L-moments: analysis and estimation of distributions using linear combinations of order statistics. Journal of the Royal Statistical Society. Series B (Methodological), 105-124.

Hosking, J. R. M. (2007). Some theory and practical uses of trimmed L-moments. Journal of Statistical Planning and Inference, 137(9), 3024-3039.
}

