% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distance_calculations.R
\name{diagram_distance}
\alias{diagram_distance}
\title{Calculate distance between a pair of persistence diagrams.}
\usage{
diagram_distance(
  D1,
  D2,
  dim = 0,
  p = 2,
  distance = "wasserstein",
  sigma = NULL
)
}
\arguments{
\item{D1}{the first persistence diagram, either computed from a TDA/TDAstats function like \code{\link[TDA]{ripsDiag}}/\code{\link[TDAstats]{calculate_homology}}, or such an object converted to a data frame with \code{\link{diagram_to_df}}.}

\item{D2}{the second persistence diagram, either computed from TDA/TDAstats function like \code{\link[TDA]{ripsDiag}}/\code{\link[TDAstats]{calculate_homology}}, or such an object converted to a data frame with \code{\link{diagram_to_df}}.}

\item{dim}{the non-negative integer homological dimension in which the distance is to be computed, default 0.}

\item{p}{a number representing the wasserstein power parameter, at least 1 and default 2.}

\item{distance}{a string which determines which type of distance calculation to carry out, either "wasserstein" (default) or "fisher".}

\item{sigma}{either NULL (default) or a positive number representing the bandwidth for the Fisher information metric}
}
\value{
the numeric value of the distance calculation.
}
\description{
Calculates the distance between a pair of persistence diagrams in a particular homological dimension,
either the output from a \code{\link{diagram_to_df}} function call or from a TDA/TDAstats homology calculation like \code{\link[TDA]{ripsDiag}}/\code{\link[TDAstats]{calculate_homology}}.
}
\details{
The most common distance calculations between persistence diagrams
are the wasserstein and bottleneck distances, both of which "match" points between
their two input diagrams and compute the "loss" of the optimal matching 
(see \url{http://www.geometrie.tugraz.at/kerber/kerber_papers/kmn-ghtcpd_journal.pdf} for details). Another 
method for computing distances, the Fisher information metric, 
converts the two diagrams into distributions
defined on the plane, and calculates a distance between the resulting two distributions
(\url{https://proceedings.neurips.cc/paper/2018/file/959ab9a0695c467e7caf75431a872e5c-Paper.pdf}).
If the `distance` parameter is "fisher" then `sigma` must not be NULL.
}
\examples{

# create two diagrams
D1 <- TDA::ripsDiag(TDA::circleUnif(n = 20,r = 1),
                    maxdimension = 1,maxscale = 2)
D2 <- TDA::ripsDiag(TDA::sphereUnif(n = 20,d = 2,r = 1),
                    maxdimension = 1,maxscale = 2)

# calculate 2-wasserstein distance between D1 and D2 in dimension 1
diagram_distance(D1,D2,dim = 1,p = 2,distance = "wasserstein")

# calculate bottleneck distance between D1 and D2 in dimension 0
diagram_distance(D1,D2,dim = 0,p = Inf,distance = "wasserstein")

# Fisher information metric calculation between D1 and D2 for sigma = 1 in dimension 1
diagram_distance(D1,D2,dim = 1,distance = "fisher",sigma = 1)
}
\references{
Kerber M, Morozov D and Nigmetov A (2017). "Geometry Helps to Compare Persistence Diagrams." \url{http://www.geometrie.tugraz.at/kerber/kerber_papers/kmn-ghtcpd_journal.pdf}.

Le T, Yamada M (2018). "Persistence fisher kernel: a riemannian manifold kernel for persistence diagrams." \url{https://proceedings.neurips.cc/paper/2018/file/959ab9a0695c467e7caf75431a872e5c-Paper.pdf}.
}
\author{
Shael Brown - \email{shaelebrown@gmail.com}
}
