# This file was generated by Rcpp::compileAttributes
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

eigenest <- function(spatpca, covariance, gamma, eigenfn2){
  p = nrow(spatpca)
  M = t(spatpca)%*%covariance%*%spatpca
  M.eig = eigen(M,symmetric=T)
  K = ncol(spatpca)
  m_star = K
  total.var = sum(diag(covariance))
  temp_v = total.var - sum(M.eig$values)
  err = (total.var - sum(M.eig$values-gamma))/(p-m_star)
  
  while(err > M.eig$values[m_star]-gamma){
    if(m_star == 1){
      err = (total.var - M.eig$values[1] + gamma)/(p-1)
      break
    }
    temp_v = temp_v + M.eig$values[m_star]
    m_star = m_star-1
    err  = (temp_v + m_star*gamma)/(p-m_star)
  }
  if(err > M.eig$values[m_star] - gamma)
    err = total.var/(p)
  eigenvalue = pmax(M.eig$values - err - gamma,0)
  
  Phi = as.matrix(spatpca)%*%M.eig$vectors
  Sigma = eigenfn2%*%diag(eigenvalue,K)%*%t(eigenfn2)
  return(list(err = err, Phi = Phi, eigenvalue = eigenvalue,Sigma = Sigma))
}

#' Regularized PCA for spatial data
#'
#' @description Produce spatial patterns at the coordinate grids according to the specified tuning parameters or the selected tuning parameters by M-fold cross-validation.
#' @param Y Data matrix (\eqn{n \times p}) stores the values at \eqn{p} locations with sample size \eqn{n}. If the dimension of grids are two, convert each sample to a vector form with length \eqn{p}.
#' @param sx Numeric vector of \eqn{x}-coordinates of grids.
#' @param sy Numeric vector of \eqn{y}-coordinates of grids. If NULL, it is a one-dimensional case.
#' @param M Optional number of folds; default is 5. Smallest value allowable is M=2.
#' @param K User-supplied number of eigenfunctions.
#' @param tau1 Optional user-supplied numeric vector of a nonnegative smoothness parameter sequence. If NULL, 10 values in a range are used.
#' @param tau2 Optional user-supplied numeric vector of a nonnegative sparseness parameter sequence. If NULL, none is used.
#' @param center If TRUE, to center the columns of Y. Default is FALSE.
#' @param plot.eigen If TRUE, plot the estimated eigenfunctions. Default is TRUE.
#' @param plot.cv If TRUE, plot the image of cv values. Default is TRUE.
#' @param parallel If TRUE, see the example below. Default is FALSE.
#' @param ... Additional arguments passed to \code{\link{spatpca}}
#' @author Wen-Ting Wang and Hsin-Cheng Huang.
#' @return \item{eigenfn.cv}{The estimated eigenfunctions over the coordinates with respect to selected tuning parameters.}
#' @return \item{stau1}{The specified or selected tau1.}
#' @return \item{stau2}{The specified or selected tau2.}
#' @return \item{cv}{The cv socres.}
#' @return \item{tau1}{The sequence of tau1-values used in the process.}
#' @return \item{tau2}{The sequence of tau2-values used in the process.}
#' @examples
#' \dontrun{
#' ## 1D case
#' x_1D <- as.matrix(seq(-5, 5, length = 50))
#' Phi_1D <- exp(-x_1D^2)/norm(exp(-x_1D^2), "F")
#' Y_1D <- rnorm(n = 100, sd = 3)\%*\%t(Phi_1D) + matrix(rnorm(n = 100*50), 100, 50)
#' cv_1D <- spatpca(x = x_1D, Y = Y_1D)
#'
#' ## 2D case
#' x_coord <- seq(-5, 5, length=20)
#' y_coord <- seq(-5, 5, length=20)
#' x_2D <- cbind(as.vector(t(matrix(rep(x_coord,20), ncol = 20))),rep(y_coord,20))
#' Phi_2D <- exp(-apply(x_2D^2, 1, sum))
#' Phi_2D <- as.matrix(Phi_2D)/norm(Phi_2D, "F")
#' Y_2D <- rnorm(n = 100, sd = 3)\%*\%t(Phi_2D) + matrix(rnorm(100*400), nrow = 100, ncol = 400)
#' cv_2D <- spatpca(x = x_2D, Y = Y_2D)
#'
#' #' @references Wang, W.-T., Huang, H.-C. (2015). Regularized principal component analysis for spatial data. Manuscript.
#' @export
spatpca <- function(x, Y, K = 1, tau1 = NULL, tau2 = NULL, M = 5, center = FALSE, x_new = NULL, plot.cv = FALSE, maxit = 50, thr = 1e-03){
  call2 <- match.call()
  x = as.matrix(x)
  
  if(nrow(x) != ncol(Y))
    stop("The number of rows of x should be equal to the number of columns of Y.")
  if (nrow(x) < 3)
    stop("Number of locations must be larger than 2.")
  if (ncol(x) > 2 )
    stop("Dimension of locations must be 1 or 2.")
  
  if (M >= nrow(Y))
    stop("Number of folds must be less than sample size.")
  if(length(K) !=1 & (K <= 0 || (K%%1) != 0))
    stop("Need a positive integer rank K value")
  else if(K >= min(dim(Y)))
    stop("K must be less than the rank of data matrix.")
  
  if(center == TRUE)
    Y = Y - apply(Y , 2, "mean")
  
  if(is.null(tau2)) {
    tau2 = 0
    ntau2 =1
  }else{ 
    ntau2 <- length(tau2)
  }
  
  if(is.null(tau1)) {
    ntau1 <- 11
    egvl <- svd(Y)$d[1]^2 
    max.tau1 <- egvl*sqrt(ncol(Y)/nrow(Y))
    tau1 <- c(0,exp(seq(log(max.tau1/1e3), log(max.tau1), length = (ntau1-1))))   
    
  }else{
    ntau1 <- length(tau1)
  }
  if(M < 2 && (ntau1 > 1 || ntau2 > 1)) {
    ntau1 = 1
    ntau2 = 1
    warning("Only produce the result based on the largest tau1 and largest tau2.")
  }
  
  stra <- sample(rep(1:M, length.out = nrow(Y)))
  
  if(ntau2 ==1 && tau2 > 0){
    l2 <- c(0,exp(seq(log(tau2/1e3), log(tau2), length = 10)))
    cvtemp <- spatpcacv_rcpp(x, Y, M, K, tau1, tau2, stra, maxit, thr, l2)
    cvtau1 <- cvtemp$cvtau1
    cvtau2 <- cvtemp$cvtau2
    cv1 <- cvtemp$cv
    est <- cvtemp$est
  }
  else{
    cvtemp <- spatpcacv_rcpp(x, Y, M, K, tau1, tau2, stra, maxit, thr, 1)
    cvtau1 <- cvtemp$cvtau1
    cvtau2 <- cvtemp$cvtau2
    cv1 <- cvtemp$cv
    est <- cvtemp$est
  } 
  if(is.null(x_new)){
    x_new = x
    estfn <- est
  }
  else{
    estfn <- tpm2(x_new, x, est)
  #  estfn <- apply(estfn, 2, function(x) x/sqrt(t(x)%*%x))
  }
  
  if(plot.cv == TRUE && !is.null(cv1)){
    if(ntau2 >1){
      par(mfrow=c(2,1))
      plot(tau1,cv1[1,],type='l',main="for tau1 selection given tau2")
      plot(tau2,cv1[2,],type='l',main="for tau2 selection given selected tau2")
    }
    else{
      plot(tau1,cv1,type='l',main="for tau1 selection given tau2")
    }
    #  image.plot(tau1, tau2, cv/M, main = paste("CV score"), xlab = "tau1", ylab = "tau2")
  }
  
  obj.cv <- list(call=call2, eigenfn = estfn, stau1 = cvtau1, stau2 = cvtau2, cv = cv1, tau1 = tau1, tau2 = tau2, Y = Y, x = x, K = K, est = est)
  class(obj.cv) <- "spatpca"
  return(obj.cv)
}


#' M-fold cross-validation for covariance estimation
#'
#' @description Execute M-fold cross-validation to select a tuning parameter, gamma, and produce the corresponding estimated covariance functions
#' @param basis "spatpca" object.
#' @param Optional number of folds; default is 5.
#' @param gamma Optional user-supplied numeric vector of a nonnegative tuning parameter sequence. If NULL, 10 values in a range are used.
#' @author Wen-Ting Wang and Hsin-Cheng Huang.
#' @return \item{cov}{Estimated covariance matrix at the locations with respect to the selected tuning parameter.}
#' @return \item{sgamma}{Selected gamma.}
#' @return \item{cv}{cv socres.}
#' @return \item{gamma}{Sequence of gamma-values used in the process.}
#' @seealso  \code{\link{spatpca}}}
#' @examples
#' \dontrun{
#' ## 1D case
#' s_1D <- as.matrix(seq(-5, 5, length = 50))
#' Phi_1D <- exp(-s_1D^2)/norm(exp(-s_1D^2),"F")
#' Y_1D <- rnorm(n = 100, sd = 3)%*%t(Phi.true.1D) + matrix(rnorm(n = 100*50), nrow = 100, ncol = 50)
#' cv_1D <- spatpca(x = s_1D, Y = Y_1D, K = 1)
#' cov_est <- covfn(basis = cv_1D)}
#' @references Wang, W.-T., Huang, H.-C. (2015). Regularized Principal Component Analysis for Spatial Data. Manuscript.
#' @export
covfn <- function(basis, M = 5, gamma = NULL){
  
  Y <- basis$Y
  x <- basis$x
  Phi <- basis$est
  Phi2 <- basis$eigenfn
  K <- basis$K
  center <- basis$center
  
  n <- nrow(Y)
  p <- ncol(Y)
  covariance <- t(Y)%*%(Y)/n
  
  if(is.null(gamma)) {
    gsize <- 10
    gammamax <- eigen(t(Phi)%*%covariance%*%Phi, symmetric = TRUE)$values[1]
    gamma <- c(0,exp(seq(log(gammamax/1e3), log(gammamax), length = gsize-1)))
  }
  nk2 = sample(rep(1:M, length.out = n))
  
  cvgamma <-spatpcacv_gamma(Y = Y, Phi = Phi, M = M, gamma = gamma, nk = nk2)
  
  for(l in 1:gsize)
    if(min(cvgamma) == min(cvgamma[l]))
      break
  gamma.cv = gamma[l]
  
  ###covariance
  temp = eigenest(Phi, covariance, gamma.cv, Phi2)
  err = temp$err
  eigenvalueour = temp$eigenvalue
  PhiourK1 = temp$Phi
  cov.spatpca = temp$Sigma
  
  
  return(list(cov = cov.spatpca, sgamma = gamma.cv, cv = cvgamma, gamma=gamma))
}

