% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simList-accessors.R
\name{inputs}
\alias{inputs}
\alias{simList-accessors-inout}
\alias{inputs,simList-method}
\alias{inputs<-}
\alias{inputs<-,simList-method}
\alias{outputs}
\alias{outputs,simList-method}
\alias{outputs<-}
\alias{outputs<-,simList-method}
\alias{inputArgs}
\alias{inputArgs,simList-method}
\alias{inputArgs<-}
\alias{inputArgs<-,simList-method}
\alias{outputArgs}
\alias{outputArgs,simList-method}
\alias{outputArgs<-}
\alias{outputArgs<-,simList-method}
\title{Inputs and outputs}
\usage{
inputs(sim)

\S4method{inputs}{simList}(sim)

inputs(sim) <- value

\S4method{inputs}{simList}(sim) <- value

outputs(sim)

\S4method{outputs}{simList}(sim)

outputs(sim) <- value

\S4method{outputs}{simList}(sim) <- value

inputArgs(sim)

\S4method{inputArgs}{simList}(sim)

inputArgs(sim) <- value

\S4method{inputArgs}{simList}(sim) <- value

outputArgs(sim)

\S4method{outputArgs}{simList}(sim)

outputArgs(sim) <- value

\S4method{outputArgs}{simList}(sim) <- value
}
\arguments{
\item{sim}{A \code{simList} object from which to extract element(s) or
in which to replace element(s).}

\item{value}{The object to be stored at the slot. See Details.}
}
\value{
Returns or sets the value(s) of the \code{input} or \code{output} slots
in the \code{simList} object.
}
\description{
These functions are one of three mechanisms to add the information about which
input files to load in a \code{spades} call and the information about which
output files to save. 1) As arguments to a \code{simInit} call. Specifically, \code{inputs}
or \code{outputs}. See \code{?simInit}. 2) With the \code{inputs(simList)} or \code{outputs(simList)}
function call.
3) By adding a function called \code{.inputObjects} inside a module, which will be executed
during the \code{simInit} call. This last way is the most "modular" way to create
default data sets for your model. See below for more details.
}
\details{
Accessor functions for the \code{inputs} and \code{outputs} slots in a
\code{simList} object.
}
\note{
The automatic file type handling only adds the correct extension from a given
\code{fun} and \code{package}. It does not do the inverse, from a given extension find the
correct \code{fun} and \code{package}.
}
\section{inputs function or argument in \code{simInit}}{


\code{inputs} accepts a data.frame, with up to 7 columns.
Columns are:

\tabular{ll}{
\code{file} \tab required, a character string indicating the file path. There is no
default.\cr

\code{objectName} \tab optional, character string indicating the name of the object
that the loaded file will be assigned to in the \code{simList}. This object
can therefore be accessed with \code{sim$xxx} in any module, where
\code{objectName = "xxx"}. Defaults to the filename without file extension or
directory information.\cr

\code{fun} \tab optional, a character string indicating the function to use to
load that file. Defaults to the known extensions in \code{SpaDES} (found by
examining \code{.fileExtensions()}). The \code{package} and \code{fun} can be
jointly specified here as \code{"packageName::functionName"}, e.g.,
\code{"raster::raster"}.\cr

\code{package} \tab optional character string indicating the package in
which to find the \code{fun});\cr

\code{loadTime} \tab optional numeric, indicating when in simulation time the file
should be loaded. The default is the highest priority at \code{start(sim)},
i.e., at the very start. \cr

\code{interval} \tab optional numeric, indicating at what interval should this same
exact file  be reloaded from disk, e.g,. 10 would mean every 10 time units. The
default is NA or no interval, i.e, load the file only once as described in
\code{loadTime} \cr

\code{arguments} \tab is a list of lists of named arguments, one list for each
\code{fun}. For example, if \code{fun="raster"}, \code{arguments = list(native = TRUE)}
will pass the argument "native = TRUE" to raster.  If there is only one list,
then it is assumed to apply to all files and will be recycled as per normal R
rules of recycling for each \code{fun}.\cr
}

Currently, only \code{file} is required. All others will be filled with defaults
if not specified.

See the modules vignette for more details (\code{browseVignettes("SpaDES.core")}).
}

\section{\code{.inputObjects} function placed inside module}{


Any code placed inside a function called \code{.inputObjects} will be run during
\code{simInit()} for the purpose of creating any objects required by this module,
i.e., objects  identified in the \code{inputObjects} element of \code{defineModule}.
This is useful if there is something required before simulation to produce the module
object dependencies, including such things as downloading default datasets, e.g.,
\code{downloadData('LCC2005', modulePath(sim))}.
Nothing should be created here that does not create an named object in \code{inputObjects}.
Any other initiation procedures should be put in the "init" eventType of the doEvent function.
Note: the module developer can use 'sim$.userSuppliedObjNames' inside the function to
selectively skip unnecessary steps because the user has provided those inputObjects in the
simInit call. e.g., the following code would look to see if the user had passed \code{defaultColor}
into during \code{simInit}. If the user had done this, then this function would not override
that value with 'red'. If the user has not passed in a value for \code{defaultColor}, then
the module will get it here:

\code{if (!('defaultColor' \%in\% sim$.userSuppliedObjNames)) \{
 sim$defaultColor <- 'red'
\}}
}

\section{outputs function or argument in \code{simInit}}{


\code{outputs} accepts a data.frame similar to the \code{inputs} data.frame, but
with up to 6 columns.

\tabular{ll}{
\code{objectName} \tab required, character string indicating the name of the object
in the \code{simList} that will be saved to disk (without the \code{sim$} prefix).\cr

\code{file} \tab optional, a character string indicating the file path to save to.
The default is to concatenate \code{objectName} with the model timeunit and
\code{saveTime}, separated by underscore, '\code{_}'. So a default filename would be
\code{"Fires_year1.rds"}.\cr

\code{fun} \tab optional, a character string indicating the function to use to
save that file. The default is \code{\link{saveRDS}} \cr

\code{package} \tab optional character string indicating the package in
which to find the \code{fun});\cr

\code{saveTime} \tab optional numeric, indicating when in simulation time the file
should be saved. The default is the lowest priority at \code{end(sim)},
i.e., at the very end. \cr

\code{arguments} \tab is a list of lists of named arguments, one list for each
\code{fun}. For example, if \code{fun = "write.csv"},
\code{arguments = list(row.names = TRUE)} will pass the argument
\code{row.names = TRUE} to \code{write.csv}  If there is only one list,
then it is assumed to apply to all files and will be recycled as per normal R
rules of recycling for each \code{fun}.\cr
}

See the modules vignette for more details (\code{browseVignettes("SpaDES.core")}).
}

\examples{
#######################
# inputs
#######################

# Start with a basic empty simList
sim <- simInit()

test <- 1:10
library(igraph) # for \%>\%
library(reproducible) # for checkPath
tmpdir <- file.path(tempdir(), "inputs") \%>\% checkPath(create = TRUE)
tmpFile <- file.path(tmpdir, "test.rds")
saveRDS(test, file = tmpFile)
inputs(sim) <- data.frame(file = tmpFile) # using only required column, "file"
inputs(sim) # see that it is not yet loaded, but when it is scheduled to be loaded
simOut <- spades(sim)
inputs(simOut) # confirm it was loaded
simOut$test

# can put data.frame for inputs directly inside simInit call
allTifs <- dir(system.file("maps", package = "quickPlot"),
               full.names = TRUE, pattern = "tif")

# next: .objectNames are taken from the filenames (without the extension)
# This will load all 5 tifs in the SpaDES sample directory, using
#   the raster fuction in the raster package, all at time = 0
if (require("rgdal", quietly = TRUE)) {
  sim <- simInit(
    inputs = data.frame(
      files = allTifs,
      functions = "raster",
      package = "raster",
      loadTime = 0,
      stringsAsFactors = FALSE)
    )

  ##############################
  #A fully described inputs object, including arguments:
  files <- dir(system.file("maps", package = "quickPlot"),
               full.names = TRUE, pattern = "tif")
  # arguments must be a list of lists. This may require I() to keep it as a list
  #   once it gets coerced into the data.frame.
  arguments = I(rep(list(native = TRUE), length(files)))
  filelist = data.frame(
     objectName = paste0("Maps", 1:5),
     files = files,
     functions = "raster::raster",
     arguments = arguments,
     loadTime = 0,
     intervals = c(rep(NA, length(files) - 1), 10)
  )
  inputs(sim) <- filelist
  spades(sim)
}

  # Example showing loading multiple objects from global environment onto the
  #   same object in the simList, but at different load times
  a1 <- 1
  a2 <- 2
  # Note arguments must be a list of NROW(inputs), with each element itself being a list,
  #  which is passed to do.call(fun[x], arguments[[x]]), where x is row number, one at a time
  args <- lapply(1:2, function(x) {
                 list(x = paste0("a", x),
                      envir = environment()) # may be necessary to specify in which envir a1, a2
                                             # are located, if not in an interactive sessino
                 })
  inputs <- data.frame(objectName = "a", loadTime = 1:2, fun = "base::get", arguments = I(args))
  a <- simInit(inputs = inputs, times = list(start = 0, end = 1))
  a <- spades(a)
  identical(a1, a$a)

  end(a) <- 3
  a <- spades(a) # different object (a2) loaded onto a$a
  identical(a2, a$a)

# Clean up after
unlink(tmpdir, recursive = TRUE)

#######################
# outputs
#######################

library(igraph) # for \%>\%
tmpdir <- file.path(tempdir(), "outputs") \%>\% checkPath(create = TRUE)
tmpFile <- file.path(tmpdir, "temp.rds")
tempObj <- 1:10

# Can add data.frame of outputs directly into simInit call
sim <- simInit(objects = c("tempObj"),
               outputs = data.frame(objectName = "tempObj"),
               paths = list(outputPath = tmpdir))
outputs(sim) # To see what will be saved, when, what filename
sim <- spades(sim)
outputs(sim) # To see that it was saved, when, what filename

# Also can add using assignment after a simList object has been made
sim <- simInit(objects = c("tempObj"), paths = list(outputPath = tmpdir))
outputs(sim) <- data.frame(objectName = "tempObj", saveTime = 1:10)
sim <- spades(sim)
outputs(sim) # To see that it was saved, when, what filename.

# can do highly variable saving
tempObj2 <- paste("val",1:10)
df1 <- data.frame(col1 = tempObj, col2 = tempObj2)
sim <- simInit(objects = c("tempObj", "tempObj2", "df1"),
  paths=list(outputPath = tmpdir))
outputs(sim) = data.frame(
     objectName = c(rep("tempObj", 2), rep("tempObj2", 3), "df1"),
     saveTime = c(c(1,4), c(2,6,7), end(sim)),
     fun = c(rep("saveRDS", 5), "write.csv"),
     package = c(rep("base", 5), "utils"),
     stringsAsFactors = FALSE)
# since write.csv has a default of adding a column, x, with rownames, must add additional
#   argument for 6th row in data.frame (corresponding to the write.csv function)
outputArgs(sim)[[6]] <- list(row.names=FALSE)
sim <- spades(sim)
outputs(sim)

# read one back in just to test it all worked as planned
newObj <- read.csv(dir(tmpdir, pattern = "year10.csv", full.name = TRUE))
newObj

# using saving with SpaDES-aware methods
# To see current ones SpaDES can do
.saveFileExtensions()

library(raster)
if (require(rgdal)) {
  ras <- raster(ncol = 4, nrow = 5)
  ras[] <- 1:20

  sim <- simInit(objects = c("ras"), paths = list(outputPath = tmpdir))
  outputs(sim) = data.frame(
    file = "test",
    fun = "writeRaster",
    package = "raster",
    objectName = "ras",
    stringsAsFactors = FALSE)

  outputArgs(sim)[[1]] <- list(format = "GTiff") # see ?raster::writeFormats
  simOut <- spades(sim)
  outputs(simOut)
  newRas <- raster(dir(tmpdir, full.name = TRUE, pattern = ".tif"))
  all.equal(newRas, ras) # Should be TRUE
}
# Clean up after
unlink(tmpdir, recursive = TRUE)
}
\seealso{
\code{\link{SpaDES.core-package}}, specifically the section 1.2.2 on loading and saving.

Other functions to access elements of a 'simList' object: 
\code{\link{.addDepends}()},
\code{\link{doEvent.checkpoint}()},
\code{\link{envir}()},
\code{\link{events}()},
\code{\link{globals}()},
\code{\link{modules}()},
\code{\link{objs}()},
\code{\link{packages}()},
\code{\link{params}()},
\code{\link{paths}()},
\code{\link{progressInterval}()},
\code{\link{times}()}
}
\concept{functions to access elements of a 'simList' object}
