% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semWeights.R
\name{activeModule}
\alias{activeModule}
\title{Active module identification}
\usage{
activeModule(
  graph,
  type,
  seed,
  eweight = "none",
  alpha = 0.05,
  top = 100,
  limit = 10000,
  ...
)
}
\arguments{
\item{graph}{An igraph object.}

\item{type}{Module identification method. If \code{type = "kou"}, 
the Steiner tree algorithm will be applied. 
If \code{type = "usp"}, the resulting graph will be the union of all 
significant shortest paths. If \code{type = "rwr"}, the random walk 
with restart algorithm will be enabled. Finally, if \code{type = "hdi"}, 
the heat diffusion algorithm is used.}

\item{seed}{Either a user-defined vector containing seed node names 
or one among: "pvlm", "proto", or "qi", corresponding to the seed 
name attribute yielded by \code{\link[SEMgraph]{weightGraph}}.}

\item{eweight}{Edge weight type derived from
\code{\link[SEMgraph]{weightGraph}} or from user-defined distances. 
This option determines the weight-to-distance transform. If set to 
"none" (default), edge weights will be set to 1. 
If \code{eweight = "kegg"}, repressing interactions (-1) will be set 
to 1 (maximum distance), neutral interactions (0) will be set to 0.5, 
and activating interactions (+1) will be set to 0 (minimum distance).
If \code{eweight = "zsign"}, all significant interactions will be set 
to 0 (minimum distance), while non-significant ones will be set to 1.
If \code{eweight = "pvalue"}, weights (p-values) will be transformed 
to the inverse of negative base-10 logarithm. 
If \code{eweight = "custom"}, the algorithm will use the distance 
measure specified by the user as "weight" edge attribute.}

\item{alpha}{Significance level to assess shortest paths significance, 
when type is "usp". By default, \code{alpha = 0.05}.}

\item{top}{Number of top nodes for the "rwr" and "hdi" algorithms. The 
output subgraph is induced by the top-n ranking nodes.
By default, \code{top = 100} (i.e., the top-100 of nodes are selected).}

\item{limit}{An integer value corresponding to the number of graph 
edges. If \code{type = "usp"}, beyond this limit, multicore computation 
is enabled to reduce the computational burden. 
By default, \code{limit = 10000}.}

\item{...}{Currently ignored.}
}
\value{
An active module, an igraph object with colored nodes
(seed = "green", and connector = "white").
}
\description{
Uses different information flow and tree-based strategies 
for identifying active modules (e.g., disease modules), including a 
perturbed subset of nodes and edges. 
Function scalability enables graph reduction at both pathway and 
entire interactome scales.
}
\details{
Graph filtering algorithms include:
\enumerate{
\item "kou", the Steiner tree connecting a set of seed nodes, using 
the algorithm suggested by Kou et al. (1981);
\item "usp", generates a subnetwork as the union of the significant 
(P-value < alpha) shortest paths between the seeds set;
\item "rwr", Random Walk with Restart, a wrapper for \code{random.walk}
function of the R package \code{diffusr};
\item "hdi", Heat Diffusion algorithm, a wrapper for \code{heat.diffusion}
function of the R package \code{diffusr}.
}
}
\examples{

# Graph weighting
G <- weightGraph(graph = sachs$graph, data = sachs$pkc, group = sachs$group,
                 method = "r2z",
                 seed = c(0.05, 0.5, 0.5))

# RWR algorithm, seeds and edge P-values as weights
R1 <- activeModule(graph = G, type = "kou", seed = "pvlm", eweight = "pvalue")
R2 <- activeModule(graph = G, type = "kou", seed = "proto", eweight = "pvalue")
R3 <- activeModule(graph = G, type = "kou", seed = "qi", eweight = "pvalue")

# Graphs
old.par <- par(no.readonly = TRUE)
par(mfrow=c(2,2), mar=rep(2, 4))
plot(G, layout = layout.circle, main = "input graph")
box(col = "gray")
plot(R1, layout = layout.circle, main = "lm P-value (alpha = 0.05)")
box(col = "gray")
plot(R2, layout = layout.circle, main = "prototype (h = 0.5)")
box(col = "gray")
plot(R3, layout = layout.circle, main = "closeness (q = 0.5)")
box(col = "gray")
par(old.par)

}
\references{
Palluzzi F, Grassi M (2021). SEMgraph: An R Package for Causal Network 
Analysis of High-Throughput Data with Structural Equation Models. 
<arXiv:2103.08332>

Kou L, Markowsky G, Berman L (1981). A fast algorithm for Steiner trees.
Acta Informatica, 15(2): 141-145. <https://doi.org/10.1007/BF00288961>

Simon Dirmeier (2018). diffusr: Network Diffusion Algorithms. R
package version 0.1.4.
<https://CRAN.R-project.org/package=diffusr/>
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
