## extract tags
tag <- function(x) attr(x, "Rd_tag")

## replace tags
untag <- function(x) {
  if (is.null(x)) return(NULL)
  attr(x, "Rd_tag") <- "TEXT"
  x
}

## construct strings from rd
reconstruct <- function(rd) {
  if (is.null(rd)) return()

  if (is.list(rd)) {
	if (length(tag(rd)) && tag(rd) %in% c('\\item', '\\tabular', '\\eqn', '\\deqn', '\\link')){
		if (tag(rd) == '\\link')
			return(paste('\\link', sprintf('[%s]', attr(rd, 'Rd_option')), '{', rd, '}', sep = ""))
		if (length(rd) == 2)
			return(paste(tag(rd), '{', rd[[1]], '}{',
				paste(sapply(rd[[2]], reconstruct), collapse = ""),
				'}', sep = "", collapse = "")) else if (length(rd) == 0) return(tag(rd))
	}
	special <- tag(rd) == toupper(tag(rd))
	singles <- tag(rd) %in% c('\\tab', '\\cr')
	prefix <- ifelse(special, "",
		paste(tag(rd), ifelse(singles, "", "{"), sep = ""))
	suffix <- ifelse(special, "", ifelse(singles, "", "}"))
	paste(prefix, paste(sapply(rd, reconstruct), collapse = ""), suffix,
		 sep = "")
  } else {
    rd
  }
}

## wrap strings with comment prefix
comment_line <- function(x, exdent = 0) {
  if (missing(x)) return(comment_prefix())

  strwrap(x, width = 80, exdent = exdent, prefix = comment_prefix())
}

## add comments
comment_tag <- function(tag, value) {
  if (is.null(value) || value == "" || length(value) == 0) return()

  comment_line(paste(tag, value), exdent = 2)
}

## access the comment prefix
comment_prefix <- function() {
	if (is.null(getOption("roxygen.comment")))
				"#' " else getOption("roxygen.comment")
}


##' Remove Rd files for the undocumented functions.
##' Usually roxygen will generate Rd files even for the undocumented functions, and
##' the \command{name} and \command{title} tags of such files will be the same.
##' This function removes all such Rd files under the 'man' directory.
##'
##' @param pkg the directory of the source package
##' @return \code{NULL} (if such Rd files exist, there will be messages printed in the
##' console showing which files are deleted)
##' @export
##' @author Yihui Xie <\url{http://yihui.name}>
##' @note
##' We can also specify the tag '@@nord' to suppress Rd creation, in which case
##' we have to use Rd2 in roxygen (this defaults to be \code{FALSE} but was changed
##' to default to \code{TRUE} in \code{\link{rab}}).
rm_undocumented = function(pkg) {
    for (f in list.files(file.path(pkg, "man"), ".*\\.Rd$", all.files = TRUE,
                         full.names = TRUE)) {
        x = readLines(f)
        cond = identical(
            gsub('\\\\name\\{(.*)\\}', '\\1', grep('^\\\\name', x, value=TRUE)),
            gsub('\\\\title\\{(.*)\\}', '\\1', grep('^\\\\title', x, value=TRUE))
        )
        if (cond) {
            unlink(f)
            message("deleted: ", f)
            flush.console()
        }
    }
}

##' Roxygenize a package, clean up and build/check the package.
##'
##' After the source package is roxygenized, this function first removes the
##' unnecessary Rd files, and build the package. Optionally
##' it also installs or checks the package, reformats the code in the usage and
##' examples section, and replaces `\%' with `\\\%'.
##'
##' @aliases roxygen_and_build rab
##' @param pkg the root directory of the source package
##' @param roxygen.dir the directory for the roxygenized package (by
##' default it is \code{pkg.roxygen})
##' @param install whether to install the package
##' @param check whether to check the package
##' @param check.opts options to check the package
##' (e.g. \code{"--no-examples"})
##' @param remove.check whether to remove the directory generated by
##' \command{R CMD check}
##' @param escape whether to escape \code{"\%"}
##' @param reformat whether to reformat the examples and usage and
##' escape the percent symbol sections; see
##' \code{\link{reformat_code}}
##' @param use.Rd2 passed to \code{\link[roxygen]{roxygenize}}; see
##' \code{\link[roxygen]{make.Rd2.roclet}}
##' @param ... other arguments passed to
##' \code{\link[roxygen]{roxygenize}}
##' @note This function also tries to remove directories \file{pkg/inst/doc} and
##' \file{pkg/inst} if they are empty; this is due to the fact that roxygen
##' will generate these directories no matter if they are needed.
##'
##' This function also has a short name \code{rab} to avoid typing efforts.
##' @return NULL
##' @author Yihui Xie <\url{http://yihui.name}>
##' @rdname roxygen_and_build
##' @export
##' @examples \dontrun{
##' roxygen_and_build("Rd2roxygen", install = TRUE)
##' ## or simply
##' rab('Rd2roxygen', install = TRUE)
##' }
roxygen_and_build = function(pkg, roxygen.dir = NULL, install = FALSE,
    check = FALSE, check.opts = "", remove.check = TRUE, escape = TRUE,
    reformat = TRUE, use.Rd2 = TRUE, ...) {
    if (is.null(roxygen.dir)) roxygen.dir = file.path(dirname(pkg), paste(basename(pkg), '.roxygen', sep = ''))
    if (file.exists(roxygen.dir) && normalizePath(pkg) != normalizePath(roxygen.dir)) unlink(roxygen.dir, recursive = TRUE)
    roxygenize(pkg, roxygen.dir, use.Rd2 = use.Rd2, ...)
    unlink(sprintf("%s/.git", roxygen.dir), recursive = TRUE)
    if (!length(list.files((inst.dir <- file.path(roxygen.dir, 'inst', 'doc')), recursive = TRUE)))
        unlink(inst.dir, recursive = TRUE)
    if (!length(list.files((inst.dir <- file.path(roxygen.dir, 'inst')), recursive = TRUE)))
        unlink(inst.dir, recursive = TRUE)
    rm_undocumented(roxygen.dir)
    rd.list = list.files(file.path(roxygen.dir, "man"), ".*\\.Rd$", all.files = TRUE, full.names = TRUE)
    if (reformat) {
        for (f in rd.list) {
            if (isTRUE(reformat_code(f))) {
                x = readLines(f)
                x = gsub('\\\\dontrun', '\\dontrun', x, fixed = TRUE)
                writeLines(x, con = f)
            }
        }
    }
    if (escape) {
        for (f in rd.list) {
            x = readLines(f)
            if (length(grep("(^|[^\\])%", x))) {
                x = gsub("(^|[^\\])%", "\\1\\\\%", x)
                writeLines(x, con = f)
                message("updated % --> \\%: ", f)
            }
        }
    }
    system(sprintf("R CMD build %s ", roxygen.dir))
    if (install)
        system(sprintf("R CMD INSTALL %s ", roxygen.dir))
    if (check) {
        system(sprintf("R CMD check %s %s", roxygen.dir, check.opts))
        if (remove.check) unlink(sprintf('%s.Rcheck', roxygen.dir), TRUE)
    }
    invisible(NULL)
}

##' @rdname roxygen_and_build
##' @export
rab = function(...) {
    roxygen_and_build(...)
}


##' Format the code in the usage and examples sections.
##' This function can polish the Rd files generated by roxygen mainly
##' in two sections: usage and examples. By default, roxygen will omit
##' the spaces and indent in the code, which makes the code
##' (especially in the examples section) hard to read. It uses the
##' function \code{\link[formatR]{tidy.source}} in the package
##' \pkg{formatR}.
##'
##' If the macro \code{"\\dontrun"} is detected in the Rd file, this
##' function will use an unstable way to reformat the example and
##' usage sections, which might destroy your original Rd (e.g. the
##' subsection macro can be mangled); otherwise it will use ordinary
##' text-processing techniques to deal with these sections and
##' generally will not affect other sections. However, neither way is
##' absolutely safe. Whenever you run into troubles, just try to turn
##' off reformatting.
##' @param path the path of the Rd file
##' @param section the sections in the Rd file to (re)format (the only
##' possible choices are \code{"examples"} and \code{"usage"})
##' @param ... other arguments passed to \code{tidy.source}
##' @return This function returns \code{TRUE} if the macro
##' \code{"\\dontrun"} is detected in the Rd; \code{FALSE} if not
##' deteced; and \code{NULL} if the package \pkg{formatR} is not
##' installed (as a side effect, the original Rd file will be updated)
##' @export
##' @author Yihui Xie <\url{http://yihui.name}>
##' @seealso \code{\link[formatR]{tidy.source}}
##' @examples
##' rd.file = system.file('examples', 'reformat_code_demo.Rd', package = 'Rd2roxygen')
##' file.copy(rd.file, tempdir())
##' fmt.file = file.path(tempdir(), 'reformat_code_demo.Rd')
##'
##' if (interactive()) file.show(fmt.file)  ## show the raw Rd
##' reformat_code(fmt.file)
##' if (interactive()) file.show(fmt.file)  ## the formatted Rd
##'
reformat_code = function(path, section = c('examples', 'usage'), ...) {
    if (require('formatR', quietly = TRUE)) {
        rd = readLines(path)
        ## easy to deal with when there are no \\dontrun's
        if (!any(grepl('\\\\dontrun', rd))) {
            message('(*) good, I did not detect the \\dontrun macro in your Rd \n  ', path)
            if ('usage' %in% section) {
                idx0 = grep('^\\\\usage\\{', rd)
                idx1 = grep('^\\\\description\\{', rd)
                if (length(idx0) && length(idx1)) {
                    message('  reformatting section usage')
                    ## \description always comes after \usage
                    idx1 = idx1 - 2
                    ## users may provide \usage{f()} (the right bracket on the same line)
                    if (flag <- idx1 < idx0) idx1 = idx0
                    tmp = rd[idx0:idx1]
                    tmp[1] = sub('^\\\\usage\\{', '', tmp[1])
                    if (flag) tmp = sub('\\}[ ]*$', '', tmp)
                    txt = paste(tidy.source(text = tmp, output = FALSE,
                                keep.blank.line = TRUE, ...)$text.tidy, collapse = '\n')
                    txt = paste('\\usage{', txt, sep = '')
                    if (flag)
                        rd[idx0] = paste(txt, '}', sep = '') else rd[idx0:(idx1 + 1)] =
                            c(txt, rep('', idx1 - idx0), '}')
                }
            }
            if ('examples' %in% section) {
                ## remove trailing spaces
                while (tail(rd, 1) == '') {
                    rd = rd[-length(rd)]
                }
                idx0 = grep('^\\\\examples\\{', rd)
                if (length(idx0)) {
                    message('  reformatting section examples')
                    ## only \alias might appear after \examples
                    idx1 = grep('^\\\\alias\\{', rd)
                    if (length(idx1) && any(idx1 > idx0))
                        idx1 = min(idx1[idx1 > idx0]) - 1 else idx1 = length(rd)
                    tmp = rd[idx0:idx1]
                    tmp[1] = sub('^\\\\examples\\{', '', tmp[1])
                    nn = length(tmp)
                    tmp[nn] = sub('\\}$', '', tmp[nn])
                    txt = tidy.source(text = tmp, output = FALSE,
                                keep.blank.line = TRUE, ...)$text.tidy
                    txt[1] = paste('\\examples{', txt[1], sep = '')
                    nn0 = length(txt)
                    txt[nn0] = paste(txt[nn0], '}', sep = '')
                    txt = paste(txt, collapse = '\n')
                    rd[idx0:idx1] = c(txt, rep('', idx1 - idx0))
                }
            }
            writeLines(rd, path)
            flush.console()
            return(invisible(FALSE))
        }
        message('(!) reformatting might destroy your Rd without warnings,\n  because I detected the \\dontrun macro in \n  ', path)
        rd = tools::parse_Rd(path)
        flag = FALSE
        for (sec in section) {
            idx = which(sapply(rd, tag) == paste('\\', sec, sep = ''))
            if (length(idx)) {
                message('  reformatting section ', sec)
                txt = rd[idx]
                class(txt) = 'Rd'
                txt = as.character(txt)
                txt = txt[-c(1, 2, length(txt))]
                txt = gsub('^[[:space:]]+', '', txt)
                txt = sub('^\\\\dontrun', 'tag_name_dontrun = function() ', txt)
                txt[txt == ''] = '\n'
                con = textConnection(paste(txt, collapse = ''))
                txt = tidy.source(con, output = FALSE, keep.blank.line = TRUE, ...)
                txt = txt$text.tidy
                close(con)
                txt = paste(txt, rep(c('\n', ''), c(length(txt) - 1, 1)), sep = '', collapse = '')
                txt = gsub('tag_name_dontrun = function() {', '\\dontrun{', txt, fixed = TRUE)
                rd[[idx]] = structure(list(structure(txt, Rd_tag = 'TEXT')), Rd_tag = paste('\\', sec, sep = ''))
                flag = TRUE
            } else message('  section ', sec, ' not found')
        }
        flush.console()
        if (flag) {
            class(rd) = 'Rd'
            cat(as.character(rd, deparse = TRUE), file = path, sep = '')
            return(invisible(TRUE))
        }
    } else {
        message('unable to tidy the code: the package "formatR" was not installed')
    }
}
