%%  RUnit : A unit test framework for the R programming language
%%  Copyright (C) 2003-2007  Thomas Koenig, Matthias Burger, Klaus Juenemann
%%
%%  This program is free software; you can redistribute it and/or modify
%%  it under the terms of the GNU General Public License as published by
%%  the Free Software Foundation; either version 2 of the License, or
%%  (at your option) any later version.
%%
%%  This program is distributed in the hope that it will be useful,
%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%  GNU General Public License for more details.
%%
%%  You should have received a copy of the GNU General Public License
%%  along with this program; if not, write to the Free Software
%%  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

%%  $Id: checkFuncs.Rd,v 1.12 2007/05/15 22:42:16 burgerm Exp $

\encoding{latin1}
\name{checkFuncs}
\alias{checkEquals}
\alias{checkEqualsNumeric}
\alias{checkIdentical}
\alias{checkTrue}
\alias{checkException}
\alias{DEACTIVATED}
\title{RUnit check functions}

\usage{
checkEquals(target, current, msg, tolerance = .Machine$double.eps^0.5, checkNames = TRUE, ...)
checkEqualsNumeric(target, current, msg, tolerance = .Machine$double.eps^0.5, ...)
checkIdentical(target, current, msg)
checkTrue(expr, msg)
checkException(expr, msg, silent=FALSE)
DEACTIVATED(msg)
}

\arguments{
  \item{current, target}{two objects to be compared (should not be S4 class objects).}
  \item{msg}{an optional message to document a check and to facilitate
    the identification of a possible failure. The message only appears
    as text in the test protocol, it is not further used in any of the check functions.}
  \item{tolerance}{numeric >= 0.  A numeric check does not fail if
    differences are smaller than `tolerance'.}
  \item{checkNames}{flag, if FALSE the names attributes are set to NULL
    for both current and target before performing the check.}
  \item{expr}{syntactically valid R expression which can be evaluated
    and must return a logical scalar (TRUE|FALSE). A named expression is
    also allowed but the name is disregarded.}
  \item{silent}{flag passed on to \code{try}, which determines if the
    error message generated by the checked function is displayed.}
  \item{...}{optional arguments passed to \code{all.equal} or \code{all.equal.numeric}}
}

\description{A set of functions used to check the results of some test
  calculation. If these functions are called within the RUnit framework,
  the results of the checks are stored and reported in the test
  protocol.
  
  \code{checkEquals} compares two R objects by invoking \code{all.equal} on
  the two objects. If the objects are not equal an error is
  generated and the failure is reported to the test logger such that it
  appears in the test protocol.
  
  \code{checkEqualsNumeric} works just like \code{checkEquals} except
  that it invokes \code{all.equal.numeric} instead of \code{all.equal}
  
  \code{checkIdentical} is a convenience wrapper around identical using
  the error loggin mechanism of RUnit.
  
  \code{checkTrue} uses the function \code{identical} to check if the expression
  provided as first argument evaluates to \code{TRUE}. If not, an error is
  generated and the failure is reported to the test logger such that it
  appears in the test protocol.
  
  \code{checkException} evaluates the passed expression and uses the
  \code{try} mechanism to check if the evaluation generates an error.
  If it does the test is OK. Otherwise an error is
  generated and the failure is reported to the test logger such that it
  appears in the test protocol.
  
  \code{DEACTIVATED} interrupts the test function and reports the test case
  as deactivated. In the test protocol deactivated test functions are
  listed separately. Test case deactivation can be useful in the case of
  major refactoring. Alternatively, test cases can be commented out
  completely but then it is easy to forget the test case altogether.
  
}

\details{
  The check functions are direct equivalents of the various
  methods of the class junit.framework.Assert of Javas junit framework
  which served as basis for the RUnit package.

  For functions defined inside a package equipped with a namespace only
  exported functions can be accessed inside test cases directly. For
  functions not exported the only way to test them is to use the ':::'
  operator combined with the packagename as a prefix.
  
  Special care is required if test cases are written for S4 classes and
  methods. If a new class is defined inside a test case via a
  \code{setClass} call the class is added to the global class
  cache and thus available outside the test case. It will persist until
  explicitly removed via a \code{removeClass} call. Same applies for new method
  and generic definitions. Be sure to remove methods and classes in
  each  test case they are defined after the checks have been
  performed. This is an advise gained from the cumbersome experience:
  not doing so leads to difficult to pin down error causes incurred
  from previously executed test cases.
  For a simple example see the provided test cases in examples/runitVirtualClassTest.r.
}



\author{Thomas \enc{Knig}{Koenig}, Klaus \enc{Jnemann}{Juenemann} \& Matthias Burger}

\seealso{
  \code{\link{all.equal}}, \code{\link{all.equal.numeric}} and
  \code{\link{identical}} are the underlying comparison functions.
  \code{\link{try}} is used for error catching.
}


\examples{

checkTrue(1 < 2, "check1")     ## passes fine
## checkTrue(1 > 2, "check2")  ## appears as failure in the test protocol

v <- 1:3
w <- 1:3
checkEquals(v,w)          ## pasess fine
names(v) <- c("A", "B", "C")
## checkEquals(v,w)      ## fails because v and w have different names
checkEqualsNumeric(v,w)  ## passes fine because names are ignored


fun <- function(x) {
  if(x)
  {
   stop("stop conditions signaled")
  }
  return()
}

checkException(fun(TRUE))   ## passes fine
## checkException(fun(FALSE))  ## failure, because f raises no error
checkException(fun(TRUE), silent=TRUE)

## DEACTIVATED("here one can document on the reason for deactivation")
}

\keyword{programming}
\concept{RUnit}
