#' X-13ARIMA model specification, SA/X13
#'
#' @description
#'
#' Function to create (and/or modify) a \code{c("SA_spec", "X13")} class object with the SA model specification for the X13 method.
#' It can be done from a pre-defined 'JDemetra+' model specification (a \code{character}), a previous specification
#' (\code{c("SA_spec", "X13")} object) or a seasonal adjustment model (\code{c("SA", "X13")} object).
#'
#' @param spec an x13 model specification. It can be the 'JDemetra+' name (\code{character}) of a predefined X13 'JDemetra+' model specification
#' (see \emph{Details}), an object of class \code{c("SA_spec","X13")} or an object of class \code{c("SA", "X13")}. The default is \code{"RSA5c"}.
#'
#' @inheritParams regarima_spec_x13
#' @param x11.mode character: the decomposition mode. Determines the mode of the seasonal adjustment decomposition to be performed:
#' \code{"Undefined"} - no assumption concerning the relationship between the time series components is made;
#' \code{"Additive"} - assumes an additive relationship;
#' \code{"Multiplicative"} - assumes a multiplicative relationship;
#' \code{"LogAdditive"} - performs an additive decomposition of the logarithms of the series being adjusted;
#' \code{"PseudoAdditive"} - assumes an pseudo-additive relationship. Could be changed by the program, if needed.
#' @param x11.seasonalComp logical: if \code{TRUE}, the program computes a seasonal component. Otherwise, the seasonal component
#' is not estimated and its values are all set to 0 (additive decomposition) or 1 (multiplicative decomposition).
#' @param x11.lsigma numeric: the lower sigma boundary for the detection of extreme values, > 0.5, default=1.5.
#' @param x11.usigma numeric: the upper sigma boundary for the detection of extreme values, > lsigma, default=2.5.
#' @param x11.trendAuto logical: automatic Henderson filter. If \code{TRUE}, an automatic selection of the Henderson filter's length
#' for the trend estimation is enabled.
#' @param x11.trendma numeric: the length of the Henderson filter. The user-defined length of the Henderson filter.
#' The option is available when the automatic Henderson filter selection is disabled (\code{x11.trendAuto=FALSE}).
#' Should be an odd number in the range (1, 101].
#' @param x11.seasonalma a vector of character(s) specifying which seasonal moving average (i.e. seasonal filter)
#' will be used to estimate the seasonal factors for the entire series. The vector can be of length:
#' 1 - the same seasonal filter is used for all periods (e.g.: `seasonal.filter = "Msr"` or `seasonal.filter = "S3X3"` );
#' or have a different value for each quarter (length 4) or each month (length 12) - (e.g. for quarterly series: `seasonal.filter = c("S3X3", "Msr", "S3X3", "Msr")`).
#' Possible filters are: `"Msr"`, `"Stable"`, `"X11Default"`, `"S3X1"`, `"S3X3"`, `"S3X5"`, `"S3X9"`, `"S3X15"`.
#' `"Msr"` - the program chooses the final seasonal filter automatically.
#' @param x11.fcasts numeric: the number of forecasts generated by the RegARIMA model in periods (positive values) or years (negative values).Default value: fcasts=-1.
#' @param x11.bcasts numeric: the number of backcasts used in X11. Negative figures are translated in years of backcasts. Default value: bcasts=0.
#' @param x11.calendarSigma character to specify if the standard errors used for extreme values detection and adjustment are computed:
#' from 5 year spans of irregulars (\code{"None"}, the default);
#' separately for each calendar month/quarter (\code{"All"});
#' separately for each period only if Cochran’s hypothesis test determines that the irregular component is heteroskedastic
#' by calendar month/quarter (\code{"Signif"});
#' separately for two complementary sets of calendar months/quarters specified by the x11.sigmaVector parameter (\code{"Select"},
#' see parameter \code{x11.sigmaVector}).
#' @param x11.sigmaVector a vector to specify one of the two groups of periods for whose standard errors used for extreme values
#' detection and adjustment will be computed. Only used if \code{x11.calendarSigma = "Select"}. Possible values are: "Group1" and "Group2".
#' @param x11.excludeFcasts logical: to exclude forecasts and backcasts. If \code{TRUE}, the RegARIMA model forecasts and backcasts are not used during the detection of extreme values in the seasonal adjustment routines.
#'
#' @details
#'
#' The available predefined 'JDemetra+' model specifications are described in the table below:
#'
#' \tabular{rrrrrrr}{
#' \strong{Identifier} |\tab \strong{Log/level detection} |\tab \strong{Outliers detection} |\tab \strong{Calendar effects} |\tab \strong{ARIMA}\cr
#' RSA0 |\tab \emph{NA} |\tab \emph{NA} |\tab \emph{NA} |\tab Airline(+mean)\cr
#' RSA1 |\tab automatic |\tab AO/LS/TC  |\tab \emph{NA} |\tab Airline(+mean)\cr
#' RSA2c |\tab automatic |\tab AO/LS/TC |\tab 2 td vars + Easter |\tab Airline(+mean)\cr
#' RSA3 |\tab automatic |\tab AO/LS/TC |\tab \emph{NA} |\tab automatic\cr
#' RSA4c |\tab automatic |\tab AO/LS/TC |\tab 2 td vars + Easter |\tab automatic\cr
#' RSA5c |\tab automatic |\tab AO/LS/TC |\tab 7 td vars + Easter |\tab automatic\cr
#' X11 |\tab \emph{NA} |\tab \emph{NA} |\tab \emph{NA} |\tab NA
#' }
#' @return
#' A two-element list of class \code{c("SA_spec", "X13")}, containing:
#' (1) an object of class \code{c("regarima_spec", "X13")} with the RegARIMA model specification;
#' (2) an object of class \code{c("X11_spec", "data.frame")} with the X11 algorithm specification.
#' Each component refers to different parts of the SA model specification, mirroring the arguments of the function (for details,
#' see the function arguments in the description).
#' Each lowest-level component (except span, pre-specified outliers, user-defined variables and pre-specified ARMA coefficients)
#' is structured as a data frame with columns denoting different variables of the model specification and rows referring to:
#' \itemize{
#' \item first row: the base specification, as provided within the argument \code{spec};
#' \item second row: user modifications as specified by the remaining arguments of the function (e.g.: \code{arima.d});
#' \item and third row: the final model specification.
#'
#' The final specification (third row) shall include user modifications (row two) unless they were wrongly specified.
#' The pre-specified outliers, user-defined variables and pre-specified ARMA coefficients consist of a list of
#' \code{Predefined} (base model specification) and \code{Final} values.
#'
#' \item{regarima}{an object of class \code{c("regarima_spec", "x13")}. See \emph{Value} of the function \code{\link{regarima_spec_x13}}.}
#'
#' \item{x11}{a data.frame of class \code{c("X11_spec", "data.frame")}, containing the \emph{x11} variables in line with
#' the names of the arguments variables. The final values can be also accessed with the function \code{\link{s_x11}}.}
#'}
#' @references
#' More information and examples related to 'JDemetra+' features in the online documentation:
#' \url{https://jdemetra-new-documentation.netlify.app/}
#' BOX G.E.P. and JENKINS G.M. (1970), "Time Series Analysis: Forecasting and Control", Holden-Day, San Francisco.
#'
#' BOX G.E.P., JENKINS G.M., REINSEL G.C. and LJUNG G.M. (2015), "Time Series Analysis: Forecasting and Control", John Wiley & Sons, Hoboken, N. J., 5th edition.
#'
#' @seealso \code{\link{x13}}
#'
#' @examples \donttest{
#' myseries <- ipi_c_eu[, "FR"]
#' myspec1 <- x13_spec(spec = "RSA5c")
#' myreg1 <- x13(myseries, spec = myspec1)
#'
#' # To modify a pre-specified model specification
#' myspec2 <- x13_spec(spec = "RSA5c", tradingdays.option = "WorkingDays")
#' myreg2 <- x13(myseries, spec = myspec2)
#'
#' # To modify the model specification of a "X13" object
#'  myspec3 <- x13_spec(myreg1, tradingdays.option = "WorkingDays")
#'  myreg3 <- x13(myseries, myspec3)
#'
#' # To modify the model specification of a "X13_spec" object
#'  myspec4 <- x13_spec(myspec1, tradingdays.option = "WorkingDays")
#'  myreg4 <- x13(myseries, myspec4)
#'
#' # Pre-specified outliers
#'  myspec1 <- x13_spec(spec = "RSA5c", usrdef.outliersEnabled = TRUE,
#'              usrdef.outliersType = c("LS", "AO"),
#'              usrdef.outliersDate = c("2008-10-01", "2002-01-01"),
#'              usrdef.outliersCoef = c(36, 14),
#'              transform.function = "None")
#'
#'  myreg1 <- x13(myseries, myspec1)
#'  myreg1
#'  s_preOut(myreg1)
#'
#'
#' # User-defined calendar regressors
#'  var1 <- ts(rnorm(length(myseries))*10, start = start(myseries), frequency = 12)
#'  var2 <- ts(rnorm(length(myseries))*100, start = start(myseries), frequency = 12)
#'  var <- ts.union(var1, var2)
#
#'  myspec1 <- x13_spec(spec = "RSA5c", tradingdays.option = "UserDefined",
#'                      usrdef.varEnabled = TRUE,
#'                      usrdef.var = var,
#'                      usrdef.varType = c("Calendar", "Calendar"))
#'  myreg1 <- x13(myseries, myspec1)
#'  myreg1
#'
#'  myspec2 <- x13_spec(spec = "RSA5c", usrdef.varEnabled = TRUE,
#'              usrdef.var = var1, usrdef.varCoef = 2,
#'              transform.function = "None")
#'  myreg2 <- x13(myseries, myspec2)
#'  s_preVar(myreg2)
#'
#' # Pre-specified ARMA coefficients
#'  myspec1 <- x13_spec(spec = "RSA5c", automdl.enabled = FALSE,
#'              arima.p = 1, arima.q = 1, arima.bp = 0, arima.bq = 1,
#'              arima.coefEnabled = TRUE,
#'              arima.coef = c(-0.8, -0.6, 0),
#'              arima.coefType = c(rep("Fixed", 2), "Undefined"))
#'
#'  s_arimaCoef(myspec1)
#'  myreg1 <- x13(myseries, myspec1)
#'  myreg1
#'
#' # To define a seasonal filter
#'  myspec1 <- x13_spec("RSA5c", x11.seasonalma = rep("S3X1", 12))
#'  mysa1 <- x13(myseries, myspec1)
#'}
#' @export
x13_spec <- function(spec = c("RSA5c", "RSA0", "RSA1", "RSA2c", "RSA3", "RSA4c","X11"),
                     preliminary.check = NA,
                     estimate.from = NA_character_,
                     estimate.to = NA_character_,
                     estimate.first = NA_integer_,
                     estimate.last = NA_integer_,
                     estimate.exclFirst = NA_integer_,
                     estimate.exclLast = NA_integer_,
                     estimate.tol = NA_integer_,
                     transform.function = c(NA, "Auto", "None", "Log"),
                     transform.adjust = c(NA, "None", "LeapYear", "LengthOfPeriod"),
                     transform.aicdiff = NA_integer_,
                     usrdef.outliersEnabled = NA,
                     usrdef.outliersType = NA,
                     usrdef.outliersDate = NA,
                     usrdef.outliersCoef = NA,
                     usrdef.varEnabled = NA,
                     usrdef.var = NA,
                     usrdef.varType = NA,
                     usrdef.varCoef = NA,
                     tradingdays.option = c(NA, "TradingDays", "WorkingDays", "UserDefined", "None"),
                     tradingdays.autoadjust = NA,
                     tradingdays.leapyear = c(NA, "LeapYear", "LengthOfPeriod", "None"),
                     tradingdays.stocktd = NA_integer_,
                     tradingdays.test = c(NA, "Remove", "Add", "None"),
                     easter.enabled = NA,
                     easter.julian = NA,
                     easter.duration = NA_integer_,
                     easter.test = c(NA, "Add", "Remove", "None"),
                     outlier.enabled = NA,
                     outlier.from = NA_character_,
                     outlier.to = NA_character_,
                     outlier.first = NA_integer_,
                     outlier.last = NA_integer_,
                     outlier.exclFirst = NA_integer_,
                     outlier.exclLast = NA_integer_,
                     outlier.ao = NA,
                     outlier.tc = NA,
                     outlier.ls = NA,
                     outlier.so = NA,
                     outlier.usedefcv = NA,
                     outlier.cv = NA_integer_,
                     outlier.method = c(NA, "AddOne", "AddAll"),
                     outlier.tcrate = NA_integer_,
                     automdl.enabled = NA,
                     automdl.acceptdefault = NA,
                     automdl.cancel = NA_integer_,
                     automdl.ub1 = NA_integer_,
                     automdl.ub2 = NA_integer_,
                     automdl.mixed = NA,
                     automdl.balanced = NA,
                     automdl.armalimit = NA_integer_,
                     automdl.reducecv = NA_integer_,
                     automdl.ljungboxlimit = NA_integer_,
                     automdl.ubfinal = NA_integer_,
                     arima.mu = NA,
                     arima.p = NA_integer_,
                     arima.d = NA_integer_,
                     arima.q = NA_integer_,
                     arima.bp = NA_integer_,
                     arima.bd = NA_integer_,
                     arima.bq = NA_integer_,
                     arima.coefEnabled = NA,
                     arima.coef = NA,
                     arima.coefType = NA,
                     fcst.horizon = NA_integer_,
                     x11.mode = c(NA, "Undefined", "Additive", "Multiplicative", "LogAdditive", "PseudoAdditive"),
                     x11.seasonalComp = NA,
                     x11.lsigma = NA_integer_,
                     x11.usigma = NA_integer_,
                     x11.trendAuto = NA,
                     x11.trendma = NA_integer_,
                     x11.seasonalma = NA_character_,
                     x11.fcasts = NA_integer_,
                     x11.bcasts = NA_integer_,
                     x11.calendarSigma = NA,
                     x11.sigmaVector = NA,
                     x11.excludeFcasts = NA){
  UseMethod("x13_spec", spec)
}
#' @export
x13_spec.character <- function(spec = c("RSA5c", "RSA0", "RSA1", "RSA2c", "RSA3", "RSA4c","X11"),
                         preliminary.check = NA,
                         estimate.from = NA_character_,
                         estimate.to = NA_character_,
                         estimate.first = NA_integer_,
                         estimate.last = NA_integer_,
                         estimate.exclFirst = NA_integer_,
                         estimate.exclLast = NA_integer_,
                         estimate.tol = NA_integer_,
                         transform.function = c(NA, "Auto", "None", "Log"),
                         transform.adjust = c(NA, "None", "LeapYear", "LengthOfPeriod"),
                         transform.aicdiff = NA_integer_,
                         usrdef.outliersEnabled = NA,
                         usrdef.outliersType = NA,
                         usrdef.outliersDate = NA,
                         usrdef.outliersCoef = NA,
                         usrdef.varEnabled = NA,
                         usrdef.var = NA,
                         usrdef.varType = NA,
                         usrdef.varCoef = NA,
                         tradingdays.option = c(NA, "TradingDays", "WorkingDays", "UserDefined", "None"),
                         tradingdays.autoadjust = NA,
                         tradingdays.leapyear = c(NA, "LeapYear", "LengthOfPeriod", "None"),
                         tradingdays.stocktd = NA_integer_,
                         tradingdays.test = c(NA, "Remove", "Add", "None"),
                         easter.enabled = NA,
                         easter.julian = NA,
                         easter.duration = NA_integer_,
                         easter.test = c(NA, "Add", "Remove", "None"),
                         outlier.enabled = NA,
                         outlier.from = NA_character_,
                         outlier.to = NA_character_,
                         outlier.first = NA_integer_,
                         outlier.last = NA_integer_,
                         outlier.exclFirst = NA_integer_,
                         outlier.exclLast = NA_integer_,
                         outlier.ao = NA,
                         outlier.tc = NA,
                         outlier.ls = NA,
                         outlier.so = NA,
                         outlier.usedefcv = NA,
                         outlier.cv = NA_integer_,
                         outlier.method = c(NA, "AddOne", "AddAll"),
                         outlier.tcrate = NA_integer_,
                         automdl.enabled = NA,
                         automdl.acceptdefault = NA,
                         automdl.cancel = NA_integer_,
                         automdl.ub1 = NA_integer_,
                         automdl.ub2 = NA_integer_,
                         automdl.mixed = NA,
                         automdl.balanced = NA,
                         automdl.armalimit = NA_integer_,
                         automdl.reducecv = NA_integer_,
                         automdl.ljungboxlimit = NA_integer_,
                         automdl.ubfinal = NA_integer_,
                         arima.mu = NA,
                         arima.p = NA_integer_,
                         arima.d = NA_integer_,
                         arima.q = NA_integer_,
                         arima.bp = NA_integer_,
                         arima.bd = NA_integer_,
                         arima.bq = NA_integer_,
                         arima.coefEnabled = NA,
                         arima.coef = NA,
                         arima.coefType = NA,
                         fcst.horizon = NA_integer_,
                         x11.mode = c(NA, "Undefined", "Additive", "Multiplicative", "LogAdditive", "PseudoAdditive"),
                         x11.seasonalComp = NA,
                         x11.lsigma = NA_integer_,
                         x11.usigma = NA_integer_,
                         x11.trendAuto = NA,
                         x11.trendma = NA_integer_,
                         x11.seasonalma = NA_character_,
                         x11.fcasts = NA_integer_,
                         x11.bcasts = NA_integer_,
                         x11.calendarSigma = NA,
                         x11.sigmaVector = NA,
                         x11.excludeFcasts = NA)
{
  spec <- match.arg(spec)
  reg_spec <- gsub("RSA", "RG", spec)
  if (spec == "X11") {
    regarima <- new_regarima_spec_x13()
  } else{
    regarima <-
      regarima_spec_x13(spec = reg_spec, preliminary.check = preliminary.check,
                        estimate.from = estimate.from, estimate.to = estimate.to,
                        estimate.first = estimate.first, estimate.last = estimate.last,
                        estimate.exclFirst = estimate.exclFirst, estimate.exclLast = estimate.exclLast,
                        estimate.tol = estimate.tol, transform.function = transform.function,
                        transform.adjust = transform.adjust, transform.aicdiff = transform.aicdiff,
                        usrdef.outliersEnabled = usrdef.outliersEnabled, usrdef.outliersType = usrdef.outliersType,
                        usrdef.outliersDate = usrdef.outliersDate, usrdef.outliersCoef = usrdef.outliersCoef,
                        usrdef.varEnabled = usrdef.varEnabled, usrdef.var = usrdef.var,
                        usrdef.varType = usrdef.varType, usrdef.varCoef = usrdef.varCoef,
                        tradingdays.option = tradingdays.option, tradingdays.autoadjust = tradingdays.autoadjust,
                        tradingdays.leapyear = tradingdays.leapyear, tradingdays.stocktd = tradingdays.stocktd,
                        tradingdays.test = tradingdays.test, easter.enabled = easter.enabled,
                        easter.julian = easter.julian, easter.duration = easter.duration,
                        easter.test = easter.test, outlier.enabled = outlier.enabled,
                        outlier.from = outlier.from, outlier.to = outlier.to, outlier.first = outlier.first,
                        outlier.last = outlier.last, outlier.exclFirst = outlier.exclFirst,
                        outlier.exclLast = outlier.exclLast, outlier.ao = outlier.ao,
                        outlier.tc = outlier.tc, outlier.ls = outlier.ls, outlier.so = outlier.so,
                        outlier.usedefcv = outlier.usedefcv, outlier.cv = outlier.cv,
                        outlier.method = outlier.method, outlier.tcrate = outlier.tcrate,
                        automdl.enabled = automdl.enabled, automdl.acceptdefault = automdl.acceptdefault,
                        automdl.cancel = automdl.cancel, automdl.ub1 = automdl.ub1,
                        automdl.ub2 = automdl.ub2, automdl.mixed = automdl.mixed,
                        automdl.balanced = automdl.balanced, automdl.armalimit = automdl.armalimit,
                        automdl.reducecv = automdl.reducecv, automdl.ljungboxlimit = automdl.ljungboxlimit,
                        automdl.ubfinal = automdl.ubfinal, arima.mu = arima.mu, arima.p = arima.p,
                        arima.d = arima.d, arima.q = arima.q, arima.bp = arima.bp,
                        arima.bd = arima.bd, arima.bq = arima.bq, arima.coefEnabled = arima.coefEnabled,
                        arima.coef = arima.coef, arima.coefType = arima.coefType,
                        fcst.horizon = fcst.horizon)
  }

  x11 <- x11_spec_def(spec = spec, x11.mode = x11.mode,
                      x11.seasonalComp = x11.seasonalComp, x11.lsigma = x11.lsigma,
                      x11.usigma = x11.usigma, x11.trendAuto = x11.trendAuto,
                      x11.trendma = x11.trendma, x11.seasonalma = x11.seasonalma,
                      x11.fcasts = x11.fcasts, x11.bcasts = x11.bcasts,
                      x11.calendarSigma = x11.calendarSigma, x11.sigmaVector = x11.sigmaVector,
                      x11.excludeFcasts = x11.excludeFcasts)
  z <- list(regarima = regarima, x11 = x11)
  class(z) <- c("SA_spec", "X13")
  return(z)
}
#' @export
x13_spec.X13 <- function(spec,
                     preliminary.check = NA,
                    estimate.from = NA_character_,
                    estimate.to = NA_character_,
                    estimate.first = NA_integer_,
                    estimate.last = NA_integer_,
                    estimate.exclFirst = NA_integer_,
                    estimate.exclLast = NA_integer_,
                    estimate.tol = NA_integer_,
                    transform.function = c(NA, "Auto", "None", "Log"),
                    transform.adjust = c(NA, "None", "LeapYear", "LengthOfPeriod"),
                    transform.aicdiff = NA_integer_,
                    usrdef.outliersEnabled = NA,
                    usrdef.outliersType = NA,
                    usrdef.outliersDate = NA,
                    usrdef.outliersCoef = NA,
                    usrdef.varEnabled = NA,
                    usrdef.var = NA,
                    usrdef.varType = NA,
                    usrdef.varCoef = NA,
                    tradingdays.option = c(NA, "TradingDays", "WorkingDays", "UserDefined", "None"),
                    tradingdays.autoadjust = NA,
                    tradingdays.leapyear = c(NA, "LeapYear", "LengthOfPeriod", "None"),
                    tradingdays.stocktd = NA_integer_,
                    tradingdays.test = c(NA, "Remove", "Add", "None"),
                    easter.enabled = NA,
                    easter.julian = NA,
                    easter.duration = NA_integer_,
                    easter.test = c(NA, "Add", "Remove", "None"),
                    outlier.enabled = NA,
                    outlier.from = NA_character_,
                    outlier.to = NA_character_,
                    outlier.first = NA_integer_,
                    outlier.last = NA_integer_,
                    outlier.exclFirst = NA_integer_,
                    outlier.exclLast = NA_integer_,
                    outlier.ao = NA,
                    outlier.tc = NA,
                    outlier.ls = NA,
                    outlier.so = NA,
                    outlier.usedefcv = NA,
                    outlier.cv = NA_integer_,
                    outlier.method = c(NA, "AddOne", "AddAll"),
                    outlier.tcrate = NA_integer_,
                    automdl.enabled = NA,
                    automdl.acceptdefault = NA,
                    automdl.cancel = NA_integer_,
                    automdl.ub1 = NA_integer_,
                    automdl.ub2 = NA_integer_,
                    automdl.mixed = NA,
                    automdl.balanced = NA,
                    automdl.armalimit = NA_integer_,
                    automdl.reducecv = NA_integer_,
                    automdl.ljungboxlimit = NA_integer_,
                    automdl.ubfinal = NA_integer_,
                    arima.mu = NA,
                    arima.p = NA_integer_,
                    arima.d = NA_integer_,
                    arima.q = NA_integer_,
                    arima.bp = NA_integer_,
                    arima.bd = NA_integer_,
                    arima.bq = NA_integer_,
                    arima.coefEnabled = NA,
                    arima.coef = NA,
                    arima.coefType = NA,
                    fcst.horizon = NA_integer_,
                    x11.mode = c(NA, "Undefined", "Additive", "Multiplicative", "LogAdditive", "PseudoAdditive"),
                    x11.seasonalComp = NA,
                    x11.lsigma = NA_integer_,
                    x11.usigma = NA_integer_,
                    x11.trendAuto = NA,
                    x11.trendma = NA_integer_,
                    x11.seasonalma = NA_character_,
                    x11.fcasts = NA_integer_,
                    x11.bcasts = NA_integer_,
                    x11.calendarSigma = NA,
                    x11.sigmaVector = NA,
                    x11.excludeFcasts = NA)
{
  if (!inherits(spec, c("SA","SA_spec")))
    stop("This function must only be used with c(\"SA\",\"X13\") and c(\"SA_spec\",\"X13\") objects", call. = FALSE)

  regarima <- regarima_spec_x13(spec = spec, preliminary.check = preliminary.check,
                                estimate.from = estimate.from, estimate.to = estimate.to,
                                estimate.first = estimate.first, estimate.last = estimate.last,
                                estimate.exclFirst = estimate.exclFirst, estimate.exclLast = estimate.exclLast,
                                estimate.tol = estimate.tol, transform.function = transform.function,
                                transform.adjust = transform.adjust, transform.aicdiff = transform.aicdiff,
                                usrdef.outliersEnabled = usrdef.outliersEnabled, usrdef.outliersType = usrdef.outliersType,
                                usrdef.outliersDate = usrdef.outliersDate, usrdef.outliersCoef = usrdef.outliersCoef,
                                usrdef.varEnabled = usrdef.varEnabled, usrdef.var = usrdef.var,
                                usrdef.varType = usrdef.varType, usrdef.varCoef = usrdef.varCoef,
                                tradingdays.option = tradingdays.option, tradingdays.autoadjust = tradingdays.autoadjust,
                                tradingdays.leapyear = tradingdays.leapyear, tradingdays.stocktd = tradingdays.stocktd,
                                tradingdays.test = tradingdays.test, easter.enabled = easter.enabled,
                                easter.julian = easter.julian, easter.duration = easter.duration,
                                easter.test = easter.test, outlier.enabled = outlier.enabled,
                                outlier.from = outlier.from, outlier.to = outlier.to, outlier.first = outlier.first,
                                outlier.last = outlier.last, outlier.exclFirst = outlier.exclFirst,
                                outlier.exclLast = outlier.exclLast, outlier.ao = outlier.ao,
                                outlier.tc = outlier.tc, outlier.ls = outlier.ls, outlier.so = outlier.so,
                                outlier.usedefcv = outlier.usedefcv, outlier.cv = outlier.cv,
                                outlier.method = outlier.method, outlier.tcrate = outlier.tcrate,
                                automdl.enabled = automdl.enabled, automdl.acceptdefault = automdl.acceptdefault,
                                automdl.cancel = automdl.cancel, automdl.ub1 = automdl.ub1,
                                automdl.ub2 = automdl.ub2, automdl.mixed = automdl.mixed,
                                automdl.balanced = automdl.balanced, automdl.armalimit = automdl.armalimit,
                                automdl.reducecv = automdl.reducecv, automdl.ljungboxlimit = automdl.ljungboxlimit,
                                automdl.ubfinal = automdl.ubfinal, arima.mu = arima.mu, arima.p = arima.p,
                                arima.d = arima.d, arima.q = arima.q, arima.bp = arima.bp,
                                arima.bd = arima.bd, arima.bq = arima.bq, arima.coefEnabled = arima.coefEnabled,
                                arima.coef = arima.coef, arima.coefType = arima.coefType,
                                fcst.horizon = fcst.horizon)

  x11 <- x11_spec(spec = spec, x11.mode = x11.mode,
                  x11.seasonalComp = x11.seasonalComp, x11.lsigma = x11.lsigma,
                  x11.usigma = x11.usigma, x11.trendAuto = x11.trendAuto,
                  x11.trendma = x11.trendma, x11.seasonalma = x11.seasonalma,
                  x11.fcasts = x11.fcasts, x11.bcasts = x11.bcasts,
                  x11.calendarSigma = x11.calendarSigma, x11.sigmaVector = x11.sigmaVector,
                  x11.excludeFcasts = x11.excludeFcasts)
  z <- list(regarima = regarima, x11 = x11)
  class(z) <- c("SA_spec", "X13")
  return(z)
}

