# Error printing function
#' @include utils.R
error_reasons <- function(x) {
    code_message <- httr::http_status(x$error$code)$message
    errors <- x$error$errors
    errors$reason <- gsub("^([[:alpha:]])", "\\U\\1", to_separated(errors$reason, sep = " "), perl = TRUE)
    errors$message <- gsub("\n", ". ", errors$message, fixed = TRUE)
    res <- sprintf("%s: %s", errors$reason, errors$message)
    if (!is.null(errors$location)) {
        errors$location <- rename_params(errors$location)
        idx_inv <- grep("Invalid parameter", errors$reason)
        idx_req <- unique(grep("Required", errors$reason), grep("parameter", errors$locationType))
        if (length(idx_inv))
            res[idx_inv] <- sprintf("%s '%s': %s", errors$reason[idx_inv], errors$location[idx_inv], errors$message[idx_inv])
        if (length(idx_req))
            res[idx_req] <- sprintf("%s %s: '%s'", errors$reason[idx_req], errors$locationType[idx_req], errors$location[idx_req])
    }
    paste(c(code_message, res), collapse = "\n")
}

# Process response
#' @include utils.R
process_response <- function(response) {
    stopifnot(inherits(response, "response"))
    if (response$status_code == 404L)
        stop(sprintf("The requested URL not found. URL: %s.", strsplit(response$url, "?", fixed = TRUE)[[1L]][1L]), call. = FALSE)
    if (response$status_code == 204L)
        return(NULL)
    if (httr::http_status(response)$category == "success") {
        text <- httr::content(response, as = "text")
        if (text == "")
            stop("No output to parse.", call. = FALSE)
        res <- jsonlite::fromJSON(text, flatten = TRUE)
        res <- convert_types.list(res)
        res <- convert_names(res)
        idx <- sapply(res, is.list)[!grepl("^(rows|items)$", names(res))]
        res[idx] <- lapply(res[idx], convert_types.list)
        res[idx] <- lapply(res[idx], convert_names)
    } else {
        type <- httr::parse_media(response$headers$`Content-type`)
        if (type$complete == "application/json") {
            res <- jsonlite::fromJSON(httr::content(response, as = "text"))
            stop(error_reasons(res), call. = FALSE)
        } else {
            res <- httr::content(response, as = "text")
            stop(sprintf("HTTP error %s:\n%s.", response$status_code, res), call. = FALSE)
        }
    }
    return(res)
}

# Get a Google Analytics API response
#' @include auth.R
api_request <- function(url, token) {
    if (missing(token) && is.null(get_token()))
        api_request(url, token = authorize(cache = FALSE))
    if (missing(token) && !is.null(get_token()))
        token <- get_token()
    if (validate_token(token))
        config <- httr::config(token = token)
    attempts <- getOption("rga.retry.attempts", 5L) + 1L
    for (i in 0L:attempts) {
        response <- httr::GET(url, config = config, httr::accept_json())
        res <- tryCatch(process_response(response), error = identity)
        if (!inherits(res, "error"))
            break
        else if (grepl("User rate limit exceeded|Quota exceeded", res$message) & i < attempts)
            Sys.sleep(2L^i + stats::runif(1L))
        else
            stop(res)
    }
    return(res)
}
