WGLVmix <- function(y, id, w, u, v, pu = 300, pv = 300, eps = 1e-6, 
	rtol = 1.0e-6, verb=0)
{

   # Kiefer-Wolfowitz Estimation of Gaussian Location and Variance Mixtures
   # Input:
   #   y is an N vector of observed values
   #   id is an N vector of indices for the n "individuals"
   #   w is an N vector of weights for the y observations
   #   u is a grid of points on which we evaluate individual means
   #   v is a grid of points on which we evaluate individual variances
   #       if v is scalar then it is treated as a fixed variance parameter
   # Output:
   #   u as above
   #   v as above
   #   fu mixing density for the means
   #   fv mixing density for the variances
   #   logLik value
   #   flag indicating (non)convergence code

wsum <- tapply(w,id,"sum")
t <- tapply(w*y,id,"sum")/wsum
m <- tapply(y,id,"length")
r <- (m-1)/2
s <- (tapply(w*y^2,id,"sum") - t^2*wsum)/(m-1)
n <- length(s)
if(missing(u)) u <- seq(min(t) - eps, max(t) + eps, length = pu)
if(missing(v)) v <- seq(min(s) - eps, max(s) + eps, length = pv)
if(length(v) == 1) {
	v0 <- v
	v <- seq(min(s) - eps, max(s) + eps, length = pv)
	}
du <- diff(u)
du <- c(du[1],du)
wu <- rep(1,n)/n
dv <- diff(v)
dv <- c(dv[1],dv)
wv <- rep(1,n)/n

# Note that 2*r*s/theta ~ chisq_2r  so A needs to be an n by p matrix with entries
# f(s,theta) = (r*s/theta)^r exp(-r*s/theta)/(s Gamma(r))

R <- outer(r*s,v,"/")  
sgamma <- outer(s * gamma(r),rep(1,pv))
r <- outer((m - 1)/2, rep(1,pv))
Av <- outer((exp(-R) * R^r)/sgamma, rep(1,pu))
Au <- dnorm(outer(outer(t, u, "-") * outer(sqrt(wsum),rep(1,pu)), sqrt(v), "/"))
Au <- Au/outer(outer(1/sqrt(wsum),rep(1,pu)),sqrt(v))
Au <- aperm(Au,c(1,3,2)) # permute Au indices so that they are aligned with those of Av
A <- Av * Au

# Initialize the variances 

if(exists("v0")){ 
	fv <- rep(0,length(v))
	v0 <- findInterval(v0,v)
	fv[v0] <- 1/diff(v)[v0]
	}
else{
	f<-WGVmix(y,id,w, rtol = rtol, verb=verb)
	fv<-f$y
	}

#Now do the mean part. 

Au<-matrix(0,n,pu)
for(i in 1:pu){
	Au[,i] <- A[,,i] %*% (fv/sum(fv) )
	}
Au <- Matrix(Au, sparse = TRUE)
f <- KWDual(t,wu,du,Au, rtol = rtol, verb = verb)
fu <- f$f
flag <- f$status
list(u = u, fu = fu, v = v, fv = fv, logLik = f$logLik, flag = flag)
}
