#' Enumerate relationship graphs (RGs)
#'
#' An RG is a graph over all per-person parasite genotypes (each as a
#' vertex), with edges between clone and sibling genotypes.
#' Valid RGs satisfy:
#' \itemize{
#'   \item{Subgraphs induced by clone edges are cluster graphs.}
#'   \item{Subgraphs induced by clone plus sibling edges are cluster graphs.}
#'   \item{Clone edges only link genotypes from different episodes.}
#' }
#'
#' RGs are generated by enumerating nested set partitions under specific
#' constraints; see
#' [Understand graph and partition enumeration](https://aimeertaylor.github.io/Pv3Rs/articles/enumerate.pdf).
#' Each nested set parition is an RG. Clone edges induce a cluster graph,
#' equivalent to a partition of genotypes, with no intra-episode clones allowed.
#' Sibling edges refine the clone partition, with no constraints (intra-episode
#' siblings allowed). Each nested set partition is encoded as a list. Each
#' partition is represented by a list of vectors (either `clone` or `sib`) and
#' by a membership vector (either `clone.vec` or `sib.vec`). By default, an RG
#' encoded as a list is converted to an `igraph` object.
#'
#' @param MOIs Vector of per-episode multiplicities of infection (MOIs),
#' i.e., numbers of per-episode genotypes / vertices.
#' @param igraph Logical; returns RGs as \code{igraph} objects (default `TRUE`).
#' @param progress.bar Logical; show progress bar (default `TRUE`).
#'
#' @return A list of RGs. If `igraph = FALSE`,
#' each RG is a list of length four with:
#'   \describe{
#'     \item{clone}{List of vectors encoding genotypes per clonal cell.}
#'     \item{clone.vec}{Numeric vector with the clonal cell membership of
#'     each genotype.}
#'     \item{sib}{List of vectors encoding clonal cells per sibling cell.}
#'     \item{sib.vec}{Numeric vector with the sibling cell membership of
#'     each clonal cell.}
#'   }
#'   If `igraph = TRUE` (default), each RG is encoded as an \code{igraph} object
#'   (see \code{\link{RG_to_igraph}}).
#'
#' @examples
#' graphs <- enumerate_RGs(c(2, 1, 2), progress.bar = FALSE) # nine graphs
#'
#' @export
enumerate_RGs <- function(MOIs, igraph = TRUE, progress.bar = TRUE) {
  # Check MOIs are positive whole numbers
  if (!all(is_wholenumber(MOIs)) | any(MOIs < 1)) stop("MOIs should be positive integers")

  if(sum(MOIs) > 10) message("Total MOI > 10 may lead to high memory use")

  infection_count <- length(MOIs) # Number of time points
  gs_count <- sum(MOIs) # Number of genotypes

  # compute set partitions
  part.list <- list()
  for (i in 1:gs_count) {
    part.list[[i]] <- partitions::setparts(i)
  }

  gs <- paste0("g", 1:gs_count) # Genotype names

  # get clonal partitions, accounting for no intra-infection clones
  CP_list <- enumerate_CPs(MOIs)

  RG_i <- 0
  RGs <- list() # list of relationship graphs

  # Count number of valid graphs and create progress bar
  n.RG <- sum(sapply(CP_list, function(x) ncol(part.list[[max(x)]])))
  if (progress.bar) pbar <- msg_progress_bar(n.RG)
  message(paste("Number of valid relationship graphs (RGs) is", n.RG))

  for (CP in CP_list) { # for each clonal partition (membership vector)
    n.clones <- max(CP) # number of clonal cells
    clone.names <- paste0("c", 1:n.clones)
    # list of vectors of genotype names by clonal cell
    clones <- stats::setNames(split(gs, CP), clone.names)

    # given clonal relationships, generate all compatible sibling relationships
    # sibling partitions are all set partitions of the clonal cells
    sib.parts <- part.list[[n.clones]]
    n.parts <- ncol(sib.parts) # number of possible partitions
    for (j in 1:n.parts) { # for each sibling partition
      sib.vec <- sib.parts[, j] # membership vector
      n.sib.clones <- max(sib.vec) # number of sibling cells
      sib.clones <- stats::setNames(
        split(clone.names, sib.vec),
        paste0("s", 1:n.sib.clones)
      ) # list of vectors of clonal cell names by sibling cell
      RG <- list(
        clone = clones,
        clone.vec = CP,
        sib = sib.clones,
        sib.vec = sib.vec
      )

      if (igraph) RG <- RG_to_igraph(RG, MOIs)

      RG_i <- RG_i + 1
      RGs[[RG_i]] <- RG
      if (progress.bar) pbar$increment()
    }
  }

  RGs
}

