#' The data.to.PIGE function prepare your data to be analysed by the functions \code{\link{permutation.snp}} and \code{\link{compute.p.snp.obs}}.
#' @export
#' @name data.to.PIGE
#' @title Generate data for Gene- and Pathway-environment interaction
#' @param data a data frame containing the variables in the model and the SNP.
#' @param data.pathway  A data frame (Gene X Pathway) of 0 and 1 values.
#' The rownames (gene name considered) and the colnames (names of the studied pathways).
#'  have to be specified. The value 1 indicates that a gene is included in the corresponding pathway.   
#' @param list.gene.snp List containing for each gene the names of the SNPs belonging to it. This list could be generated by \code{data.to.list.gene.snp} function.
#' @param choice.pathway names or indice of the pathway to be analysed.
#' @return The returned value is a list containing:
#' \item{data.pathway}{ A data frame (Gene X Pathway) of 0 and 1 values similar as \code{data.pathway} but only for the selected pathways.}
#' \item{tab.snp.gene}{A matrix with in the first column the names of the SNPs and the corresponding gene on the second column. Note that a SNPs could belong to several gene.}
#' \item{snp.selected}{A vector of names corresponding to the SNPs analalysed.} 
#' @author Benoit Liquet \email{benoit.liquet@@isped.u-bordeaux2.fr}\cr
#'  Therese Truong \email{therese.truong@@inserm.fr}
#' @examples data(data.pathway)
#' data(data.pige)
#' data(list.gene.snp)
#' #Example: one pathway
#' res1 <-data.to.PIGE(data=data.pige,data.pathway=data.pathway,
#' list.gene.snp=list.gene.snp,choice.pathway=c(2))
#' #Example: two pathways
#' res <-data.to.PIGE(data=data.pige,data.pathway=data.pathway
#' ,list.gene.snp=list.gene.snp,choice.pathway=c(1,2))
data.to.PIGE <- function(data,data.pathway,list.gene.snp,choice.pathway=NULL){
  
  ##definition data.pathway
  if(is.null(choice.pathway)) choice.pathway <- colnames(data.pathway)
  if(length(choice.pathway)==1){
    if(is.character(choice.pathway)) name.col <- choice.pathway else name.col <- colnames(data.pathway)[choice.pathway]
    name.row <- rownames(data.pathway)
     data.pathway <- as.data.frame(data.pathway[,choice.pathway]) 
    rownames(data.pathway) <- name.row
    colnames(data.pathway) <- name.col }else{
    data.pathway <- data.pathway[,choice.pathway]}
    n.gene.path <- apply(data.pathway,MARGIN=1,FUN=function(x)sum(x))
    gene.selected <- rownames(data.pathway)[which(n.gene.path>0)] 
    data.pathway.bis <- as.data.frame(data.pathway[gene.selected,])
  rownames(data.pathway.bis) <- gene.selected
  colnames(data.pathway.bis) <- colnames(data.pathway)
  ##definition snp
  SNP <- list.gene.snp[gene.selected]
  GENE <- rep(names(SNP),times=sapply(SNP,FUN=length))
  tab <- cbind(SNP=unlist(SNP),GENE=GENE)
  snp.selected <- unique(unlist(SNP))
  if(sum(snp.selected%in%colnames(data))>=length(snp.selected)) cat("Perfect all the snp considered belong to the data frame") else stop("Problem in your data")
  result <- list(data.pathway=data.pathway.bis,tab.snp.gene=tab,snp.selected=snp.selected)  
}  
  
  
  
