#' Optimal Binning using K-means Inspired Initialization (KMB)
#'
#' Performs supervised discretization of continuous numerical variables using a
#' K-means inspired binning strategy. Initial bin boundaries are determined by placing
#' centroids uniformly across the feature range and defining cuts at midpoints.
#' The algorithm then optimizes these bins using statistical constraints.
#'
#' @param feature A numeric vector representing the continuous predictor variable.
#'   Missing values (NA) should be handled prior to binning.
#' @param target An integer vector of binary outcomes (0/1) corresponding to
#'   each observation in \code{feature}. Must have the same length as \code{feature}.
#' @param min_bins Integer. The minimum number of bins to produce. Must be \eqn{\ge} 2.
#'   Defaults to 3.
#' @param max_bins Integer. The maximum number of bins to produce. Must be \eqn{\ge}
#'   \code{min_bins}. Defaults to 5.
#' @param bin_cutoff Numeric. The minimum fraction of total observations required
#'   for a bin to be considered valid. Bins smaller than this threshold are merged.
#'   Value must be in (0, 1). Defaults to 0.05.
#' @param max_n_prebins Integer. The number of initial centroids/bins to generate
#'   during the initialization phase. Defaults to 20.
#' @param enforce_monotonic Logical. If \code{TRUE}, the algorithm enforces a
#'   monotonic relationship in the Weight of Evidence (WoE) across bins.
#'   Defaults to \code{TRUE}.
#' @param convergence_threshold Numeric. The threshold for determining convergence
#'   during the iterative optimization process. Defaults to 1e-6.
#' @param max_iterations Integer. Safety limit for the maximum number of iterations.
#'   Defaults to 1000.
#'
#' @return A list containing the binning results:
#'   \itemize{
#'     \item \code{id}: Integer vector of bin identifiers.
#'     \item \code{bin}: Character vector of bin labels in interval notation.
#'     \item \code{woe}: Numeric vector of Weight of Evidence for each bin.
#'     \item \code{iv}: Numeric vector of Information Value contribution per bin.
#'     \item \code{count}: Integer vector of total observations per bin.
#'     \item \code{count_pos}: Integer vector of positive cases.
#'     \item \code{count_neg}: Integer vector of negative cases.
#'     \item \code{centroids}: Numeric vector of bin centroids (mean feature value per bin).
#'     \item \code{cutpoints}: Numeric vector of upper boundaries (excluding Inf).
#'     \item \code{total_iv}: The total Information Value of the binned variable.
#'     \item \code{converged}: Logical indicating if the algorithm converged.
#'   }
#'
#' @details
#' The KMB algorithm offers a unique initialization strategy compared to standard
#' binning methods:
#'
#' \enumerate{
#'   \item \strong{Initialization (K-means Style):}
#'   Instead of using quantiles, \code{max_n_prebins} centroids are placed uniformly
#'   across the range \eqn{[min(x), max(x)]}. Bin boundaries are then defined as the
#'   midpoints between adjacent centroids. This can lead to more evenly distributed
#'   initial bin widths in terms of the feature's scale.
#'
#'   \item \strong{Optimization:}
#'   The initialized bins undergo standard post-processing:
#'   \itemize{
#'     \item \strong{Rare Bin Merging:} Bins below \code{bin_cutoff} are merged with
#'           their most similar neighbor (by event rate).
#'     \item \strong{Monotonicity:} If \code{enforce_monotonic = TRUE}, adjacent bins
#'           violating the dominant WoE trend are merged.
#'     \item \strong{Bin Count Adjustment:} If the number of bins exceeds \code{max_bins},
#'           the algorithm greedily merges adjacent bins with the smallest absolute
#'           difference in Information Value.
#'   }
#' }
#'
#' This method can be advantageous when the underlying distribution of the feature
#' is relatively uniform, as it avoids creating overly granular bins in dense regions
#' from the start.
#'
#' @seealso \code{\link{ob_numerical_ewb}}, \code{\link{ob_numerical_cm}}
#'
#' @examples
#' # Example: Comparing KMB with EWB on uniform data
#' set.seed(123)
#' feature <- runif(1000, 0, 100)
#' target <- rbinom(1000, 1, plogis(0.02 * feature))
#'
#' result_kmb <- ob_numerical_kmb(feature, target, max_bins = 5)
#' print(result_kmb$bin)
#' print(paste("KMB Total IV:", round(result_kmb$total_iv, 4)))
#'
#' @export
ob_numerical_kmb <- function(feature, target, min_bins = 3, max_bins = 5,
                             bin_cutoff = 0.05, max_n_prebins = 20,
                             enforce_monotonic = TRUE, convergence_threshold = 1e-6,
                             max_iterations = 1000) {
  # Type Validation
  if (!is.numeric(feature)) {
    warning("Feature converted to numeric for processing.")
    feature <- as.numeric(feature)
  }

  if (!is.integer(target)) {
    target <- as.integer(target)
  }

  # Dimension Check
  if (length(feature) != length(target)) {
    stop("Length of 'feature' and 'target' must match.")
  }

  # NA Check
  if (any(is.na(feature))) {
    warning("Feature contains NA values. These should be handled before binning.")
  }

  # .Call Interface
  .Call("_OptimalBinningWoE_optimal_binning_numerical_kmb",
    target,
    feature,
    as.integer(min_bins),
    as.integer(max_bins),
    as.numeric(bin_cutoff),
    as.integer(max_n_prebins),
    as.logical(enforce_monotonic),
    as.numeric(convergence_threshold),
    as.integer(max_iterations),
    PACKAGE = "OptimalBinningWoE"
  )
}
