% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DPMGibbsSkewN_parallel.R
\name{DPMGibbsSkewN_parallel}
\alias{DPMGibbsSkewN_parallel}
\title{Parallel Implementation of Slice Sampling of Dirichlet Process Mixture of skew normal distributions}
\usage{
DPMGibbsSkewN_parallel(
  Ncpus,
  type_connec,
  z,
  hyperG0,
  a = 1e-04,
  b = 1e-04,
  N,
  doPlot = FALSE,
  nbclust_init = 30,
  plotevery = N/10,
  diagVar = TRUE,
  use_variance_hyperprior = TRUE,
  verbose = FALSE,
  monitorfile = "",
  ...
)
}
\arguments{
\item{Ncpus}{the number of processors available}

\item{type_connec}{The type of connection between the processors. Supported
cluster types are \code{"SOCK"}, \code{"FORK"}, \code{"MPI"}, and
\code{"NWS"}. See also \code{\link[parallel:makeCluster]{makeCluster}}.}

\item{z}{data matrix \code{d x n} with \code{d} dimensions in rows
and \code{n} observations in columns.}

\item{hyperG0}{prior mixing distribution.}

\item{a}{shape hyperparameter of the Gamma prior
on the concentration parameter of the Dirichlet Process. Default is \code{0.0001}.}

\item{b}{scale hyperparameter of the Gamma prior
on the concentration parameter of the Dirichlet Process. Default is \code{0.0001}. If \code{0}, 
then the concentration is fixed set to \code{a}.}

\item{N}{number of MCMC iterations.}

\item{doPlot}{logical flag indicating whether to plot MCMC iteration or not.
Default to \code{TRUE}.}

\item{nbclust_init}{number of clusters at initialization.
Default to 30 (or less if there are less than 30 observations).}

\item{plotevery}{an integer indicating the interval between plotted iterations when \code{doPlot}
is \code{TRUE}.}

\item{diagVar}{logical flag indicating whether the variance of each cluster is
estimated as a diagonal matrix, or as a full matrix.
Default is \code{TRUE} (diagonal variance).}

\item{use_variance_hyperprior}{logical flag indicating whether a hyperprior is added 
for the variance parameter. Default is \code{TRUE} which decrease the impact of the variance prior
on the posterior. \code{FALSE} is useful for using an informative prior.}

\item{verbose}{logical flag indicating whether partition info is
written in the console at each MCMC iteration.}

\item{monitorfile}{a writable \link{connections} or a character string naming a file to write into,
to monitor the progress of the analysis.
Default is \code{""} which is no monitoring.  See Details.}

\item{...}{additional arguments to be passed to \code{\link{plot_DPM}}.
Only used if \code{doPlot} is \code{TRUE}.}
}
\value{
a object of class \code{DPMclust} with the following attributes:
 \itemize{
     \item{\code{mcmc_partitions}:}{a list of length \code{N}. Each
      element \code{mcmc_partitions[n]} is a vector of length
      \code{n} giving the partition of the \code{n} observations.}
     \item{\code{alpha}:}{a vector of length \code{N}. \code{cost[j]} is the cost
associated to partition \code{c[[j]]}}
      \item{\code{U_SS_list}:}{a list of length \code{N} containing the lists of
      sufficient statistics for all the mixture components at each MCMC iteration}
     \item{\code{weights_list}:}{}
     \item{\code{logposterior_list}:}{a list of length \code{N} containing the logposterior values
      at each MCMC iterations}
     \item{\code{data}:}{the data matrix \code{d x n} with \code{d} dimensions in rows
and \code{n} observations in columns}
     \item{\code{nb_mcmcit}:}{the number of MCMC iterations}
     \item{\code{clust_distrib}:}{the parametric distribution of the mixture component - \code{"skewnorm"}}
     \item{\code{hyperG0}:}{the prior on the cluster location}
 }
}
\description{
If the \code{monitorfile} argument is a character string naming a file to
write into, in the case of a new file that does not exist yet, such a new
file will be created. A line is written at each MCMC iteration.
}
\examples{
rm(list=ls())
#Number of data
n <- 2000
set.seed(1234)

d <- 4
ncl <- 5

# Sample data

sdev <- array(dim=c(d,d,ncl))

xi <- matrix(nrow=d, ncol=ncl, c(runif(n=d*ncl,min=0,max=3)))
psi <- matrix(nrow=d, ncol=ncl, c(runif(n=d*ncl,min=-1,max=1)))
p <- runif(n=ncl)
p <- p/sum(p)
sdev0 <- diag(runif(n=d, min=0.05, max=0.6))
for (j in 1:ncl){
     sdev[, ,j] <- invwishrnd(n = d+2, lambda = sdev0)
}


c <- rep(0,n)
z <- matrix(0, nrow=d, ncol=n)
for(k in 1:n){
 c[k] = which(rmultinom(n=1, size=1, prob=p)!=0)
 z[,k] <- xi[, c[k]] + psi[, c[k]]*abs(rnorm(1)) + sdev[, , c[k]]\%*\%matrix(rnorm(d, mean = 0,
                                                                        sd = 1), nrow=d, ncol=1)
 #cat(k, "/", n, " observations simulated\n", sep="")
}

# Set parameters of G0
hyperG0 <- list()
hyperG0[["b_xi"]] <- rep(0,d)
hyperG0[["b_psi"]] <- rep(0,d)
hyperG0[["kappa"]] <- 0.001
hyperG0[["D_xi"]] <- 100
hyperG0[["D_psi"]] <- 100
hyperG0[["nu"]] <- d + 1
hyperG0[["lambda"]] <- diag(d)/10

 # hyperprior on the Scale parameter of DPM
 a <- 0.0001
 b <- 0.0001

 # do some plots
 doPlot <- TRUE
 nbclust_init <- 30


 z <- z*200
 ## Data
 ########
library(ggplot2)
 p <- (ggplot(data.frame("X"=z[1,], "Y"=z[2,]), aes(x=X, y=Y))
       + geom_point()
       + ggtitle("Simple example in 2d data")
       +xlab("D1")
       +ylab("D2")
       +theme_bw())
 p


 ## alpha priors plots
 #####################
 prioralpha <- data.frame("alpha"=rgamma(n=5000, shape=a, scale=1/b),
                         "distribution" =factor(rep("prior",5000),
                         levels=c("prior", "posterior")))
 p <- (ggplot(prioralpha, aes(x=alpha))
       + geom_histogram(aes(y=..density..),
                        colour="black", fill="white")
       + geom_density(alpha=.2, fill="red")
       + ggtitle(paste("Prior distribution on alpha: Gamma(", a,
                 ",", b, ")\n", sep=""))
      )
 p



 # Gibbs sampler for Dirichlet Process Mixtures
 ##############################################
 if(interactive()){
  MCMCsample_sn_par <- DPMGibbsSkewN_parallel(Ncpus=parallel::detectCores()-1,
                                              type_connec="SOCK", z, hyperG0,
                                              a, b, N=5000, doPlot, nbclust_init,
                                              plotevery=25, gg.add=list(theme_bw(),
                                guides(shape=guide_legend(override.aes = list(fill="grey45")))))
  plot_ConvDPM(MCMCsample_sn_par, from=2)
 }




}
\references{
Hejblum BP, Alkhassim C, Gottardo R, Caron F and Thiebaut R (2019) 
Sequential Dirichlet Process Mixtures of Multivariate Skew t-distributions for 
Model-based Clustering of Flow Cytometry Data. The Annals of Applied Statistics, 
13(1): 638-660. <doi: 10.1214/18-AOAS1209> <arXiv: 1702.04407> 
\url{https://arxiv.org/abs/1702.04407} \url{https://doi.org/10.1214/18-AOAS1209}
}
\author{
Boris Hejblum
}
