\name{generate_weights}
\alias{generate_weights}
\title{Generate a matrix containing weight combinations}
\description{
\code{generate_weights()} generates a matrix containing weight combinations for a set of variables such that each set of weights sums to 1. This can be supplied to \code{\link{calc_all_lscps}} to calculate fitness landscapes corresponding to a variety of possible sets of weights for weighting functional characteristics. The weights are generated by partitioning a weight of 1 across however many variables are requested in all possible ways.
}
\usage{
generate_weights(step, n, data = NULL, nvar = NULL,
                 varnames = NULL, verbose = TRUE)
}
\arguments{
\item{step}{
  \code{numeric}. The step size between weight partitions. Only one of \code{step} and \code{n} can be specified.
}
\item{n}{
  \code{numeric}. The number of weight partitions between 0-1. Only one of \code{step} and \code{n} can be specified.
}
\item{data}{
  an optional \code{fnc_df} (the output of \code{\link{as_fnc_df}}) or \code{kriged_surfaces} (the output \code{\link{krige_surf}}) object. The number of variabes and their names will be extracted from the data as the functional characteristics present in them.
}
\item{nvar}{
  the number of variables across which to allocate the weights. Ignored if \code{data} is not \code{NULL}. If \code{nvar = NULL} and \code{varnames} is supplied, the length of \code{varnames} will be used for \code{nvar}.
}
\item{varnames}{
  the names of the variables across which to allocate the weights. Ignored if \code{data} is not \code{NULL}. If \code{varnames = NULL} and \code{nvar} is supplied, the sequence from 1 to \code{nar} will be used for \code{varnames}.
}
\item{verbose}{
  whether to display a message noting the number of sets of weights created.
}
}
\details{
\code{generate_weights()} works by fining all possible allocations of \code{n} objects into \code{nvar} bins. When \code{step} is supplied, \code{n} is computed as \code{round(1/step)}, so the resulting weight partitions may not be exactly equal to \code{step} when its inverse is not an integer. The larger \code{n} is (or the smaller \code{step}) is, the more possible allocations will be produced (i.e., and the resulting object will have more rows). The output of \code{generate_weights()} can quickly become very large with increasing number of variables, and will make subsequent analyses slow. It is recommended to start with a large \code{step} size, or small \code{n}, and increment up.
}

\value{
A \code{grid_weights} object, which is a matrix with a row for each each set of weights and a column for each variable over which the weights are allocated. The weights in each row will sum to 1.
}

\examples{
# Allocating 10 partitions of .1 across 3 variables
wmat <- generate_weights(n = 10, nvar = 3)
head(wmat)

# Allocating 5 partitions of .2 across the 4 functional
# characteristics in the warps dataset
data("warps")

warps_fnc <- as_fnc_df(warps)
wmat <- generate_weights(n = 5, data = warps_fnc)
head(wmat)

# Using 'step' for the same result:
wmat <- generate_weights(step = .2, data = warps_fnc)
head(wmat)
}
