#' RSD Filteirng
#'
#' Filters out EICs with RSD % above a user provided threshold.
#'
#' @param peakTable peak table generated by xcms group object
#' @param eicColumn name of the EIC column
#' @param rsdColumns names of the sample columns to be used to calcualte RSD
#' @param rsdThreshold RSD percent threshold for filtering; default 0.3
#'
#' @return peakTable with filtered EICs
#'
#' @examples
#' \donttest{rsd_filtered_table <- rsdFilter(peakTable = group_table,
#'                                           eicColumn = eicColumn,
#'                                           rsdColumns = rsdColumns)}
#'
#' @import xcms
#'
#' @export


rsdFilter <- function(peakTable,
                      eicColumn,
                      rsdColumns,
                      rsdThreshold=0.3){


  if(missing(eicColumn) | missing(rsdColumns)){
    stop("One or more of the following arguments are missing values:\neicColumn\nrsdColumns\nsampleColumns")
  }

  if(missing(peakTable)){
    stop("An xcms peak table must be supplied.")
  }

  if(eicColumn %in% colnames(peakTable)==F){
    stop("eicColumn is not present in peak table")
  }

  if(any(rsdColumns %in% colnames(peakTable)==F)){
    stop("One of more of the names in the rsdColumns argument is not present in the peak table")
  }

  lqc_rsd <- apply(peakTable[,rsdColumns], 1, function(x){sd(x,na.rm=T)/mean(x,na.rm=T)})
  filterTable <- cbind(lqc_rsd = lqc_rsd, peakTable)

  # remove EICs with NA values for RSD of cc Samples
  filter_na_new_pt <- filterTable[!is.na(filterTable$lqc_rsd),]

  filter_rsd_peakTable <- filter_na_new_pt[filter_na_new_pt$lqc_rsd <= rsdThreshold,]

  peakTable <- filter_rsd_peakTable

  return(peakTable)

}
