#' @title Pools Estimates by Rubin's Rules
#'
#' @rdname pool
#'
#' @aliases pool
#'
#' @param object An object of the \code{mimira} class (produced by a previous call to \code{with()}).
#' @param dfcom A positive number representing the degrees of freedom in the data analysis. The default is \code{NULL}, which means to extract this information from the fitted model with the lowest number of observations or the first fitted model (when that fails the parameter is set to \code{999999}).
#'
#' @description \code{pool()} pools estimates from the ana;yses done withi neach imputed dataset. The typical sequence of steps to do a matching procedure on the imputed datasets are:
#' \enumerate{
#'  \item Impute the missing values using the \code{mice()} function (from the \pkg{mice} package) or the \code{amelia()} function (from the \pkg{Amelia} package), resulting in a multiple imputed dataset (an object of the \code{mids} or \code{amelia} class);
#'  \item Match or weight each imputed dataset using \code{matchthem()} or \code{weightthem()}, resulting in an object of the \code{mimids} or \code{wimids} class;
#'  \item Check the extent of balance of covariates across the matched datasets (using functions in \pkg{cobalt});
#'  \item Fit the statistical model of interest on each matched dataset by the \code{with()} function, resulting in an object of the \code{mimira} class; and
#'  \item Pool the estimates from each model into a single set of estimates and standard errors, resulting in an object of the \code{mipo} class.
#' }
#'
#' @details \code{pool()} function averages the estimates of the model and computes the total variance over the repeated analyses by Rubin’s rules. It calls \code{\link[mice:pool]{mice::pool()}} after computing the model degrees of freedom.
#'
#' @return This function returns an object of the \code{mipo} class. Methods for \code{mipo} objects (e.g., \code{print()}, \code{summary}, etc.) are available in \pkg{mice}, which does not need to be attached to use them.
#'
#' @seealso \code{\link[=with]{with()}}
#' \code{\link[mice:pool]{mice::pool()}}
#'
#' @references Stef van Buuren and Karin Groothuis-Oudshoorn (2011). \code{mice}: Multivariate Imputation by Chained Equations in \code{R}. \emph{Journal of Statistical Software}, 45(3): 1-67. \url{https://www.jstatsoft.org/v45/i03/}
#'
#' @export
#'
#' @examples \donttest{#Loading libraries
#' library(MatchThem)
#' library(survey)
#'
#' #Loading the dataset
#' data(osteoarthritis)
#'
#' #Multiply imputing the missing values
#' imputed.datasets <- mice::mice(osteoarthritis, m = 5)
#'
#' #Weighting the multiply imputed datasets
#' weighted.datasets <- weightthem(OSP ~ AGE + SEX + BMI + RAC + SMK,
#'                                 imputed.datasets,
#'                                 approach = 'within',
#'                                 method = 'ps')
#'
#' #Analyzing the weighted datasets
#' models <- with(weighted.datasets,
#'                svyglm(KOA ~ OSP, family = quasibinomial))
#'
#' #Pooling results obtained from analyzing the datasets
#' results <- pool(models)
#' summary(results)}

pool <- function (object, dfcom = NULL) {

  #External function
  #S3 method

  #Based on: mice::pool()
  #URL: <https://cran.r-project.org/package=mice>
  #URL: <https://github.com/stefvanbuuren/mice>
  #URL: <https://cran.r-project.org/web/packages/mice/mice.pdf>
  #URL: <https://www.jstatsoft.org/article/view/v045i03/v45i03.pdf>
  #Authors: Stef van Buuren et al.
  #Changes: Few

  UseMethod("pool")
}

#' @export

pool.mira <- function (object, dfcom = NULL) {

  #External function
  #S3 method

  #Based on: The mice::pool()
  #URL: <https://cran.r-project.org/package=mice>
  #URL: <https://github.com/stefvanbuuren/mice>
  #URL: <https://cran.r-project.org/web/packages/mice/mice.pdf>
  #URL: <https://www.jstatsoft.org/article/view/v045i03/v45i03.pdf>
  #Authors: Stef van Buuren et al.
  #Changes: Few

  #Importing functions
  #' @importFrom mice pool
  mice::pool
  #' @export

  #mids
  mice::pool(object = object, dfcom = dfcom)
}

#' @export

pool.mimira <- function (object, dfcom = NULL) {

  #External function
  #S3 method

  #Based on: The mice::pool()
  #URL: <https://cran.r-project.org/package=mice>
  #URL: <https://github.com/stefvanbuuren/mice>
  #URL: <https://cran.r-project.org/web/packages/mice/mice.pdf>
  #URL: <https://www.jstatsoft.org/article/view/v045i03/v45i03.pdf>
  #Authors: Stef van Buuren et al.
  #Changes: Few

  #Importing functions
  #' @importFrom mice pool getfit
  #' @importFrom utils packageVersion
  mice::pool
  #' @export

  call <- match.call()

  dfcom <- get.2dfcom(object, dfcom)

  #Make sure robust SEs are used for coxph models
  #Not needed for mice version >= 3.13.2
  if (utils::packageVersion("mice") < "3.13.2") {
    model <- mice::getfit(object, 1L)
    if (inherits(model, "coxph")) {
      for (i in seq_along(object$analyses)) {
        #Remove naive.var so summary.coxph uses var as std.error
        object$analyses[[i]]$naive.var <- NULL
      }
    }
  }

  output <- mice::pool(object, dfcom = dfcom)
  output$call <- NULL
  output <- c(list(call = call), as.list(output))
  class(output) <- c("mimipo", "mipo", "data.frame")

  #Return the output
  return(output)
}
