% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ML_MSBD.R
\name{ML_MSBD}
\alias{ML_MSBD}
\title{Full Maximum Likelihood inference of birth and death rates together with their changes along a phylogeny under a multi-type birth-death model.}
\usage{
ML_MSBD(tree, initial_values, uniform_weights = TRUE, p_lambda = 0,
  p_mu = 0, rho = 1, sigma = 0, rho_sampling = TRUE,
  lineage_counts = c(), tcut = 0, stepsize = NULL,
  no_extinction = FALSE, fixed_gamma = NULL, unique_lambda = FALSE,
  unique_mu = FALSE, optim_control = list(), attempt_remove = TRUE,
  max_nshifts = Inf, saved_state = NULL, save_path = NULL,
  time_mode = c("3pos", "tip", "mid", "root"), fast_optim = FALSE)
}
\arguments{
\item{tree}{Phylogenetic tree (in ape format) to calculate the likelihood on.}

\item{initial_values}{Initial values for the optimizer, to be provided as a vector in this order: gamma (optional), lambda, lambda decay rate (optional), mu (optional). See 'Details'.}

\item{uniform_weights}{Whether all states are weighted uniformly in shifts, default TRUE. If FALSE, the weights of states are calculated from the distributions \code{p_lambda} and \code{p_mu}. See 'Details'.}

\item{p_lambda}{Prior probability distribution on lambdas, used if \code{uniform_weights = FALSE}.}

\item{p_mu}{Prior probability distribution on mus, used if \code{uniform_weights = FALSE}.}

\item{rho}{Sampling proportion on extant tips, default 1.}

\item{sigma}{Sampling probability on extinct tips (tips are sampled upon extinction), default 0.}

\item{rho_sampling}{Whether the most recent tips should be considered extant tips, sampled with sampling proportion \code{rho}. If FALSE, all tips will be considered extinct tips, sampled with sampling probability \code{sigma}. Should be TRUE for most macroevolution datasets and FALSE for most epidemiology datasets.}

\item{lineage_counts}{For trees with clade collapsing. Number of lineages collapsed on each tip. Should be set to 1 for extinct tips.}

\item{tcut}{For trees with clade collapsing. Times of clade collapsing for each tip (i.e time of the MRCA of all collapsed lineages). Can be a single number or a vector of length the number of tips.}

\item{stepsize}{Size of the step to use for time discretization with exponential decay, default NULL. To use exponential decay, an initial value for \code{lambda_rates} should also be provided.}

\item{no_extinction}{Whether to use the Yule process (\code{mu=0}) for all states, default FALSE. If TRUE no initial value for \code{mu} is needed.}

\item{fixed_gamma}{Value to which \code{gamma} should be fixed, default NULL. If provided no initial value for \code{gamma} is needed.}

\item{unique_lambda}{Whether to use the same value of \code{lambda} for all states, default FALSE. If TRUE and exponential decay is active all states will also share the same value for \code{lambda_rate}.}

\item{unique_mu}{Whether to use the same value of \code{mu} for all states, default FALSE.}

\item{optim_control}{Control list for the optimizer, corresponds to control input in optim function, see \code{?optim} for details.}

\item{attempt_remove}{Whether to attempt to remove shifts at the end of the inference, default TRUE. If FALSE, use a pure greedy algorithm.}

\item{max_nshifts}{Maximum number of shifts to test for, default \code{Inf}.}

\item{saved_state}{If provided, the inference will be restarted from this state.}

\item{save_path}{If provided, the progress of the inference will be saved to this path after each optimization step.}

\item{time_mode}{String controlling the time positions of inferred shifts. See 'Details'.}

\item{fast_optim}{Whether to use the faster mode of optimization, default FALSE. If TRUE only rates associated with the state currently being added to the tree and its ancestor will be optimized at each step, otherwise all rates are optimized.}
}
\value{
Returns a list describing the most likely model found, with the following components:
\item{\code{likelihood}}{the negative log likelihood of the model}
\item{\code{shifts.edge}}{the indexes of the edges where shifts happen, 0 indicates the root state}
\item{\code{shifts.time}}{the time positions of shifts}
\item{\code{gamma}}{the rate of state change}
\item{\code{lambdas}}{the birth rates of all states}
\item{\code{lambda_rates}}{if exponential decay was activated, the rates of decay of birth rate for all states}
\item{\code{mus}}{the death rates of all states}
\item{\code{best_models}}{a vector containing the negative log likelihood of the best model found for each number of states tested (\code{best_models[i]} corresponds to i states, i.e i-1 shifts)}
All vectors are indexed in the same way, so that the state with parameters \code{lambdas[i]}, \code{lambda_rates[i]} and \code{mus[i]} starts on edge \code{shifts.edge[i]} at time \code{shifts.time[i]}.
}
\description{
Infers a complete MSBD model from a phylogeny, including the most likely number of states, positions and times of state changes, and parameters associated with each state. 
Uses a greedy approach to add states and Maximum Likelihood inference for the other parameters.
}
\details{
It is to be noted that all times are counted backwards, with the most recent tip positioned at 0. \cr\cr

Five time modes are possible for the input \code{time_mode}. 
In \code{tip} mode, the shifts will be placed at 10\% of the length of the edge. 
In \code{mid} mode, the shifts will be placed at 50\% of the length of the edge. 
In \code{root} mode, the shifts will be placed at 90\% of the length of the edge. 
In \code{3pos} mode, the three "tip", "mid" and "root" positions will be tested.\cr\cr

The weights w are used for calculating the transition rates q from each state i to j: \eqn{q_{i,j}=\gamma*w_{i,j}}{q(i,j)=\gamma*w(i,j)}. 
If \code{uniform_weights = TRUE}, \eqn{w_{i,j} = \frac{1}{N-1}}{w(i,j)=1/(N-1)} for all i,j, where N is the total number of states. 
If \code{uniform_weights = FALSE}, \eqn{w_{i,j} = \frac{p_\lambda(\lambda_j)p_\mu(\mu_j)}{sum_{k \ne i}p_\lambda(\lambda_k)p_\mu(\mu_k)}}{w(i,j)=p\lambda(\lambdaj)p\mu(\muj)/sum(p\lambda(\lambdak)p\mu(\muk)) for all k!=i}
where the distributions \eqn{p_\lambda}{p\lambda} and \eqn{p_\mu}{p\mu} are provided by the inputs \code{p_lambda} and \code{p_mu}.\cr\cr

Initial values for the optimization need to be provided as a vector and contain the following elements (in order): 
an initial value for gamma, which is required unless \code{fixed_gamma} is provided, 
an initial value for lambda which is always required, 
an initial value for lambda decay rate, which is required if \code{stepsize} is provided, 
and an initial value for mu, which is required unless \code{no_extinction = TRUE}. 
An error will be raised if the number of initial values provided does not match the one expected from the rest of the settings, 
and the function will fail if the likelihood cannot be calculated at the initial values.
}
\examples{
# Simulate a random phylogeny
set.seed(25)
tree <- ape::rtree(10)

# Infer the most likely multi-states birth-death model 
# with full extant & extinct sampling
\dontrun{ML_MSBD(tree, initial_values = c(0.1, 10, 1), sigma = 1, time_mode = "mid") }
# Infer the most likely multi-states birth-death model with exponential decay
# and full extant & extinct sampling
\dontrun{ML_MSBD(tree, initial_values = c(0.1, 10, 0.5, 1), sigma = 1, 
                 stepsize = 0.1, time_mode = "mid")}

# Simulate a random phylogeny with extant samples
set.seed(24)
tree2 <- ape::rcoal(10)

# Infer the most likely multi-states Yule model with partial extant sampling
\dontrun{ML_MSBD(tree2, initial_values = c(0.1, 10), no_extinction = TRUE, 
                  rho = 0.5, time_mode = "mid")}
# Infer the most likely multi-states birth-death model with full extant sampling 
# and unresolved extant tips
\dontrun{ML_MSBD(tree2, initial_values = c(0.1, 10, 1), 
                  lineage_counts = c(2,5,1,3,1,1,1,1,2,6), tcut = 0.05, time_mode = "mid")}

}
