% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/QLindley.R
\name{QLindley}
\alias{QLindley}
\alias{dqlindley}
\alias{hqlindley}
\alias{pqlindley}
\alias{qqlindley}
\alias{rqlindley}
\title{Quasi Lindley Distribution}
\source{
[dpqh]qlindley are calculated directly from the definitions. \code{rqlindley} uses either a two-component mixture of gamma distributions or the inverse transform method.
}
\usage{
dqlindley(x, theta, alpha, log = FALSE)

pqlindley(q, theta, alpha, lower.tail = TRUE, log.p = FALSE)

qqlindley(p, theta, alpha, lower.tail = TRUE, log.p = FALSE)

rqlindley(n, theta, alpha, mixture = TRUE)

hqlindley(x, theta, alpha, log = FALSE)
}
\arguments{
\item{x, q}{vector of positive quantiles.}

\item{theta}{positive parameter.}

\item{alpha}{greater than -1.}

\item{log, log.p}{logical. If TRUE, probabilities p are given as log(p).}

\item{lower.tail}{logical. If TRUE (default) \eqn{P(X \leq x)} are returned, otherwise \eqn{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{number of observations. If \code{length(n) > 1}, the length is taken to be the number required.}

\item{mixture}{logical. If TRUE (default), random values are generated from a two-component mixture of gamma distributions, otherwise from the quantile function.}
}
\value{
\code{dqlindley} gives the density, \code{pqlindley} gives the distribution function, \code{qqlindley} gives the quantile function, \code{rqlindley} generates random deviates and \code{hqlindley} gives the hazard rate function.

Invalid arguments will return an error message.
}
\description{
Density function, distribution function, quantile function, random numbers generation and hazard rate function for the quasi Lindley distribution with parameters theta and alpha.
}
\details{
Probability density function
\deqn{f(x\mid \theta ,\alpha )={\frac{\theta \left( \alpha +\theta x\right) {{e}^{-\theta x}}}{1+\alpha }}}

Cumulative distribution function
\deqn{F(x\mid \theta ,\alpha )=1-{\frac{\left( 1+\alpha +\theta x\right) }{1+\alpha }{e}^{-\theta x}}}

Quantile function
\deqn{Q(p\mid \theta ,\alpha )=-\frac{1}{{\theta }}-{\frac{\alpha }{\theta }}-\frac{1}{{\theta }}{W}_{-1}\left( \left( p-1\right) \left( 1+\alpha	\right) {{e}^{-1-\alpha }}\right) }

Hazard rate function
\deqn{h(x\mid \theta ,\alpha )=\frac{\theta \left( \alpha +\theta x\right) }{\left( 1+\alpha +\theta x\right) }}

where \eqn{W_{-1}} denotes the negative branch of the Lambert W function.

\bold{Particular cases:} \eqn{\alpha = \theta} the one-parameter Lindley distribution and for \eqn{\alpha=0} the gamma distribution with shape = 2 and scale = \eqn{\theta}.
}
\examples{
set.seed(1)
x <- rqlindley(n = 1000, theta = 1.5, alpha = 1.5, mixture = TRUE)
R <- range(x)
S <- seq(from = R[1], to = R[2], by = 0.1)
plot(S, dqlindley(S, theta = 1.5, alpha = 1.5), xlab = 'x', ylab = 'pdf')
hist(x, prob = TRUE, main = '', add = TRUE)

p <- seq(from = 0.1, to = 0.9, by = 0.1)
q <- quantile(x, prob = p)
pqlindley(q, theta = 1.5, alpha = 1.5, lower.tail = TRUE)
pqlindley(q, theta = 1.5, alpha = 1.5, lower.tail = FALSE)
qqlindley(p, theta = 1.5, alpha = 1.5, lower.tail = TRUE)
qqlindley(p, theta = 1.5, alpha = 1.5, lower.tail = FALSE)

library(fitdistrplus)
fit <- fitdist(x, 'qlindley', start = list(theta = 1.5, alpha = 1.5))
plot(fit)


}
\author{
Josmar Mazucheli \email{jmazucheli@gmail.com}

Larissa B. Fernandes \email{lbf.estatistica@gmail.com}
}
\references{
Shanker, R. and Mishra, A. (2013). A quasi Lindley distribution. \emph{African Journal of Mathematics and Computer Science Research},  \bold{6}, (4), 64-71.
}
\seealso{
\code{\link[LambertW]{W}}.
}

