% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Real-Options-OF.R
\name{LSM.RealOption.OF}
\alias{LSM.RealOption.OF}
\title{Value Operationally Flexible Capital Investment Projects Through Least-Squares Monte Carlo (LSM) Simulation:}
\usage{
LSM.RealOption.OF(
  state.variables,
  NCF,
  CAPEX,
  dt,
  rf,
  construction = 0,
  orthogonal = "Laguerre",
  degree = 9,
  cross.product = TRUE,
  Suspend.CAPEX = NULL,
  Suspend.OPEX = NULL,
  Resume.CAPEX = NULL,
  Abandon.CAPEX = NULL,
  Save.States = FALSE,
  verbose = FALSE,
  debugging = FALSE
)
}
\arguments{
\item{state.variables}{\code{matrix} or \code{array}. The simulated states of the underlying stochastic variables. The first dimension corresponds to the simulated values
of the state variables at each discrete observation point. The second dimension corresponds to each individual simulation of the state variable. The third dimension
corresponds to each state variable considered.}

\item{NCF}{The net cash flows resulting from operating the underlying capital investment project for one discrete time step at the current simulated values of the state variables.
Each column corresponds to a simulated price path of underlying stochastic variables, and each row the net cash flows at a discrete time point for each simulated path.
The dimensions of object 'NCF' must match the first two dimensions of the object passed to the 'state.variables' argument.}

\item{CAPEX}{\code{numeric} or \code{vector} object. The initial capital investment of the project. This value can be either constant or time dependent. When the 'CAPEX' argument
is time dependent, it must be a vector object of length equal to the number of discrete observations of the simulations (i.e. the number of rows of both 'state.variables' and 'NCF')}

\item{dt}{Constant, discrete time step of simulated observations}

\item{rf}{The annual risk-free interest rate}

\item{construction}{An integer corresponding to the number of periods of construction of the underlying asset. The construction time represents the time between
the initial capital expenditure and when net cash flows are accrued, representing the construction time required of the investment project.}

\item{orthogonal}{\code{character}. The orthogonal polynomial used to develop basis functions that estimate the continuation value in the LSM simulation method.
\code{orthogonal} arguments available are: "Power", "Laguerre", "Jacobi", "Legendre", "Chebyshev", "Hermite". See \bold{details}.}

\item{degree}{The number of orthogonal polynomials used in the least-squares fit. See \bold{details}.}

\item{cross.product}{\code{logical}. Should a cross product of state variables be considered? Relevant only when the number of state variables
is greater than one}

\item{Suspend.CAPEX}{The cash flow resulting from suspending an operational project}

\item{Suspend.OPEX}{the ongoing cash flow resulting from being suspended for one discrete time period}

\item{Resume.CAPEX}{the cash flow resulting from resuming a suspended project}

\item{Abandon.CAPEX}{the terminal cash flow resulting from permanently abandoning a project}

\item{Save.States}{\code{logical}. Should the operational state of each simulated price path be returned? See \bold{values}.}

\item{verbose}{\code{logical}. Should additional information be output? See \bold{values}.}

\item{debugging}{\code{logical} Should additional simulation information be output? See \bold{values}.}
}
\value{
The \code{LSM.RealOption.OF} function by default returns a \code{list} object containing 3 objects, with further objects returned within this list when the
\code{Save.States}, \code{verbose} and \code{debugging} arguments are set to \code{TRUE}.

The objects returned are:
\tabular{ll}{

\code{ROV} \tab Real option value. The value of an investment project when actively managing the investment and appropriately exercising options \cr
\code{NPV + OF} \tab Net present Value + operational flexibility. The value immediate investment into an investment project when actively managing and appropriately exercising operational flexibility \cr
\code{NPV} \tab  Net present value. The value immediate investment into an investment project \cr
\code{WOV} \tab  Waiting option value. The value of the option to delay initial investment into an investment project\cr
}

When \code{Save.States = TRUE}, an additional object is returned:
\tabular{ll}{

\code{Project States} \tab  A \code{data.frame} object detailing the proportion of simulated price paths in each operating state at each discrete time point\cr
}


When \code{verbose = T}, an additional 7 objects are returned within the \code{list} class object, providing further analysis
into the capital investment project:
\tabular{ll}{
\code{ROV SE} \tab  The standard error of 'ROV'. \cr
\code{NPV SE} \tab  The standard error of 'NPV'. \cr
\code{WOV SE} \tab  The standard error of 'WOV'. \cr
\code{Expected Timing} \tab  The expected timing of investment, in years. \cr
\code{Expected Timing SE} \tab  The standard error of the expected timing of investment. \cr
\code{Investment Prob} \tab  The probability of investment within the forecasting horizon. \cr
\code{Cumulative Investment Prob} \tab The cumulative probability of investment at each discrete time point over the forecasting horizon. \cr
}

When \code{debugging = T}, an additional 5 objects are returned within the \code{list} class object.
These objects provide information about the values of individual simulated price paths:
\tabular{ll}{
\code{Investment Period} \tab The time of investment of invested price paths. Price paths that did not trigger investment are represented as \code{NA} \cr
\code{Project Value} \tab The calculated project value at time zero for each simulated price path. The 'ROV' is equal to the mean of this vector. \cr
\code{Immediate Profit} \tab The profit resulting from immediate investment for each discrete time period and for all simulated price paths  \cr
\code{Running Present Value} \tab  The present value of all future cash flows of an investment project for each discrete time period and for all simulated price paths\cr
\code{Path Project States} \tab The current project state  for each discrete time period and for all simulated price paths \cr
}
}
\description{
Given a set of state variables and associated net cash flows for an investment project simulated through Monte Carlo simulation,
solve for the real option value of a capital investment project that has the flexibility to temporarily suspend or permanently abandon operations
through the least-squares Monte Carlo simulation method.
}
\details{
The \code{LSM.RealOption.OF} function provides an implementation of the least-squares Monte Carlo (LSM) simulation approach to numerically approximate
the value of capital investment projects considering the flexibility of timing of investment and operational states under stochastically evolving uncertainty. The function provides flexibility in the stochastic behavior of the underlying uncertainty, with simulated values
of state variables provided within the \code{state.variables} argument. Multiple stochastically evolving uncertainties can also be considered. The \code{LSM.RealOption.OF} function also provides analysis into the
expected investment timing, probability, and operating modes of the project.

\bold{Least-Squares Monte Carlo Simulation:}

The least-squares Monte Carlo (LSM) simulation method is a numeric approach first presented by Longstaff and Schwartz (2001) that
approximates the value of options with early exercise opportunities. The LSM simulation method is considered one of the most efficient
methods of valuing American-style options due to its flexibility and computational efficiency. The approach can feature multiple
stochastically evolving underlying uncertainties, following both standard and exotic stochastic processes.

The LSM method first approximates stochastic variables through a stochastic process to develop cross-sectional information,
then directly estimates the continuation value of in-the-money simulation paths by "(regressing) the ex-post realized payoffs from
continuation on functions of the values of the state variables" (Longstaff and Schwartz, 2001).

\bold{Real Options Analysis}

Real options analysis of investment projects considers the value of the option to delay investment in a project until
underlying, stochastically evolving uncertainty has revealed itself. Real options analysis treats investment into capital investment projects as an optimal stopping problem, optimizing the timing of investment that maximizes the payoffs of investment under underlying stochastic uncertainty.
Real options analysis is also capable of evaluating the critical value of underlying state variables at which immediate investment into a project is optimal. See Dixit and Pindyck (1994) for more details of real options analysis.

The \code{LSM.RealOption.OF} function considers the option to invest into a capital investment project within a finite forecasting horizon.
Investment into the project results in accruing all future net cash flows (NCF) until the end of the forecasting horizon at the cost of the capital expenditure (CAPEX).

The primary difference between the \code{LSM.RealOption} and \code{LSM.AmericanOption.OF} function is the way in which they evaluate the payoff of
exercise of the American-style option. The \code{LSM.AmericanOption} function considers the payoff of exercise to be a one time payoff (i.e. buying or selling the security in a vanilla call or put option) corresponding to the respective payoff argument.
The \code{LSM.RealOption.OF} function instead, at each discrete time period, for each simulated price path, compares the sum of all remaining discounted cash flows that are accrued following immediate investment into
a project to the end of the forecasting horizon with the expected value of delaying investment. This is is known as the 'running present value' (RPV) of the project, which is the discretised present value of all
future cash flows of the project. The RPV of a project increases as the size of the discrete time step decreases, highlighting the need
for small discrete time steps to accurately value investment projects. This is due to the discounting effect of discounting larger
cash flows over greater time periods compared to smaller cash flows being discounted more frequently.

\bold{Orthogonal Polynomials:}

To improve the accuracy of estimation of continuation values, the economic values in each period are regressed on a linear
combination of a set of basis functions of the stochastic variables. These estimated regression parameters and the simulated
stochastic variables are then used to calculate the estimator for the expected economic values.

Longstaff and Schwartz (2001) state that as the conditional expectation of the continuation value belongs to a Hilbert space,
it can be represented by a combination of orthogonal basis functions. Increasing the number of stochastic state variables
therefore increases the number of required basis functions exponentially. The orthogonal polynomials available in the
\code{LSM.RealOptions} package are: Laguerre, Jacobi, Legendre (spherical), Hermite (probabilistic), Chebyshev (of the first kind).
The simple powers of state variables is further available. Explicit expressions of each of these orthogonal polynomials are
available within the textbook of Abramowitz and Stegun (1965).

\bold{Operational Flexibility:}

Operational flexibility is adjustment made to a project in response to, or anticipation of, positive or negative economic outcomes
that occur after a project has begun. Operational flexibility has the ability to mitigate the effect of poor timing of initial
investment by maximizing gains and minimizing losses. Operational flexibility that reduces losses in response to negative economic
conditions (e.g. output prices falling to levels where operations are no longer profitable) reduce the irreversibility of initial
investment into a project. This is particularly valuable for investment projects in highly competitive business operating conditions,
high initial investment costs, long residual lifetimes of operations, high levels of volatility in underlying assets and net present
values near zero.

The operational flexibility supported by the \code{LSM.RealOption.OF} function is the ability to temporarily suspend and permanently
abandon an operational investment project. The ability to temporarily suspend and permanently abandon operations was first presented
in the prolific work of Brennan and Schwartz (1985), whom solved for the trigger values of suspension, resumption and abandonment of
a finite natural resource investment under natural resource price uncertainty. The work of Cortazar, Gravet and Urzua (2008) solved
this optimal switching problem using the LSM simulation method.

Operational flexibilities are path dependent, compound and mutually exclusive real options. Under consideration of operational
flexibility, there are several different "states" or operating modes available for an investment project. These are:

* Not yet invested
* Operational
* Suspended
* Abandoned

With the "Under construction" state further considered when construction time is also considered.

Generally, there is a cost of switching between operating modes (for example, switching from the "not yet invested" state into the
"operational" state incurs the cost of the initial capital investment, 'CAPEX'). The costs incurred in temporarily suspending,
resuming, or abandoning a project could be attributed to firing/hiring project employees, loss in perishable products and general cost
of moving a project into a 'suspended' state. Operational expenditures of a suspended project could include maintenance to inactive
machinery, preventing weathering of project sites, ongoing rent or utility payments, etc. Finally, the abandonment expenditure could
be negative, representing a 'scrap' value of the project that is returned as project assets are sold on abandonment of the project.

Solving for the value of an investment project with operational flexibility is classified as an optimal switching or optimal control
model. The objective is to optimize the value of the project by taking the optimal set of decisions for the project operating mode
at each time point under uncertainty.

The 'LSM.RealOption.OF' function solves for the optimal switching problem at each discrete time point throughout the backward induction
process. For each operating mode, the immediate payoff of switching to a different operating mode is compared against the discounted 'continuation' value
of staying in the current operating mode. These continuation values are directly estimated through least-squares regression of all simulated price paths (rather than just 'in-the-money' price paths, as is
considered in the value of the option to delay the initial investment decision).

The 'ROV' value of the investment project is then calculated by considering the option to invest into the 'operational' state. Investment into the project is optimal when the value of the 'operational' state is
greater than the expected continuation value.

The introduction of operational flexibility to the LSM simulation method greatly increases the level of computational complexity and thus the
\code{LSM.RealOption.OF} function is considerably slower than the \code{LSM.RealOption} function.
}
\examples{

# Example: Value a capital investment project where the revenues follow a
# Geometric Brownian Motion stochastic process:

## Step 1 - Simulate asset prices:
AssetPrices <- GBM.Simulate(n = 100, t = 10, mu = 0.05,
                         sigma = 0.2, S0 = 100, dt = 1/2)

## Step 2 - Perform Real Option Analysis (ROA):
ROA <- LSM.RealOption.OF(state.variables = AssetPrices,
                     NCF = AssetPrices - 100,
                     CAPEX = 1000,
                     dt = 1/2,
                     rf = 0.05,
                     Suspend.CAPEX = 100,
                     Suspend.OPEX = 10,
                     Resume.CAPEX = 100,
                     Abandon.CAPEX = 0
                     )

}
\references{
Abramowitz, M., and I. A. Stegun, (1965). Handbook of mathematical functions with formulas, graphs, and mathematical tables. Courier Corporation.

Brennan, M. J., and E. S. Schwartz, (1985). Evaluating natural resource investments. Journal of business, 135-157.

Dixit, A. K., and R. S. Pindyck, (1994). Investment under uncertainty. Princeton university press.

Longstaff, F. A., and E. S. Schwartz, (2001). Valuing American options by simulation: a simple least-squares approach. The review of financial studies, 14(1), 113-147.

Cortazar, G., M. Gravet, and J. Urzua, (2008). The valuation of multidimensional American real options using the LSM simulation method. Computers & Operations Research, 35(1), 113-129.
}
