% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IIWcode.R
\name{iiw.weights}
\alias{iiw.weights}
\title{Compute inverse-intensity weights.}
\usage{
iiw.weights(
  formulaph,
  formulanull = NULL,
  data,
  id,
  time,
  event,
  lagvars,
  invariant = NULL,
  maxfu,
  lagfirst = lagfirst,
  first,
  frailty = FALSE
)
}
\arguments{
\item{formulaph}{the formula for the proportional hazards model for the visit intensity that will be used to derive inverse-intensity weights. The formula should usually use the counting process format (i.e. Surv(start,stop,event)). If a frailty model is used, the cluster(id) term should appear before other covariates}

\item{formulanull}{if stabilised weights are to be used, the formula for the null model used to stabilise the weights}

\item{data}{data frame containing the variables in the model}

\item{id}{character string indicating which column of the data identifies subjects}

\item{time}{character string indicating which column of the data contains the time at which the visit occurred}

\item{event}{character string indicating which column of the data indicates whether or not a visit occurred. If every row corresponds to a visit, then this column will consist entirely of ones}

\item{lagvars}{a vector of variable names corresponding to variables which need to be lagged by one visit to fit the visit intensity model. Typically time will be one of these variables. The function will internally add columns to the data containing the values of the lagged variables from the previous visit. Values of lagged variables for a subject's first visit will be set to NA. To access these variables in specifying the proportional hazards formulae, add ".lag" to the variable you wish to lag. For example, if time is the variable for time, time.lag is the time of the previous visit}

\item{invariant}{a vector of variable names corresponding to variables in data that are time-invariant. It is not necessary to list every such variable, just those that are invariant and also included in the proportional hazards model}

\item{maxfu}{the maximum follow-up time(s). If everyone is followed for the same length of time, this can be given as a single value. If individuals have different follow-up times, maxfu should have the same number of elements as there are rows of data}

\item{lagfirst}{A vector giving the value of each lagged variable for the first time within each subject. This is helpful if, for example, time is the variable to be lagged and you know that all subjects entered the study at time zero}

\item{first}{logical variable. If TRUE, the first observation for each individual is assigned an intensity of 1. This is appropriate if the first visit is a baseline visit at which recruitment to the study occurred; in this case the baseline visit is observed with probability 1.}

\item{frailty}{logical variable. If TRUE, a frailty model is fit to calculate the inverse intensity weights. If FALSE, a marginal semi-parametric model is fit. Frailty models are helpful when fitting semi-parametric joint models.}
}
\value{
a vector of inverse-intensity weights, ordered on id then time
}
\description{
Since the vector of weights is ordered on id and time, if you intend to merge these weights onto your original dataset it is highly recommended that you sort the data before running iiw.weights
}
\details{
Given longitudinal data with irregular visit times, fit a Cox proportional hazards model for the visit intensity, then use it to compute inverse-intensity weights
}
\examples{
library(nlme)
data(Phenobarb)
library(survival)
library(geepack)
Phenobarb$event <- 1-as.numeric(is.na(Phenobarb$conc))
data <- lagfn(Phenobarb, lagvars="dose", id="Subject", time="time", lagfirst = 0)
data <- lagfn(data, lagvars="dose.lag", id="Subject", time="time", lagfirst = 0)
data <- lagfn(data, lagvars="dose.lag.lag", id="Subject", time="time", lagfirst = 0)
data$dose.lag[is.na(data$dose.lag)] <- data$dose.lag.lag[is.na(data$dose.lag)]
data$dose.lag[is.na(data$dose.lag)] <- data$dose.lag.lag.lag[is.na(data$dose.lag)]
data <- data[data$event==1,]
data$id <- as.numeric(data$Subject)
data <- data[data$time<16*24,]
i <- iiw.weights(Surv(time.lag,time,event)~Wt *dose.lag + dose.lag*(I(conc.lag>0) + conc.lag) +
cluster(Subject),id="id",time="time",event="event",data=data,
invariant="Subject",lagvars=c("time","conc"),maxfu=16*24,lagfirst=0,first=TRUE)
data$weight <- i$iiw.weight
summary(i$m)
# can use to fit a weighted GEE
mw <- geeglm(conc ~ I(time^3) + log(time) , id=Subject, data=data, weights=weight)
summary(mw)
# agrees with results through the single command iiwgee
miiwgee <- iiwgee(conc ~ I(time^3) + log(time),Surv(time.lag,time,event)~Wt *dose.lag +
dose.lag*(I(conc.lag>0) + conc.lag) +cluster(Subject),
formulanull=NULL,id="id",time="time",event="event",data=data,
invariant="id",lagvars=c("time","conc"),maxfu=16*24,lagfirst=0,first=TRUE)
summary(miiwgee$geefit)
}
\references{
\itemize{
\item Lin H, Scharfstein DO, Rosenheck RA. Analysis of Longitudinal data with Irregular, Informative Follow-up. Journal of the Royal Statistical Society, Series B (2004), 66:791-813
\item Buzkova P, Lumley T. Longitudinal data analysis for generalized linear models with follow-up dependent on outcome-related variables. The Canadian Journal of Statistics 2007; 35:485-500.}
}
\seealso{
Other iiw: 
\code{\link{iiwgee}()},
\code{\link{iiw}()}

Other iiw: 
\code{\link{iiwgee}()},
\code{\link{iiw}()}
}
\concept{iiw}
