#' importPlatemapXML
#'
#' Imports .Platemap XML files from Incucyte Zoom software and extracts information in the Compound,
#' GrowthCondition and CellType fields.
#'
#' @param filepath Path to a .Platemap XML file generated by the Incucyte Zoom software
#' @param control_cpd Specify the compound to use as baseline.  Defaults to DMSO
#'
#' @return data frame
#' @importFrom magrittr %>%
#' @export
#'
#' @examples
#' #example data 1
#' pm_file <- system.file(file='extdata/example.PlateMap', package='IncucyteDRC')
#' test_pm <- importPlatemapXML(pm_file)
#' head(test_pm)
#'
#' #example data 2
#' pm_file2 <- system.file(file='extdata/example2.PlateMap', package='IncucyteDRC')
#' test_pm2 <- importPlatemapXML(pm_file2)
#' head(test_pm2)
importPlatemapXML <- function(filepath, control_cpd='DMSO') {
    message(sprintf("Importing platemap xml from %s",filepath))
    platemap_xml <- XML::xmlTreeParse(filepath)

    #import the XML file
    xml.top <- XML::xmlRoot(platemap_xml[[1]])
    #reference the wellstore tree
    wellstore <- xml.top[[2]][[1]]
    #set up the output dataframe
    platemap_list <- list()
    #iterate through wells
    for (i in 1:length(wellstore)) {
        #print(i)
        w <- wellstore[i]
        this_row <- as.numeric(XML::xmlGetAttr(w[[1]],'row')) + 1
        this_col <- as.numeric(XML::xmlGetAttr(w[[1]],'col')) + 1
        #default values
        this_samptype <- 'B'
        this_sampleid <- '-'
        this_sampleconc <- NA
        this_sampleconcunits <- NA
        this_growthcondition <- NA
        this_celltype <- NA
        this_celltype_passage <- NA
        this_celltype_seedingdensity <- NA

        if (XML::xmlSize(w[[1]]) > 0) { #blanks don't have the rest of the xml tree, so detect this using xmlSize function

            items <- w[[1]][[1]]
            for (k in 1:length(items)) {
                j<- items[[k]]
                if (XML::xmlGetAttr(j, 'type') == 'Compound') {
                    this_samptype <- 'S'
                    this_sampleid <- XML::xmlGetAttr(j[[1]], 'description')
                    this_sampleconc <- XML::xmlGetAttr(j, 'concentration')
                    this_sampleconcunits <- XML::xmlGetAttr(j, 'concentrationUnits')
                } else if (XML::xmlGetAttr(j, 'type') == 'GrowthCondition') {
                    this_growthcondition <- XML::xmlGetAttr(j[[1]], 'description')
                }  else if (XML::xmlGetAttr(j, 'type') == 'CellType') {
                    this_celltype <- XML::xmlGetAttr(j[[1]], 'description')
                    this_celltype_passage <- XML::xmlGetAttr(j, 'passage')
                    this_celltype_seedingdensity <- XML::xmlGetAttr(j, 'seedingDensity')
                }
            }
        }

        #check for DMSO
        if (grepl ( control_cpd , this_sampleid, ignore.case=T  ) ) { this_samptype <- 'C'  }

        #create a dataframe
        this_well <- data.frame(row=as.numeric(this_row),
                                col=as.numeric(this_col),
                                sampleid=as.character(this_sampleid),
                                conc=as.numeric(this_sampleconc),
                                samptype=as.character(this_samptype),
                                concunits=as.character(this_sampleconcunits),
                                growthcondition=as.character(this_growthcondition),
                                celltype=as.character(this_celltype),
                                passage=as.character(this_celltype_passage),
                                seedingdensity=as.character(this_celltype_seedingdensity),
                                stringsAsFactors=F)

        platemap_list[[i]] <- this_well

    }

    #make a data fram from the list
    platemap_df <- dplyr::bind_rows(platemap_list) %>%
        dplyr::mutate(wellid = paste0(LETTERS[row], col)) %>%
        as.data.frame()

    attr(platemap_df, 'IncucyteDRCPlatemap') <- TRUE

    message('Plate map import successful!')
    return (platemap_df)

}
