#' Plotting output and parameters of inferential interest for IMIFA and related models
#'
#' @param x An object of class "\code{Results_IMIFA}" generated by \code{\link{get_IMIFA_results}}.
#' @param plot.meth The type of plot to be produced for the \code{param} of interest, where \code{correlation} refers to ACF/PACF plots, \code{means} refers to posterior means, \code{density}, \code{trace} and \code{parallel.coords} are self-explanatory. "\code{all}" in this case, the default, refers to {\code{trace, density, means, correlation}}. \code{parallel.coords} is only available when \code{param} is one of \code{means}, \code{loadings} or \code{uniquenesses} - note that this method applies a small amount of horizontal jitter to avoid overplotting. Special types of plots which don't require a \code{param} are \code{GQ}, for plotting the posterior summaries of the numbers of groups/factors, if available, \code{zlabels} for plotting clustering uncertainties if clustering has taken place (and, if available, the average similarity matrix, reorder according to the map labels) with or without the clustering labels being supplied via the \code{zlabels} argument), and \code{errors} for visualing the difference between the estimated and empirical covariance matrix/matrices.
#' @param param The parameter of interest for any of the following \code{plot.meth} options: \code{trace}, \code{density}, \code{means}, \code{correlation}. The \code{param} must have been stored when \code{\link{mcmc_IMIFA}} was initially ran. Includes \code{pis} for methods where clustering takes place, and allows posterior inference on \code{alpha} and \code{discount} for the "\code{IMFA}" and "\code{IMIFA}" methods.
#' @param zlabels The true labels can be supplied if they are known. If this is not supplied, the function uses the labels that were supplied, if any, to \code{\link{get_IMIFA_results}}. Only relevant when \code{plot.meth = "zlabels"}.
#' @param load.meth Switch which allows plotting posterior mean loadings as a heatmap (the default), or as something akin to \code{link{plot}} with \code{type = "h"}. Only relevant if \code{param = "loadings"}. Heatmaps are produced with the aid of \code{\link{mat2cols}} and \code{\link[gclus]{plotcolors}}.
#' @param palette An optional colour palette to be supplied if overwriting the default palette set inside the function by \code{\link[viridis]{viridis}} is desired.
#' @param g Optional argument that allows specification of exactly which cluster the plot of interest is to be produced for. If not supplied, the user will be prompted to cycle through plots for all clusters. Also functions as an index for which plot to return when \code{plot.meth} is \code{GQ} or \code{zlabels} in much the same way.
#' @param mat Logical indicating whether a \code{\link{matplot}} is produced (defaults to \code{TRUE}). If given as \code{FALSE}, \code{ind} is invoked.
#' @param ind Either a single number indicating which variable to plot when \code{param} is one of \code{means} or \code{uniquenesses}, or which cluster to plot if \code{param} is \code{pis}. If \code{scores} are plotted, a vector of length two giving which observation and factor to plot; If \code{loadings} are plotted, a vector of length two giving which variable and factor to plot. Only relevant when \code{mat} or \code{by.fac} is \code{FALSE}.
#' @param fac Optional argument that provides an alternative way to specify \code{ind[2]} when \code{mat} is \code{FALSE} and \code{param} is one of \code{scores} or \code{loadings}.
#' @param by.fac Optionally allows (mat)plotting of scores and loadings by factor - i.e. observation(s) (scores) or variable(s) (loadings) for a given factor, respectively, controlled by \code{ind} or \code{fac}) when set to \code{TRUE}. Otherwise factor(s) are plotted for a given observation or variable when set to \code{FALSE} (the default), again controlled by \code{ind} or \code{fac}. Only relevant when \code{param} is one of \code{scores} or \code{loadings}.
#' @param type The manner in which the plot is to be drawn, as per the \code{type} argument to \code{\link{plot}}.
#' @param intervals Logical indicating whether credible intervals around the posterior mean(s) are to be plotted when \code{is.element(plot.meth, c("all", "means"))}. Defaults to \code{TRUE}.
#' @param partial Logical indicating whether plots of type "\code{correlation}" use the PACF. The default, \code{FALSE}, ensures the ACF is used. Only relevant when \code{plot.meth = "all"}, otherwise both plots are produced when \code{plot.meth = "correlation"}.
#' @param titles Logical indicating whether default plot titles are to be used (\code{TRUE}), or suppressed (\code{FALSE}).
#' @param transparency A factor in [0, 1] modifying the opacity for overplotted lines. Defaults to 0.75.
#' @param ... Other arguments typically passed to \code{\link{plot}}.
#'
#' @return The desired plot with appropriate output and summary statistics printed to the console screen.
#' @export
#' @import graphics
#' @importFrom grDevices "adjustcolor" "col2rgb" "palette" "heat.colors"
#' @importFrom Rfast "Order" "med" "colMedians"
#' @importFrom plotrix "plotCI"
#' @importFrom gclus "plotcolors"
#' @importFrom e1071 "classAgreement"
#' @importFrom mclust "classError"
#' @importFrom viridis "viridis"
#' @seealso \code{\link{mcmc_IMIFA}}, \code{\link{get_IMIFA_results}}, \code{\link{mat2cols}}, \code{\link[gclus]{plotcolors}}
#' @references Murphy, K., Gormley, I. C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \code{https://arxiv.org/abs/1701.07010}
#'
#' @examples
#' # See the vignette associated with the package for more graphical examples:
#' # vignette("IMIFA", package = "IMIFA")
#'
#' # data(olive)
#' # area     <- olive$area
#' # simIMIFA <- mcmc_IMIFA(olive, method="IMIFA")
#' # resIMIFA <- get_IMIFA_results(simIMIFA, z.avgsim=TRUE)
#'
#' # Examine the posterior distribution(s) of the number(s) of clusters (G) &/or latent factors (Q)
#' # For the IM(I)FA and OM(I)FA methods, this also plots the trace of the active/non-empty clusters
#' # plot(resIMIFA, plot.meth="GQ")
#' # plot(resIMIFA, plot.meth="GQ", g=2)
#'
#' # Plot clustering uncertainty (and, if available, the similarity matrix)
#' # plot(resIMIFA, plot.meth="zlabels", zlabels=area)
#'
#' # Visualise empirical vs. estimated covariance error metrics
#' # plot(resIMIFA, plot.meth="errors")
#'
#' # Look at the trace, density, posterior mean and correlation of various parameters of interest
#' # plot(resIMIFA, plot.meth="all", param="means", g=1)
#' # plot(resIMIFA, plot.meth="all", param="means", g=1, ind=2)
#' # plot(resIMIFA, plot.meth="all", param="scores")
#' # plot(resIMIFA, plot.meth="all", param="scores", by.fac=TRUE)
#' # plot(resIMIFA, plot.meth="all", param="loadings", g=1)
#' # plot(resIMIFA, plot.meth="all", param="loadings", g=1, load.meth="raw")
#' # plot(resIMIFA, plot.meth="parallel.coords", param="uniquenesses")
#' # plot(resIMIFA, plot.meth="all", param="pis", intervals=FALSE, partial=TRUE)
#' # plot(resIMIFA, plot.meth="all", param="alpha")
plot.Results_IMIFA  <- function(x = NULL, plot.meth = c("all", "correlation", "density", "errors", "GQ", "means", "parallel.coords", "trace", "zlabels"),
                                param = c("means", "scores", "loadings", "uniquenesses", "pis", "alpha", "discount"), zlabels = NULL, load.meth = c("heatmap", "raw"), palette = NULL, g = NULL,
                                mat = TRUE, ind = NULL, fac = NULL, by.fac = FALSE, type = c("h", "n", "p", "l"), intervals = TRUE, partial = FALSE, titles = TRUE, transparency = 0.75, ...) {

  if(missing(x))                      stop("'x' must be supplied")
  if(!exists(deparse(substitute(x)),
             envir=.GlobalEnv))       stop(paste0("Object ", match.call()$x, " not found\n"))
  if(class(x) != "Results_IMIFA")     stop(paste0("Results object of class 'Results_IMIFA' must be supplied"))
  GQ.res  <- x$GQ.results
  G       <- GQ.res$G
  Gseq    <- seq_len(G)
  Qs      <- GQ.res$Q
  Q.max   <- max(Qs)
  defpar  <- suppressWarnings(par(no.readonly=TRUE))
  defpar$new        <- FALSE
  mispal            <- missing(palette)
  if(mispal)             palette <- viridis(min(10, max(G, Q.max, 5)))
  if(!all(.are_cols(cols=palette)))   stop("Supplied colour palette contains invalid colours")
  if(length(palette) < 5)             stop("Palette must contain 5 or more colours")
  if(length(transparency) != 1   &&
     any(!is.numeric(transparency),
         (transparency     < 0 ||
          transparency     > 1)))     stop("'transparency' must be a single number in [0, 1]")
  tmp.pal <- palette
  palette <- adjustcolor(palette, alpha.f=transparency)
  palette(palette)
  grey    <- adjustcolor("#999999", alpha.f=0.3)
  defopt  <- options()
  options(warn=1)
  suppressWarnings(par(cex.axis=0.8, new=FALSE))
  on.exit(suppressWarnings(par(defpar)))
  on.exit(do.call(clip, as.list(defpar$usr)), add=TRUE)
  on.exit(palette("default"), add=TRUE)
  on.exit(suppressWarnings(options(defopt)),  add=TRUE)
  n.grp   <- attr(GQ.res, "Groups")
  n.fac   <- attr(GQ.res, "Factors")
  G.supp  <- attr(GQ.res, "Supplied")["G"]
  Q.supp  <- attr(GQ.res, "Supplied")["Q"]
  method  <- attr(x, "Method")
  store   <- attr(x, "Store")
  n.var   <- attr(x, "Vars")
  n.obs   <- attr(x, "Obs")
  z.sim   <- attr(x, "Z.sim")
  if(missing(plot.meth))              stop("'plot.meth' not supplied:\nWhat type of plot would you like to produce?")
  if(is.element(plot.meth,
     c("G", "Q",
       "QG")))  {      plot.meth <- "GQ"
  }
  uni.type     <- unname(attr(x, "Uni.Meth")['Uni.Type'])
  plot.meth    <- match.arg(plot.meth)
  if(!is.element(plot.meth, c("errors", "GQ", "zlabels")) &&
     missing(param))                  stop("'param' not supplied:\nWhat variable would you like to plot?")
  param        <- match.arg(param)
  load.meth    <- match.arg(load.meth)
  type.x       <- missing(type)
  type         <- match.arg(type)
  m.sw         <- c(G.sw = FALSE, Z.sw = FALSE, E.sw = FALSE, P.sw = FALSE, C.sw = FALSE, D.sw = FALSE, M.sw = FALSE, T.sw = FALSE)
  v.sw         <- attr(x, "Switch")
  obs.names    <- if(v.sw["s.sw"]) rownames(x$Scores$post.eta) else attr(x, "Obsnames")
  var.names    <- if(v.sw["l.sw"]) rownames(x$Loadings$post.load[[1]]) else if(v.sw["mu.sw"]) rownames(x$Means$post.mu) else if(v.sw["psi.sw"]) rownames(x$Uniquenesses$post.psi) else attr(x, "Varnames")
  obs.names    <- if(is.null(obs.names)) seq_len(n.obs) else obs.names
  var.names    <- if(is.null(var.names)) seq_len(n.var) else var.names
  names(v.sw)  <- formals(sys.function(sys.parent()))$param
  ci.sw        <- v.sw
  all.ind      <- plot.meth == "all"
  grp.ind      <- !is.element(method, c("FA", "IFA"))
  if(grp.ind)   {
    clust      <- x$Clust
    grp.size   <- clust$post.sizes
    labelmiss  <- !is.null(attr(clust, "Label.Sup")) && !attr(clust, "Label.Sup")
  }
  grp.ind      <- all(G != 1, grp.ind)
  if(all.ind)   {
    if(v.sw[param]) {
      m.sw[-(1:4)]  <- !m.sw[-(1:4)]
      layout(matrix(c(1, 2, 3, 4), nrow=2, ncol=2, byrow=TRUE))
      par(cex=0.8, mai=c(0.5, 0.5, 0.5, 0.2), mgp=c(2, 1, 0), oma=c(0, 0, 2, 0))
    }
  } else {
    sw.n  <- paste0(toupper(substring(plot.meth, 1, 1)), ".sw")
    m.sw[sw.n] <- TRUE
  }
  z.miss  <- missing(zlabels)
  if(!z.miss) {
    z.nam <- gsub("[[:space:]]", "", deparse(substitute(zlabels)))
    nam.z <- gsub("\\[.*", "", z.nam)
    nam.x <- gsub(".*\\[(.*)\\].*", "\\1)", z.nam)
    if(grepl("$", z.nam, fixed=TRUE)) {
      x.nam    <- strsplit(nam.z, "$", fixed=TRUE)[[1]]
      nam.z    <- z.nam  <- x.nam[1]
      if(x.nam[2] %in% colnames(get(nam.z))) {
        zlabels <- get(nam.z)[x.nam[2]][,1]
      } else                          stop(paste0("'", x.nam[2], "' not found within '", nam.z, "'"))
    }
    ptrn  <- c("(", ")")
    if(!exists(nam.z,
               envir=.GlobalEnv))     stop(paste0("Object ", match.call()$zlabels, " not found\n"))
    if(any(unlist(vapply(seq_along(ptrn), function(p) grepl(ptrn[p], nam.z, fixed=TRUE), logical(1L))),
           !identical(z.nam,   nam.z) && (any(grepl("[[:alpha:]]", gsub('c', '', nam.x))) || grepl(":",
           nam.x, fixed=TRUE))))      stop("Extremely inadvisable to supply 'zlabels' subsetted by any means other than row/column numbers or c() indexing: best to create new object")
    labs  <- as.integer(as.factor(zlabels))
    if(length(labs) != n.obs)         stop(paste0("'zlabels' must be a factor of length N=",  n.obs))
  }
  if(m.sw["P.sw"]) {
    if(!is.element(param, c("means",
       "loadings", "uniquenesses")))  stop("Can only plot parallel coordinates for means, loadings or uniquenesses")
  }
  if(!grp.ind)  {
    if(m.sw["Z.sw"])                  stop("Can't use 'Z' for 'plot.meth' as no clustering has taken place")
    if(param == "pis")                stop("Can't plot mixing proportions as no clustering has taken place")
  }
  if(all(m.sw["E.sw"],
         !attr(x, "Errors")))         stop("Can't plot error metrics as they were not calculated due to storage switches")
  if(all(!m.sw["G.sw"], !m.sw["Z.sw"], !m.sw["E.sw"],
     missing(param)))                 stop("What variable would you like to plot?")
  if(all(any(m.sw["M.sw"], all.ind),
     is.element(param, c("means", "uniquenesses")),
     !v.sw[param],
     is.element(method, c("FA", "IFA")))) {
    if(all.ind)                       warning(paste0("Can only plot posterior mean, as ", param, switch(param, alpha=, discount=" wasn't", " weren't"), " stored"), call.=FALSE)
    v.sw[param]    <- !v.sw[param]
    all.ind        <- FALSE
    m.sw["M.sw"]   <- TRUE
  }
  if(all(!v.sw[param], !m.sw["G.sw"],
     !m.sw["Z.sw"],   !m.sw["E.sw"])) stop(paste0("Nothing to plot: ", param, ifelse(is.element(param, c("alpha", "discount")), ifelse(any(all(param == "alpha", is.element(method, c("FA", "IFA"))),
                                           all(param == "discount", !is.element(method, c("IMFA", "IMIFA")))), paste0(" not used for the ", method, " method"), paste0(" was fixed at ", ifelse(param == "alpha", attr(x, "Alpha"), attr(x, "Discount")))), " weren't stored")))
  if(any(!is.logical(intervals),
         length(intervals) != 1))     stop("'intervals' must be TRUE or FALSE")
  if(any(!is.logical(mat),
         length(mat)       != 1))     stop("'mat' must be TRUE or FALSE")
  if(any(!is.logical(partial),
         length(partial)   != 1))     stop("'partial' must be TRUE or FALSE")
  if(any(!is.logical(titles),
         length(titles)    != 1))     stop("'titles' must be TRUE or FALSE")
  if(any(!is.logical(by.fac),
         length(by.fac)    != 1))     stop("'by.fac' must be TRUE or FALSE")
  indx    <- missing(ind)
  facx    <- missing(fac)
  gx      <- missing(g)
  if(!indx) {
    ind   <- as.integer(ind)
    xind  <- ind
  }
  if(!facx) {
    fac   <- as.integer(fac)
    flen  <- length(fac)
    if(flen  == 1 && gx)     fac <- rep(fac, G)
    flen  <- length(fac)
    if(flen  != G && all(gx,
       param == "loadings"))          stop(paste0("'fac' must be supplied for each of the ", G, " groups"))
  }
  g.score <- all(grp.ind, !all.ind, param == "scores")
  if(!gx)                      g <- as.integer(g)
  if(!gx  && any(length(g) != 1,
                 !is.numeric(g)))     stop("If 'g' is supplied it must be of length 1")
  if(any(all(is.element(method, c("IMFA", "OMFA")), m.sw["G.sw"]), m.sw["Z.sw"])) {
    if(m.sw["G.sw"]) {
      Gs  <- if(gx) seq_len(2L) else ifelse(g <= 2, g,
                                      stop("Invalid 'g' value"))
    } else if(m.sw["Z.sw"]) {
      Gs  <- if(gx) (if(z.sim) seq_len(3L) else seq_len(2L)) else ifelse(g <=
             ifelse(z.sim, 3, 2), g,  stop(paste0("Invalid 'g' value", ifelse(z.sim, ": similarity matrix not available", ""))))
    }
  } else if(all(is.element(method, c("IMIFA", "OMIFA")), m.sw["G.sw"]))           {
    if(m.sw["G.sw"]) {
      Gs  <- if(gx) seq_len(3L) else ifelse(g <= 3, g,
                                      stop("Invalid 'g' value"))
    } else if(m.sw["Z.sw"]) {
      Gs  <- if(gx) (if(z.sim) seq_len(3L) else seq_len(2L)) else ifelse(g <=
             ifelse(z.sim, 3, 2), g,  stop(paste0("Invalid 'g' value", ifelse(z.sim, ": similarity matrix not available", ""))))
    }
  } else if(any(all(is.element(param, c("scores", "pis", "alpha", "discount")), any(all.ind, param != "scores", !m.sw["M.sw"])),
            m.sw["G.sw"], all(m.sw["P.sw"], param != "loadings"), m.sw["E.sw"]))  {
    Gs    <- 1L
  } else if(!gx) {
    if(!is.element(method, c("FA", "IFA"))) {
      if(!is.element(g, Gseq))        stop("This g value was not used during simulation")
      Gs  <- g
    } else if(g > 1)     {            message(paste0("Forced g=1 for the ", method, " method"))
      Gs  <- 1L
    }
  } else if(!interactive())  {        stop("g must be supplied for non-interactive sessions")
  } else {
    Gs    <- Gseq
  }

  for(g in Gs) {
    Q     <- Qs[g]
    ng    <- ifelse(grp.ind, grp.size[g], n.obs)
    g.ind <- which(Gs == g)
    msgx  <- all(interactive(), g != max(Gs))
    .ent_exit  <- function() {
      ent      <- readline("Hit <Return> to see next plot or type 'EXIT'/hit <Esc> to exit: ")
      options(show.error.messages=FALSE)
      on.exit(suppressWarnings(options(defopt)), add=TRUE)
      if(ent  %in% c("exit", "EXIT")) stop()
    }
    if(any(all(Qs == 0, param == "scores"),
           all(Q  == 0, param == "loadings"),
           all(ng == 0, param == "scores", m.sw["M.sw"]))) {
                                      warning(paste0("Can't plot ", param, paste0(ifelse(any(all(param == "scores", ng == 0), all(param == "loadings", grp.ind)), paste0(" for group ", g), "")), " as they contain no ", ifelse(all(param == "scores", ng == 0), "rows/observations", "columns/factors")), call.=FALSE)
      if(g == max(Gs)) {
        break
      } else {
        if(isTRUE(msgx)) .ent_exit()
        next
      }
    }
    if(any(is.element(param, c("alpha", "discount")),
           all(is.element(param, c("means", "uniquenesses")), !indx),
           all(param == "loadings", Q == 1), all(param == "scores",
           Q.max == 1)))  { matx <- FALSE
    } else   {
      matx     <- mat
    }
    if(!matx) {
      iter     <- switch(param, scores=seq_len(attr(x$Score, "Eta.store")), loadings=seq_len(attr(x, "N.Loadstore")[g]), seq_along(store))
    }
    if(is.element(param, c("scores", "loadings"))) {
      if(indx)               ind <- c(1L, 1L)
      if(!facx)           ind[2] <- fac[g]
      if(all(mat,
         length(ind) == 1))  ind <- rep(ind, 2)
      if(length(ind) != 2)            stop(paste0("Length of plotting indices must be 2 for the ", param, "parameter when 'mat' is FALSE"))
      if(param == "scores") {
        if(ind[1] >  n.obs)           stop(paste0("First index can't be greater than the number of observations: ",  n.obs))
        if(ind[2] >  Q.max) {         warning(paste0("Second index can't be greater than ", Q.max, ", the total number of factors", if(grp.ind) paste0(" in the widest loadings matrix")), call.=FALSE)
        if(isTRUE(msgx)) .ent_exit()
        next
        }
      } else {
        if(ind[1] > n.var)            stop(paste0("First index can't be greater than the number of variables: ",  n.var))
        if(ind[2] > Q) {              warning(paste0("Second index can't be greater than ", Q, ", the number of factors", if(grp.ind) paste0(" in group ", g), ".\n Try specifying a vector of fac values with maximum entries ", paste0(Qs, collapse=", "), "."), call.=FALSE)
        if(isTRUE(msgx)) .ent_exit()
        next
        }
      }
    } else   {
      if(any(is.element(param, c("alpha", "discount")),
             indx))       ind    <- 1L
      if(length(ind) >  1)            stop("Length of plotting indices can't be greater than 1")
      if(param == "pis")    {
        if(ind       >  G)            stop(paste0("Index can't be greater than the number of groups: ", G))
      } else {
        if(ind       > n.var)         stop(paste0("Index can't be greater than the number of variables: ", n.var))
      }
    }

    if(m.sw["T.sw"]) {
      if(param == "means")  {
        plot.x <- x$Means$mus[[g]]
        if(matx) {
          if(mispal) palette(viridis(n.var, alpha=transparency))
          matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1), col=seq_along(palette()))
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else {
          plot(x=iter, y=plot.x[ind,], type="l", ylab="", xlab="Iteration", ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1))
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nMean - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
        }
      }
      if(param == "scores") {
        x.plot <- x$Scores$eta
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
          if(mispal) palette(viridis(min(10, max(2, n.obs)), alpha=transparency))
        } else {
          plot.x  <- if(Q.max > 1) x.plot[ind[1],,] else t(x.plot[ind[1],,])
          if(mispal) palette(viridis(min(10, max(2, Q.max)), alpha=transparency))
        }
        if(matx) {
          matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, col=seq_along(palette()))
          if(by.fac) {
            if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Factor ", ind[2])))
          } else {
            if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]])))
          }
        } else {
          plot(x=iter, y=x.plot[ind[1],ind[2],], type="l", ylab="", xlab="Iteration")
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]], ", Factor ", ind[2])))
        }
      }
      if(param == "loadings") {
        x.plot <- x$Loadings$lmats[[g]]
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
          if(mispal) palette(viridis(min(10, max(2, n.var)), alpha=transparency))
        } else {
          plot.x  <- x.plot[ind[1],,]
          if(mispal) palette(viridis(min(10, max(2, Q)), alpha=transparency))
        }
        if(matx) {
          matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, col=seq_along(palette()))
          if(by.fac) {
            if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), "Factor ", ind[2])))
          } else {
            if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable")))
          }
        } else   {
          plot(x=iter, y=x.plot[ind[1],ind[2],], type="l", ylab="", xlab="Iteration")
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable, Factor ", ind[2])))
        }
      }
      if(param == "uniquenesses") {
        plot.x <- x$Uniquenesses$psis[[g]]
        if(matx) {
          if(mispal) palette(viridis(n.var, alpha=transparency))
          matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, col=seq_along(palette()))
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          plot(x=iter, y=plot.x[ind,], ylab="", type="l", xlab="Iteration")
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, ":\n", paste0(":\nUniqueness - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
        }
      }
      if(param == "pis") {
        plot.x <- clust$pi.prop
        if(matx) {
          if(mispal) palette(viridis(G, alpha=transparency))
          matplot(t(plot.x), type="l", ylab="", xlab="Iteration", lty=1, col=seq_along(palette()))
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else   {
          plot(x=iter, y=plot.x[ind,], ylab="", type="l", xlab="Iteration")
          if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nMixing Proportion - Group ", ind)))))
        }
      }
      if(param == "alpha") {
        plot.x <- clust$DP.alpha
        plot(plot.x$alpha, ylab="", type="l", xlab="Iteration", main="")
        if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nAlpha")))))
        if(all(intervals, ci.sw[param])) {
          ci.x <- plot.x$ci.alpha
          abline(h=plot.x$post.alpha, col=2, lty=2)
          abline(h=ci.x[1], col=grey, lty=2)
          abline(h=ci.x[2], col=grey, lty=2)
        }
      }
      if(param == "discount") {
        plot.x <- clust$PY.disc
        plot(as.vector(plot.x$discount), ylab="", type="l", xlab="Iteration", main="", ylim=c(0, 1))
        if(titles) title(main=list(paste0("Trace", ifelse(all.ind, "", paste0(":\nDiscount")))))
        if(all(intervals, ci.sw[param])) {
          ci.x <- plot.x$ci.disc
          abline(h=plot.x$post.disc,  col=2, lty=2)
          abline(h=ci.x[1], col=grey, lty=2)
          abline(h=ci.x[2], col=grey, lty=2)
        }
      }
      if(!indx) {         ind[1] <- xind[1]
        if(all(facx, is.element(param, c("scores",
           "loadings")))) ind[2] <- xind[2]
      }
      if(all.ind)          xxind <- ind
    }

    if(m.sw["D.sw"]) {
      if(param == "means") {
        x.plot <- x$Means$mus[[g]]
        if(matx) {
          if(mispal) palette(viridis(n.var, alpha=transparency))
          plot.x  <- apply(x.plot, 1, density)
          fitx    <- sapply(plot.x, "[[", "x")
          fity    <- sapply(plot.x, "[[", "y")
          matplot(fitx, fity, type="l", xlab="", ylab="", lty=1, col=seq_along)
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          plot.d  <- density(x.plot[ind,])
          plot(plot.d, main="", ylab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nMeans - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
          polygon(plot.d, col=grey, border=NA)
        }
      }
      if(param == "scores") {
        x.plot <- x$Scores$eta
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
          if(mispal) palette(viridis(min(10, max(2, n.obs)), alpha=transparency))
        } else   {
          plot.x  <- if(Q > 1) x.plot[ind[1],,] else t(x.plot[ind[1],,])
          if(mispal) palette(viridis(min(10, max(2, Q.max)), alpha=transparency))
        }
        if(matx) {
          plot.x  <- apply(plot.x, 1, density)
          fitx    <- sapply(plot.x, "[[", "x")
          fity    <- sapply(plot.x, "[[", "y")
          matplot(fitx, fity, type="l", xlab="", ylab="", lty=1, col=seq_along(palette()))
          if(by.fac) {
            if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Factor ", ind[2])))
          } else {
            if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]])))
          }
        } else   {
          plot.d  <- density(x.plot[ind[1],ind[2],])
          plot(plot.d, main="", ylab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", ":\nScores - "), "Observation ", obs.names[ind[1]], ", Factor ", ind[2])))
          polygon(plot.d, col=grey, border=NA)
        }
      }
      if(param == "loadings") {
        x.plot <- x$Loadings$lmats[[g]]
        if(by.fac) {
          plot.x  <- x.plot[,ind[2],]
          if(mispal) palette(viridis(min(10, max(2, n.var)), alpha=transparency))
        } else {
          plot.x  <- x.plot[ind[1],,]
          if(mispal) palette(viridis(min(10, max(2, Q)), alpha=transparency))
        }
        if(matx) {
          plot.x  <- apply(plot.x, 1, density)
          fitx    <- sapply(plot.x, "[[", "x")
          fity    <- sapply(plot.x, "[[", "y")
          matplot(fitx, fity, type="l", xlab="", ylab="", lty=1, col=seq_along(palette()))
          if(by.fac) {
            if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), "Factor ", ind[2])))
          } else {
            if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable")))
          }
        } else   {
          plot.d  <- density(x.plot[ind[1],ind[2],])
          plot(plot.d, main="", ylab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nLoadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind[1]], " Variable, Factor ", ind[2])))
          polygon(plot.d, col=grey, border=NA)
        }
      }
      if(param == "uniquenesses") {
        x.plot <- x$Uniquenesses$psis[[g]]
        if(matx) {
          if(mispal) palette(viridis(n.var, alpha=transparency))
          plot.x  <- apply(x.plot, 1, density)
          h       <- sapply(plot.x, "[[", "bw")
          w       <- lapply(seq_along(plot.x), function(d) 1/pnorm(0, mean=x.plot[d,], sd=h[d], lower.tail=FALSE))
          plot.x  <- lapply(seq_along(plot.x), function(d) suppressWarnings(density(x.plot[d,], bw=h[d], kernel="gaussian", weights=w[[d]]/length(x.plot[d,]))))
          fitx    <- sapply(plot.x, "[[", "x")
          fity    <- sapply(plot.x, "[[", "y")
          matplot(fitx, fity, type="l", xlab="", ylab="", lty=1, col=seq_along(palette()))
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        } else   {
          plot.d  <- density(x.plot[ind,])
          h       <- plot.d$bw
          w       <- 1/pnorm(0, mean=x.plot[ind,], sd=h, lower.tail=FALSE)
          plot.d  <- suppressWarnings(density(x.plot[ind,], bw=h, kernel="gaussian", weights=w/length(x.plot[ind,])))
          plot.d$y[plot.d$x < 0] <- 0
          plot(plot.d, main="", ylab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, ":\n", paste0(":\nUniquenesses - ", ifelse(grp.ind, paste0("Group ", g, " - "), ""))), var.names[ind], " Variable")))
          polygon(plot.d, col=grey, border=NA)
        }
      }
      if(param == "pis") {
        x.plot <- clust$pi.prop
        if(matx) {
          if(mispal) palette(viridis(G, alpha=transparency))
          plot.x  <- lapply(seq_len(G), function(g, x=x.plot[g,]) .logitdensity(x[x > 0 & x < 1]))
          fitx    <- sapply(plot.x, "[[", "x")
          fity    <- sapply(plot.x, "[[", "y")
          matplot(fitx, fity, type="l", ylab="", lty=1, col=seq_along(palette()), xlab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else   {
          x.plot  <- x.plot[ind,]
          fit     <- .logitdensity(x.plot[x.plot > 0 & x.plot < 1])
          fitx    <- fit$x
          fity    <- fit$y
          plot(fitx, fity, type="l", main="", ylab="", xlab="")
          if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nMixing Proportions - Group ", ind)))))
          polygon(c(min(fitx), fitx), c(0, fity), col=grey, border=NA)
        }
      }
      if(param == "alpha") {
        plot.x <- clust$DP.alpha
        plot.d <- density(plot.x$alpha)
        h      <- plot.d$bw
        tr     <- ifelse(attr(x, "Pitman"), - max(clust$PY.disc$discount, 0), 0)
        w      <- 1/pnorm(tr, mean=plot.x$alpha, sd=h, lower.tail=FALSE)
        plot.d <- suppressWarnings(density(plot.x$alpha, bw=h, kernel="gaussian", weights=w/length(plot.x$alpha)))
        plot.d$y[plot.d$x < tr]  <- 0
        plot(plot.d, main="", ylab="")
        if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nAlpha")))))
        polygon(plot.d, col=grey, border=NA)
        if(intervals) {
          avg  <- plot.x$post.alpha
          clip(avg, avg, 0, plot.d$y[which.min(abs(plot.d$x - avg))])
          abline(v=avg, col=2, lty=2)
        }
      }
      if(param == "discount") {
        plot.x <- clust$PY.disc
        x.plot <- as.vector(plot.x$discount)
        fit    <- .logitdensity(x.plot[x.plot > 0])
        fitx   <- fit$x
        fity   <- fit$y * (1 - plot.x$post.kappa)
        plot(fitx, fity, type="l", main="", xlab="", ylab="", xlim=c(0, max(fitx)))
        usr    <- par("usr")
        if(plot.x$post.kappa > 0)  {
          clip(usr[1], usr[2], 0, usr[4])
          abline(v=0,  col=3,  lwd=2)
          clip(usr[1], usr[2], usr[3], usr[4])
        }
        if(titles) title(main=list(paste0("Density", ifelse(all.ind, "", paste0(":\nDiscount")))))
        polygon(c(min(fitx), fitx), c(0, fity), col=grey, border=NA)
        if(intervals) {
          avg  <- plot.x$post.disc
          clip(avg, avg, 0, fity[which.min(abs(fitx - avg))])
          abline(v=avg, col=2, lty=2)
          clip(usr[1], usr[2], usr[3], usr[4])
        }
      }
    }

    if(m.sw["M.sw"])  {
      if(is.element(param, c("scores", "loadings"))) {
        if(indx)  {
          ind     <- switch(param, scores=c(1L, min(Q.max, 2L)), c(1L, 1L))
        }
        if(!facx) {
          ind[2]  <- fac[g]
        }
        if(param  == "scores") {
          if(any(ind[1]  > Q.max,
                 ind[2]  > Q.max))    stop(paste0("Only the first ", Q.max, " columns can be plotted"))
        } else if(ind[2] > Q)         stop(paste0("Only the first ", Q, " columns can be plotted"))
      }
      if(param == "means") {
        plot.x <- x$Means$post.mu[,g]
        if(g   == min(Gs)) {
          pxx  <- range(sapply(x$Means$post.mu,    range))
          cixx <- if(all(intervals, ci.sw[param])) range(sapply(x$Means$ci.mu, range))
        }
        if(ci.sw[param])  ci.x   <- x$Means$ci.mu[[g]]
        plot(plot.x, type=type, ylab="", xlab="Variable", ylim=if(is.element(method, c("FA", "IFA"))) c(-1, 1) else if(all(intervals, ci.sw[param])) cixx else pxx)
        if(all(intervals, ci.sw[param])) plotCI(plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
        if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMeans", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        if(type  == "n") text(x=seq_along(plot.x), y=plot.x, var.names, cex=0.5)
      }
      if(param == "scores") {
        labs   <- if(grp.ind) clust$map else 1
        if(g.score)  {
          if(g.ind == 1)  tmplab <- labs
          z.ind  <- tmplab %in% g
          plot.x <- x$Scores$post.eta[z.ind,,drop=FALSE]
          ind2   <- ifelse(any(!facx, Q <= 1), ind[2], if(Q > 1) max(2, ind[2]))
          if(ci.sw[param]) ci.x  <- x$Scores$ci.eta[,z.ind,, drop=FALSE]
          labs   <- g
        } else       {
          plot.x <- x$Scores$post.eta
          ind2   <- ifelse(any(!facx, Q.max <= 1), ind[2], if(Q.max > 1) max(2, ind[2]))
          if(ci.sw[param]) ci.x  <- x$Scores$ci.eta
        }
        col.s  <- if(is.factor(labs)) as.integer(levels(labs))[labs] else labs
        type.s <- ifelse(any(type.x, type == "l"), "p", type)
        if(ind2 != 1)  {
          if(all(intervals, ci.sw[param])) {
            plotCI(plot.x[,ind[1]], plot.x[,ind2], li=ci.x[1,,ind2], ui=ci.x[2,,ind2], gap=TRUE, pch=NA, scol=grey, slty=3, xlab=paste0("Factor ", ind[1]), ylab=paste0("Factor ", ind2))
            plotCI(plot.x[,ind[1]], plot.x[,ind2], li=ci.x[1,,ind[1]], ui=ci.x[2,,ind[1]], add=TRUE, gap=TRUE, pch=NA, scol=grey, slty=3, err="x")
            if(type.s != "n") points(plot.x[,ind[1]], plot.x[,ind2], type=type.s, col=col.s, pch=20)
          } else {
            plot(plot.x[,ind[1]], plot.x[,ind2], type=type.s, col=col.s, pch=20,
                 xlab=paste0("Factor ", ind[1]), ylab=paste0("Factor ", ind2))
          }
          if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", ":\nScores"), ifelse(g.score, paste0(" - Group ", g), ""))))
          if(type.s == "n") text(plot.x[,ind[1]], plot.x[,ind2], obs.names, col=col.s, cex=0.5)
        } else   {
          if(all(intervals, ci.sw[param])) {
            plotCI(if(!g.score) seq_len(n.obs) else seq_len(grp.size[g]), plot.x[,ind[1]], li=ci.x[1,,ind[1]], ui=ci.x[2,,ind[1]], gap=TRUE, pch=NA, scol=grey, slty=3, xlab="Observation", ylab=paste0("Factor ", ind[1]))
            points(plot.x[,ind[1]], type=type.s, col=col.s, pch=20)
          } else {
            plot(plot.x[,ind[1]], type=type.s, col=col.s, xlab="Observation", ylab=paste0("Factor ", ind[1]), pch=20)
          }
          if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", ":\nScores"), ifelse(g.score, paste0(" - Group ", g), ""))))
          if(type.s == "n") text(plot.x[,ind[1]], col=col.s, cex=0.5)
        }
      }
      if(param == "loadings") {
        plot.x <- x$Loadings$post.load[[g]]
        if(g   == min(Gs)) {
          pxx  <- range(sapply(x$Loadings$post.load, range))
          cixx <- if(all(intervals, ci.sw[param], load.meth == "raw")) { if(by.fac) range(sapply(x$Loadings$ci.load, function(x) range(x[,,ind[2]]))) else range(sapply(x$Loadings$ci.load, function(x) range(x[,ind[1],]))) }
        }
        lcols  <- if(mispal) viridis(30, option="C") else palette
        if(load.meth == "heatmap") {
          if(titles) par(mar=c(4.1, 4.1, 4.1, 4.1))
          if(Q  > 1) {
            plotcolors(mat2cols(plot.x, cols=lcols))
          } else {
            graphics::image(z=t(plot.x)[,seq(n.var, 1), drop=FALSE], col=lcols, xlab="", ylab="", xaxt="n", yaxt="n")
          }
          if(titles) {
            title(main=list(paste0("Posterior Mean", ifelse(!all.ind, " Loadings ", " "), "Heatmap", ifelse(all(!all.ind, grp.ind), paste0(" - Group ", g), ""))))
            axis(1, line=-0.5, tick=FALSE, at=if(Q != 1) seq_len(Q) else 0, labels=seq_len(Q))
            if(n.var < 100) {
              axis(2, cex.axis=0.5, line=-0.5, tick=FALSE, las=1, at=if(Q > 1) seq_len(n.var) else seq(from=0, to=1, by=1/(n.var - 1)), labels=substring(var.names[n.var:1], 1, 10))
            }
            heat_legend(data=pxx, cols=lcols)
          }
          box(lwd=2)
          mtext(ifelse(Q > 1, "Factors", "Factor"), side=1, line=2)
          if(Q != 1) abline(v=seq(1, Q - 1, 1) + 0.5, lty=2, lwd=1)
        } else {
          if(ci.sw[param]) ci.x  <- x$Loadings$ci.load[[g]]
          if(!by.fac) {
           if(ci.sw[param]) ci.x <- as.matrix(ci.x[,ind[1],])
            plot(plot.x[ind[1],], type=type, xaxt="n", xlab="", ylab="Loading", ylim=if(all(intervals, ci.sw[param])) cixx else pxx)
            if(all(intervals, ci.sw[param])) plotCI(plot.x[ind[1],], li=ci.x[1,], ui=ci.x[2,], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
            axis(1, line=-0.5, tick=FALSE, at=seq_len(Q), labels=seq_len(Q))
            mtext("Factors", side=1, line=2)
            if(titles) title(main=list(paste0(ifelse(!all.ind, paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), "")), ""), var.names[ind[1]], " Variable")))
            if(type == "n") text(x=plot.x[ind[1],], paste0("Factor ", seq_len(Q)), cex=0.5)
          } else     {
           if(ci.sw[param]) ci.x <- as.matrix(ci.x[,,ind[2],])
            plot(plot.x[,ind[2]], type=type, xaxt="n", xlab="", ylab="Loading", ylim=if(all(intervals, ci.sw[param])) cixx else pxx)
            if(all(intervals, ci.sw[param])) plotCI(plot.x[,ind[2]], li=ci.x[1,], ui=ci.x[2,], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
            axis(1, line=-0.5, tick=FALSE, at=seq_len(n.var), labels=seq_len(n.var))
            mtext("Variable #", side=1, line=2, cex=0.8)
            if(titles) title(main=list(paste0(ifelse(!all.ind, paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, " - "), "")), ""), "Factor ", ind[2])))
            if(type == "n") text(x=plot.x, var.names, cex=0.5)
          }
        }
      }
      if(param == "uniquenesses") {
        plot.x <- x$Uniquenesses$post.psi[,g]
        if(g   == min(Gs)) {
          pxx  <- c(0, max(sapply(x$Uniquenesses$post.psi, max)))
          cixx <- if(all(intervals, ci.sw[param])) c(0, max(sapply(x$Uniquenesses$ci.psi, max)))
        }
        if(ci.sw[param])  ci.x   <- x$Uniquenesses$ci.psi[[g]]
        plot(plot.x, type=type, ylab="", xlab="Variable", ylim=if(all(intervals, ci.sw[param])) cixx else pxx)
        if(all(intervals, ci.sw[param])) plotCI(plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=grey, add=TRUE, gap=TRUE, pch=ifelse(type == "n", NA, 20))
        if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nUniquenesses", ifelse(grp.ind, paste0(" - Group ", g), ""))))))
        if(type  == "n") text(seq_along(plot.x), plot.x, var.names, cex=0.5)
      }
      if(param == "pis") {
        plot.x <- clust$post.pi
        if(ci.sw[param])  ci.x   <- clust$ci.pi
        if(matx) {
          if(all(intervals, ci.sw[param])) {
            plotCI(barplot(plot.x, ylab="", xlab="", col=grey, ylim=c(0, 1), cex.names=0.7),
                   plot.x, li=ci.x[,1], ui=ci.x[,2], slty=3, scol=2, add=TRUE, gap=TRUE, pch=20)
          } else {
            barplot(plot.x, ylab="", xlab="", ylim=c(0, 1), cex.names=0.7)
          }
          if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMixing Proportions")))))
        } else {
          if(all(intervals, ci.sw[param])) {
            plotCI(barplot(plot.x[ind], ylab="", xlab="", ylim=c(0, 1), cex.names=0.7),
                   plot.x[ind], li=ci.x[ind,1], ui=ci.x[ind,2], slty=3, scol=2, add=TRUE, gap=TRUE, pch=20)
          } else {
            barplot(plot.x[ind], ylab="", xlab="Variable", ylim=c(0, 1), cex.names=0.7)
          }
          if(titles) title(main=list(paste0("Posterior Mean", ifelse(all.ind, "", paste0(":\nMixing Proportions - Group ", ind)))))
        }
      }
      if(is.element(param, c("alpha", "discount"))) {
        plot(c(0, 1), c(0, 1), ann=FALSE, bty='n', type='n', xaxt='n', yaxt='n')
        if(titles) title(main=list(paste0("Summary Statistics", ifelse(all.ind, "", paste0(":\n", switch(param, alpha="Alpha", discount="Discount"))))))
        plot.x <- switch(param, alpha=clust$DP.alpha[-1],     discount=clust$PY.disc[-1])
        x.step <- switch(param, alpha=attr(x, "Alph.step"),   discount=attr(x, "Disc.step"))
        conf   <- attr(x, "Conf.Level")
        digits <- options()$digits
        MH     <- switch(param, alpha=plot.x$alpha.rate != 1, discount=plot.x$disc.rate != 1)
        a.adj  <- rep(0.5, 2)
        a.cex  <- par()$fin[2]/ifelse(MH, 4, 3)
        pen    <- ifelse(MH, 0,    0.15)
        y1     <- switch(param, alpha=0.8, discount=0.85)
        y2     <- switch(param, alpha=0.8, discount=0.85)
        y3     <- switch(param, alpha=0.6, discount=0.65)
        y4     <- switch(param, alpha=0.6, discount=0.65)
        y5     <- ifelse(MH, switch(param, alpha=0.45, discount=0.5),  0.475)
        y6     <- ifelse(MH, switch(param, alpha=0.4,  discount=0.45), 0.4)
        text(x=0.5, y=y1  - pen, cex=a.cex, col="black", adj=a.adj, expression(bold("Posterior Mean:\n")))
        text(x=0.5, y=y2  - pen, cex=a.cex, col="black", adj=a.adj, bquote(.(round(switch(param, alpha=plot.x$post.alpha, discount=plot.x$post.disc), digits))))
        text(x=0.5, y=y3  - pen, cex=a.cex, col="black", adj=a.adj, expression(bold("\nVariance:\n")))
        text(x=0.5, y=y4  - pen, cex=a.cex, col="black", adj=a.adj, bquote(.(round(switch(param, alpha=plot.x$var.alpha,  discount=plot.x$var.disc),  digits))))
        text(x=0.5, y=y5  - pen, cex=a.cex, col="black", adj=a.adj, bquote(bold(.(100 * conf)) * bold("% Confidence Interval:")))
        text(x=0.5, y=y6  - pen, cex=a.cex, col="black", adj=a.adj, bquote(paste("[", .(round(switch(param, alpha=plot.x$ci.alpha[1], discount=plot.x$ci.disc[1]), digits)), ", ", .(round(switch(param, alpha=plot.x$ci.alpha[2], discount=plot.x$ci.disc[2]), digits)), "]")))
        if(isTRUE(MH)) {
          rate <- switch(param,  alpha="Acceptance Rate:", discount="Mutation Rate:")
          y7   <- switch(param,  alpha=0.25, discount=0.325)
          y8   <- switch(param,  alpha=0.2,  discount=0.275)
          text(x=0.5, y=y7,      cex=a.cex, col="black", adj=a.adj, substitute(bold(rate)))
          text(x=0.5, y=y8,      cex=a.cex, col="black", adj=a.adj, bquote(paste(.(round(100 * switch(param, alpha=plot.x$alpha.rate, discount=plot.x$disc.rate), 2)), "%")))
        }
        if(param == "discount") {
          text(x=0.5, y=0.15,    cex=a.cex, col="black", adj=a.adj, bquote(bold(hat(kappa)) * bold(" - Posterior Proportion of Zeros:")))
          text(x=0.5, y=0.1,     cex=a.cex, col="black", adj=a.adj, bquote(.(round(plot.x$post.kappa, digits))))
        }
      }
      if(!indx) {         ind[1] <- xind[1]
        if(all(facx, is.element(param, c("scores",
           "loadings")))) ind[2] <- xind[2]
      }
      if(all.ind)          ind   <- xxind
    }

    if(m.sw["G.sw"]) {
      plotG.ind  <- is.element(method, c("IMIFA", "IMFA", "OMIFA", "OMFA"))
      plotQ.ind  <- any(is.element(method, c("IFA", "MIFA")), all(is.element(method, c("IMIFA", "OMIFA")), g == 2))
      plotT.ind  <- any(all(g == 2, is.element(method, c("IMFA", "OMFA"))), all(is.element(method, c("IMIFA", "OMIFA")), g == 3))
      aicm       <- round(GQ.res$Criteria$AICMs, 2)
      bicm       <- round(GQ.res$Criteria$BICMs, 2)
      log.iLLH   <- round(GQ.res$Criteria$LogIntegratedLikelihoods, 2)
      dic        <- round(GQ.res$Criteria$DICs, 2)
      if(is.element(method, c("FA", "MFA", "OMFA", "IMFA"))) {
        aic.mcmc <- round(GQ.res$Criteria$AIC.mcmcs, 2)
        bic.mcmc <- round(GQ.res$Criteria$BIC.mcmcs, 2)
      }
      if(all(plotG.ind, g == 1))  {
        layout(1)
        par(mar=c(5.1, 4.1, 4.1, 2.1))
        plot.G <- GQ.res$G.Counts
        G.name <- names(plot.G)
        rangeG <- as.numeric(G.name)
        rangeG <- seq(from=min(rangeG), to=max(rangeG), by=1)
        missG  <- setdiff(rangeG, G.name)
        missG  <- setNames(rep(NA, length(missG)), as.character(missG))
        plot.G <- c(plot.G, missG)
        plot.G <- plot.G[Order(as.numeric(names(plot.G)))]
        col.G  <- c(1, ceiling(length(palette)/2))[(rangeG == G) + 1]
        G.plot <- barplot(plot.G, ylab="Frequency", xaxt="n", col=col.G)
        if(titles) title(main=list("Posterior Distribution of G"))
        axis(1, at=G.plot, labels=names(plot.G), tick=FALSE)
        axis(1, at=med(G.plot), labels="G", tick=FALSE, line=1.5)
      }
      if(all(method == "IFA", plotQ.ind)) {
        layout(1)
        par(mar=c(5.1, 4.1, 4.1, 2.1))
        plot.Q <- GQ.res$Q.Counts
        Q.name <- names(plot.Q)
        rangeQ <- as.numeric(Q.name)
        rangeQ <- seq(from=min(rangeQ), to=max(rangeQ), by=1)
        missQ  <- setdiff(rangeQ, Q.name)
        missQ  <- setNames(rep(NA, length(missQ)), as.character(missQ))
        plot.Q <- c(plot.Q, missQ)
        plot.Q <- plot.Q[Order(as.numeric(names(plot.Q)))]
        col.Q  <- c(1, ceiling(length(palette)/2))[(rangeQ == Q) + 1]
        Q.plot <- barplot(plot.Q, ylab="Frequency", xaxt="n", col=col.Q)
        if(titles) title(main=list("Posterior Distribution of Q"))
        axis(1, at=Q.plot, labels=names(plot.Q), tick=FALSE)
        axis(1, at=med(Q.plot), labels="Q", tick=FALSE, line=1.5)
      }
      if(all(method != "IFA", plotQ.ind)) {
        layout(1)
        par(mar=c(5.1, 4.1, 4.1, 2.1))
        plot.Q <- GQ.res$Q.Counts
        plot.Q <- if(is.list(plot.Q)) plot.Q else list(plot.Q)
        Q.name <- lapply(plot.Q, names)
        rangeQ <- as.numeric(unique(unlist(Q.name, use.names=FALSE)))
        rangeQ <- seq(from=min(rangeQ), to=max(rangeQ), by=1)
        missQ  <- lapply(Gseq, function(g) setdiff(rangeQ, as.numeric(Q.name[[g]])))
        missQ  <- lapply(Gseq, function(g) setNames(rep(NA, length(missQ[[g]])), as.character(missQ[[g]])))
        plot.Q <- lapply(Gseq, function(g) c(plot.Q[[g]], missQ[[g]]))
        plot.Q <- do.call(rbind, lapply(Gseq, function(g) plot.Q[[g]][Order(as.numeric(names(plot.Q[[g]])))]))
        if(titles)  {
          layout(rbind(1, 2), heights=c(9, 1))
          par(mar=c(3.1, 4.1, 4.1, 2.1))
        }
        Q.plot <- barplot(plot.Q, beside=TRUE, ylab="Frequency", xaxt="n", col=Gseq, space=c(0, 2))
        if(titles) title(main=list(expression('Posterior Distribution of Q'["g"])))
        axis(1, at=Rfast::colMedians(Q.plot), labels=colnames(plot.Q), tick=FALSE)
        axis(1, at=med(Q.plot), labels="Q", tick=FALSE, line=1)
        if(titles)  {
          par(mar=c(0, 0, 0, 0))
          plot.new()
          tmp  <- if(G > 5) unlist(lapply(Gseq, function(g) c(Gseq[g], Gseq[g + ceiling(G/2)])))[Gseq] else Gseq
          ltxt <- paste0("Group ", tmp)
          lcol <- Gseq[tmp]
          legend("center", legend=ltxt, ncol=if(G > 5) ceiling(G/2) else G, bty="n", pch=15, col=lcol, cex=max(0.7, 1 - 0.03 * G))
        }
      }
      if(plotT.ind) {
        layout(1)
        par(mar=c(5.1, 4.1, 4.1, 2.1))
        col.G  <- c(ceiling(length(palette)/2), 1)
        x.plot <- GQ.res$Stored.G
        plot.x <- if(is.element(method, c("IMFA", "IMIFA"))) t(x.plot) else cbind(as.vector(x.plot), rep(attr(x, "range.G"), ncol(x.plot)))
        matplot(plot.x, type="l", col=col.G, ylab="G", xlab="Iteration", main="", lty=if(is.element(method, c("IMFA", "IMIFA"))) 1 else 1:2, ylim=c(1, max(plot.x)), las=1)
        if(titles) {
          title(main=list("Trace:     \n\n"))
          title(expression("Active" * phantom(" and Non-empty Groups")), col.main = 1)
          title(expression(phantom("Active ") * "and" * phantom(" Non-empty Groups")), col.main="black")
          title(expression(phantom("Active and ") * "Non-empty" * phantom(" Groups")), col.main = col.G[1])
          title(expression(phantom("Active and Non-empty ") * "Groups"), col.main="black")
          if(length(unique(plot.x[,1])) > 1) {
            G.ci    <- GQ.res$G.CI
            lines(x=c(0, nrow(plot.x)), y=rep(G, 2), col=length(palette),   lty=2, lwd=1)
            if(G.ci[1] != G) lines(x=c(0, nrow(plot.x)), y=rep(G.ci[1], 2), lty=2, lwd=0.5, col=grey)
            if(G.ci[2] != G) lines(x=c(0, nrow(plot.x)), y=rep(G.ci[2], 2), lty=2, lwd=0.5, col=grey)
          }
        }
      }
      if(!any(plotQ.ind,
              plotG.ind, plotT.ind))  message("Nothing to plot")
      gq.nam <- substring(names(GQ.res), 1, 1)
      if(is.element(method, c("IMIFA", "OMIFA")))      {
        if(g == 1)   {
          print(GQ.res[gq.nam == "G"])
        } else if(g == 2) {
          print(GQ.res[gq.nam == "Q"])
        }
      } else if(is.element(method, c("OMFA", "IMFA"))) {
          if(g == 1) {
            print(GQ.res[gq.nam == "G"])
          } else {
            print(GQ.res[gq.nam != "G" & gq.nam != "S"])
          }
      } else switch(method, MFA=, MIFA={
          print(GQ.res[gq.nam != "S"])
        }, IFA=  {
          print(tail(GQ.res[gq.nam != "S"], -1))
        },
          cat(paste0("Q = ", Q, "\n"))
      )
      if(all(g == max(Gs), any(dim(bicm) > 1))) {
        G.ind  <- ifelse(any(G.supp, !is.element(method, c("MFA", "MIFA"))), 1, which(n.grp == G))
        Q.ind  <- ifelse(any(Q.supp, !is.element(method, c("FA", "MFA"))),   1, which(n.fac == Q))
        if(!is.element(method, c("IFA", "MIFA"))) {
          cat(paste0("AIC.mcmc = ", aic.mcmc[G.ind,Q.ind], "\n"))
          cat(paste0("BIC.mcmc = ", bic.mcmc[G.ind,Q.ind], "\n"))
        }
          cat(paste0("AICM = ", aicm[G.ind,Q.ind], "\n"))
          cat(paste0("BICM = ", bicm[G.ind,Q.ind], "\n"))
          cat(paste0("Log Integrated Likelihood = ", log.iLLH[G.ind,Q.ind], "\n"))
          cat(paste0("DIC = ", dic[G.ind,Q.ind], "\n"))
      }
      if(all(plotQ.ind,
             attr(GQ.res, "Q.big")))  warning("Q had to be prevented from exceeding its initial value.\n Consider re-running the model with a higher value for 'range.Q'", call.=FALSE)
    }

    if(m.sw["Z.sw"]) {
      if(type == "l")                 stop("'type' cannot be 'l' for clustering uncertainty plots")
      plot.x <- as.vector(clust$uncertainty)
      if(g == 1) {
        col.x  <- c(1, ceiling(length(palette)/2))[(plot.x >= 1/G) + 1]
        if(type != "n") col.x[plot.x == 0] <- NA
        if(titles) {
          layout(rbind(1, 2), heights=c(1, 6))
          par(mar=c(0, 4.1, 0.5, 2.1))
          plot.new()
          legend("center", legend=bquote(1/G == 1/.(G)), title="", lty=2, col=2, bty="n", y.intersp=par()$fin[2] * 7/5)
          legend("center", legend=c(" "," "), title=expression(bold("Clustering Uncertainty")), bty='n', y.intersp=par()$fin[2] * 2/5, cex=par()$cex.main)
          par(mar=c(5.1, 4.1, 0.5, 2.1))
        }
        plot(plot.x, type=type, ylim=c(0, 1 - 1/G), col=col.x, axes=FALSE, ylab="Uncertainty", xlab="Observation", pch=ifelse(type == "n", NA, 16))
        if(G == 2) {
          abline(h=0.5, col=par()$bg)
          abline(v=0,   col=par()$bg)
        }
        lines(x=c(0, n.obs), y=c(1/G, 1/G), lty=2, col=2)
        axis(1, las=1, pos=0, cex.axis=0.9, lty=0)
        axis(2, at=c(seq(from=0, to=min(1 - 1/G - 1/1000, 0.8), by=0.1), 1 - 1/G), labels=c(seq(from=0, to=min(1 - 1/G - 1/1000, 0.8), by=0.1), "1 - 1/G"), las=2, pos=0, cex.axis=0.9)
        if(type == "n")  {
          znam  <- obs.names
          znam[plot.x == 0] <- ""
          text(x=seq_along(plot.x), y=plot.x, znam, col=col.x, cex=0.5)
        }
      } else if(g == 2)  {
        if(titles) {
          layout(rbind(1, 2), heights=c(1, 6))
          par(mar=c(0, 4.1, 0.5, 2.1))
          plot.new()
          legend("center", legend=bquote({NA >= 1/G} == 1/.(G)), title="", pch=15, col=3, bty="n", y.intersp=par()$fin[2] * 7/5)
          legend("center", legend=c(" "," "), title=expression(bold("Clustering Uncertainty")), bty='n', y.intersp=par()$fin[2] * 2/5, cex=par()$cex.main)
          par(mar=c(5.1, 4.1, 0.5, 2.1))
        }
        x.plot  <- hist(plot.x, plot=FALSE)
        breaks  <- if(sum(plot.x   != 0)) x.plot$breaks else seq(from=0, to=max(plot.x, 1/G), by=1/G)
        cols    <- 2     + (breaks >= 1/G)
        cols[cols == 2] <- grey
        plot(x.plot, main="", xlab="Uncertainties", xlim=c(0, 1 - 1/G), col=cols, xaxt="n", ylim=c(0, max(x.plot$counts)), yaxt="n")
        axis(1, at=c(breaks[round(breaks, 1) < min(0.8, 1 - 1/G)], 1 - 1/G), labels=(c(round(breaks[round(breaks, 1) < min(0.8, 1 - 1/G)], 3), "1 - 1/G")), las=2, pos=0, cex.axis=0.8)
        axis(2, at=if(sum(plot.x)  == 0) c(axTicks(2), max(x.plot$counts)) else axTicks(2), las=1, cex.axis=0.8)
      }
      if(all(g  == 3, z.sim)) {
        plot.x  <- as.matrix(clust$Z.avgsim$z.sim)
        perm    <- order(clust$map)
        p.ind   <- !identical(perm, clust$map)
        plot.x  <- if(p.ind) plot.x[perm,perm] else plot.x
        plot.x  <- t(plot.x[,seq(from=ncol(plot.x), to=1, by=-1)])
        par(defpar)
        if(titles) par(mar=c(4.1, 4.1, 4.1, 4.1))
        z.col   <- if(!any(mispal, gx)) palette else heat.colors(12)[12:1]
        plotcolors(mat2cols(replace(plot.x, plot.x == 0, NA), cols=z.col))
        if(titles) {
          title(main=list("Average Similarity Matrix"))
          axis(1, at=n.obs/2, labels=paste0("Observation 1:N", if(p.ind) " (Reordered)"), tick=FALSE)
          axis(2, at=n.obs/2, labels=paste0("Observation 1:N", if(p.ind) " (Reordered)"), tick=FALSE)
          heat_legend(data=plot.x, cols = z.col)
        }
        box(lwd=2)
        if(p.ind)                     message("Rows and columns of similarity matrix reordered to correspond to MAP clustering")
      }
      if(all(g  != 3, g == min(Gs))) {
        prf     <- NULL
        if(any(!labelmiss,  !z.miss)) {
          if(all(!labelmiss, z.miss)) {
           prf  <- clust$perf
          } else   {
           pzs  <- factor(clust$map, levels=seq_len(G))
           if(nlevels(pzs) == length(unique(labs))) {
            pzs <- factor(.lab_switch(z.new=as.numeric(levels(pzs))[pzs], z.old=labs)$z)
           }
           tab  <- table(pzs, labs, dnn=list("Predicted", "Observed"))
           prf  <- c(classAgreement(tab), classError(pzs, labs))
           if(nrow(tab) != ncol(tab))   {
            prf <- prf[-seq_len(2)]
            names(prf)[4]        <- "error.rate"
           } else {
            names(prf)[6]        <- "error.rate"
           }
           if(prf$error.rate     == 0)  {
            prf$misclassified    <- NULL
           }
           prf  <- c(list(confusion.matrix = tab), prf)
           if(length(unique(pzs)) == length(unique(labs))) {
            names(prf)[1]  <- "matched.confusion.matrix"
           }
          }
          prf$error.rate   <- paste0(round(100 * prf$error.rate, 2), "%")
        } else {
          prf     <- list(uncertain = attr(clust$uncertainty, "Obs"))
        }
        if(!is.null(prf))     {
          class(prf)       <- "listof"
          print(prf)
        }
      }
    }

    if(m.sw["P.sw"]) {
      plot.x <- switch(param, means=x$Means$post.mu, uniquenesses=x$Uniquenesses$post.psi, x$Loadings$post.load[[g]])
      x.plot <- apply(plot.x, 1L, range, na.rm=TRUE)
      plot.x <- if(all(param == "uniquenesses", uni.type == "isotropic")) plot.x else apply(plot.x, 2L, function(x) (x - min(x, na.rm=TRUE))/(max(x, na.rm=TRUE) - min(x, na.rm=TRUE)))
      varnam <- paste0(toupper(substr(param, 1, 1)), substr(param, 2, nchar(param)))
      if(any(grp.ind, param == "loadings")) {
        if(mispal) palette(viridis(max(Q, 2), alpha=transparency))
        layout(rbind(1, 2), heights=c(9, 1))
        par(mar=c(3.1, 4.1, 4.1, 2.1))
      }
      jitcol <- switch(param, loadings=Q, G)
      matplot(seq_len(n.var) + matrix(rnorm(jitcol * n.var, 0, min(0.1, 1/n.var^2)), nrow=n.var, ncol=jitcol), plot.x, type=switch(param, uniquenesses=switch(uni.type, unconstrained="p", isotropic="l"), "p"),
                        col=switch(param, loadings=seq_len(Q), seq_len(G)), pch=15, xlab="Variable", ylab=paste0("Standardised ", varnam), axes=FALSE, main=paste0("Parallel Coordinates: ", varnam, ifelse(all(grp.ind, param == "loadings"), paste0("\n Group ", g), "")), lty=1)
      axis(1, at=seq_len(n.var), labels=if(titles && n.var < 100) rownames(plot.x) else rep("", n.var), cex.axis=0.5, tick=FALSE)
      for(i in seq_len(n.var))    {
        lines(c(i, i), c(0, 1), col=grey)
        if(titles && n.var < 100) {
          text(c(i, i), c(0, 1), labels=format(x.plot[,i], digits=3), xpd=NA, offset=0.3, pos=c(1, 3), cex=0.5)
        }
      }
      if(any(grp.ind, param  == "loadings")) {
        par(mar=c(0, 0, 0, 0))
        plot.new()
        Xp   <- switch(param, loadings=Q, G)
        Xseq <- seq_len(Xp)
        tmp  <- if(Xp > 5) unlist(lapply(Xseq, function(x) c(Xseq[x], Xseq[x + ceiling(Xp/2)])))[Xseq] else Xseq
        ltxt <- paste0(switch(param, loadings="Factor", "Group"), tmp)
        lcol <- Xseq[tmp]
        legend("center", pch=15, col=lcol, legend=ltxt, ncol=if(Xp > 5) ceiling(Xp/2) else Xp, bty="n", cex=max(0.7, 1 - 0.03 * Xp))
      }
    }

    if(m.sw["E.sw"]) {
      x.plot <- x$Error[seq_len(which(names(x$Error) == "Averages"))]
      plot.x      <- if(G > 1) cbind(do.call(rbind, x.plot[-length(x.plot)]), Averages = x.plot$Averages) else x.plot
      if(titles) {
        layout(rbind(1, 2), heights=c(9, 1))
        par(mar=c(3.1, 4.1, 4.1, 2.1))
      }
      if(mispal) palette(viridis(nrow(plot.x), option="C"))
      col.e  <- if(G > 1) seq_len(nrow(plot.x)) else seq_along(plot.x)
      if(G    > 1)   {
        dens <- matrix(-1, nrow=nrow(plot.x), ncol=G + 1)
        dens[,G + 1]       <- 30
      } else  {
        dens <- NULL
      }
      pl.x   <- barplot(plot.x, beside=TRUE, col=col.e, main="", ylab="Deviation", density=dens)
      na.x   <- G > 1 & is.na(x.plot[[1]])
      if(G > 1) points(x=Rfast::colMedians(pl.x[,which(na.x), drop=FALSE]), y=rep(0, sum(na.x)), pch=8, col="red", cex=1.5)
      if(titles) title(main=list("Error Metrics"))
      if(titles) {
        par(mar=c(0, 0, 0, 0))
        plot.new()
        ltxt <- c("MSE", "MAE", "MEDSE", "MEDAE", "RMSE", "NRMSE")
        lnc  <- length(col.e)
        lcol <- col.e
        xna  <- sum(na.x)   > 0
        lpch <- rep(15, nrow(plot.x))
        temp <- legend("center", legend=if(xna) c(ltxt, "Missing") else ltxt, ncol=ifelse(xna, lnc + 1, lnc), bty="n",
                       pch=if(xna) c(lpch, 8) else lpch, col=if(xna) c(lcol, "red") else lcol, cex=0.8)
      }
      if(G > 1) {
        avg  <- setNames(list(x.plot$Averages), "Average Error Metrics")
        class(avg)         <- "listof"
      } else {
        avg  <- x.plot
      }
        print(avg)
    }

    if(m.sw["C.sw"]) {
      palette(tmp.pal)
      if(!all.ind)   {
       partial <- FALSE
       par(mai=c(1.25, 1, 0.75, 0.5), mfrow=c(1, 2), oma=c(0, 0, 2, 0))
      }
      if(param == "means")    {
        plot.x <- x$Means$mus[[g]]
        if(!partial) {
          acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind], " Variable"), ""))))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind], " Variable"), ""))))
          if(all(!all.ind, titles)) title(main=list(paste0("Means - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind], " Variable")), outer=TRUE)
        }
      }
      if(param == "scores")   {
        plot.x <- x$Scores$eta
        if(!partial) {
          acf(plot.x[ind[1],ind[2],], main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", "Observation ", obs.names[ind[1]], ", Factor ", ind[2]), ""))))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x[ind[1],ind[2],], main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", "Observation ", obs.names[ind[1]], ", Factor ", ind[2]), ""))))
          if(all(!all.ind, titles)) title(main=list(paste0("Scores - ", "Observation ", obs.names[ind[1]], ", Factor ", ind[2])), outer=TRUE)
        }
      }
      if(param == "loadings") {
        plot.x <- x$Loadings$lmats[[g]]
        if(!partial) {
          acf(plot.x[ind[1],ind[2],], main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind[1]], " Variable, Factor ", ind[2]), ""))))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x[ind[1],ind[2],], main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind[1]], " Variable, Factor ", ind[2]), ""))))
          if(all(!all.ind, titles)) title(main=list(paste0("Loadings - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind[1]], " Variable, Factor ", ind[2])), outer=TRUE)
        }
      }
      if(param == "uniquenesses")  {
        plot.x <- x$Uniquenesses$psis[[g]]
        if(!partial) {
          acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF", ifelse(all.ind, paste0(":\n", var.names[ind], " Variable"), ""))))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF", ifelse(partial, paste0(":\n", var.names[ind], " Variable"), ""))))
          if(all(!all.ind, titles)) title(main=list(paste0("Uniquenesses - ", ifelse(grp.ind, paste0("Group ", g, ":\n "), ""), var.names[ind], " Variable")), outer=TRUE)
        }
      }
      if(param == "pis")  {
        plot.x <- clust$pi.prop
        if(!partial) {
          acf(plot.x[ind,], main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF", ifelse(all(all.ind, matx), paste0(" - Group ", ind), ""))))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x[ind,], main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF", ifelse(all(all.ind, matx), paste0(" - Group ", ind), ""))))
          if(all(!all.ind, titles)) title(main=list(paste0("Mixing Proportions - Group ", ind)), outer=TRUE)
        }
      }
      if(is.element(param, c("alpha", "discount"))) {
        plot.x <- switch(param, alpha=clust$DP.alpha$alpha,   discount=as.vector(clust$PY.disc$discount))
        if(any(switch(param, alpha=clust$DP.alpha$alpha.rate, discount=clust$PY.disc$disc.rate) == 0,
           length(unique(round(plot.x, nchar(plot.x)))) == 1)) {
                                      warning(paste0(switch(param, alpha="Acceptance", discount="Mutation"), " rate too low: can't plot ", ifelse(all.ind, ifelse(partial, "partial-", "auto-"), ""), "correlation function", ifelse(all.ind, "", "s")), call.=FALSE)
          next
        }
        if(!partial) {
          acf(plot.x, main="", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("ACF")))
        }
        if(any(!all.ind, partial)) {
          acf(plot.x, main="", type="partial", ci.col=4, ylab="")
          if(titles) title(main=list(paste0("PACF")))
          if(all(!all.ind, titles)) title(main=list(paste0(switch(param, alpha="Alpha", discount="Discount"))), outer=TRUE)
        }
      }
    }
    if(all(all.ind, titles)) title(ifelse(param != "pis", paste0(toupper(substr(param, 1, 1)), substr(param, 2, nchar(param)),
                             ifelse(all(grp.ind, !is.element(param, c("scores", "pis", "alpha", "discount"))), paste0(" - Group ", g), "")),
                             paste0("Mixing Proportions", ifelse(matx, "", paste0(" - Group ", ind)))), outer=TRUE)
    if(isTRUE(msgx)) .ent_exit()
  }
}

# Loadings Heatmaps
#' Convert a numeric matrix to colours
#'
#' Converts a matrix to a hex colour code representation for plotting using \code{\link[gclus]{plotcolors}}. Used internally by \code{\link{plot.Results_IMIFA}} for plotting posterior mean loadings heatmaps.
#' @param mat A matrix.
#' @param cols The colour palette to be used. The default palette uses \code{\link[viridis]{viridis}}. Will be checked for validity.
#' @param byrank Logical indicating whether to convert the matrix itself or the sample ranks of the values therein. Defaults to \code{FALSE}.
#' @param breaks Number of gradations in colour to use. Defaults to \code{length(cols)}.
#' @param na.color Colour to be used to represent missing data.
#'
#' @return A matrix of hex colour code representations.
#' @export
#' @importFrom gclus "plotcolors"
#' @importFrom viridis "viridis"
#'
#' @seealso \code{\link[gclus]{plotcolors}}
#'
#' @examples
#' mat      <- matrix(rnorm(100), nrow=10, ncol=10)
#' mat[2,3] <- NA
#' cols     <- heat.colors(12)[12:1]
#' matcol   <- mat2cols(mat, cols=cols)
#' matcol
#'
#' # Use plotcolors() to visualise the colours matrix
#' par(mar=c(5.1, 4.1, 4.1, 4.1))
#' gclus::plotcolors(matcol)
#'
#' # Add a legend
#' heat_legend(mat, cols=cols); box(lwd=2)
  mat2cols     <- function(mat, cols = NULL, byrank = FALSE, breaks = length(cols), na.color = "#808080FF") {
    m          <- as.matrix(mat)
    if(missing(cols)) cols <- viridis(30L, option="C")
    if(!all(.are_cols(cols)))         stop("Invalid colours supplied")
    if(any(!is.logical(byrank),
           length(byrank)  != 1))     stop("'byrank' must be TRUE or FALSE")
    if(any(!is.numeric(breaks),
           length(breaks)  != 1))     stop("'breaks' must be a single digit")
    m1         <- if(isTRUE(byrank))  rank(m) else m
    facs       <- cut(m1, breaks, include.lowest=TRUE)
    answer     <- matrix(cols[as.numeric(facs)], nrow=nrow(m), ncol=ncol(m))
    NM         <- is.na(m)
    if(any(NM)) {
      if(length(na.color   != 1)  &&
         !.are_cols(na.color))        stop("'na.color' must be a valid colour in the presence of missing data")
      answer   <- replace(answer, NM, na.color)
    }
    rownames(answer)       <- rownames(m)
    colnames(answer)       <- colnames(m)
      answer
  }

# Colour Checker
  .are_cols    <- function(cols) {
    vapply(cols,  function(x) { tryCatch(is.matrix(col2rgb(x)), error = function(e) FALSE) }, logical(1L))
  }

# Heatmap Legends
#' Add a colour key legend to heatmap plots
#'
#' Using only base graphics, this function appends a colour key legend for heatmaps produced by, for instance, \code{\link[graphics]{image}} or \code{\link[gclus]{plotcolors}}.
#' @param data Either the data with which the heatmap was created or a vector containing its minimum and maximum values. Missing values are ignored.
#' @param cols The palette used when the heatmap was created.
#'
#' @return Modifies an existing plot by adding a legend.
#' @export
#'
#' @seealso \code{\link[graphics]{image}}, \code{\link[gclus]{plotcolors}}, \code{\link{mat2cols}}
#' @examples
#' # Generate a matrix, flip it, and plot it with a legend
#' data <- matrix(rnorm(50), nrow=10, ncol=5)
#' cols <- heat.colors(12)[12:1]
#' par(mar=c(5.1, 4.1, 4.1, 4.1))
#'
#' image(t(data)[,nrow(data):1], col=cols)
#' heat_legend(data, cols); box(lwd=2)
  heat_legend  <- function(data, cols) {
    bx         <- par("usr")
    xpd        <- par()$xpd
    box.cx     <- c(bx[2] + (bx[2]  - bx[1])/1000, bx[2] + (bx[2] - bx[1])/1000 + (bx[2] - bx[1])/50)
    box.cy     <- c(bx[3],   bx[3])
    box.sy     <- (bx[4]  -  bx[3]) / length(cols)
    xx         <- rep(box.cx, each  = 2)
    par(xpd = TRUE)
    for(i in seq_along(cols)) {
      yy   <- c(box.cy[1] + (box.sy * (i - 1)),
                box.cy[1] + (box.sy * (i)),
                box.cy[1] + (box.sy * (i)),
                box.cy[1] + (box.sy * (i - 1)))
      polygon(xx, yy, col =  cols[i], border = cols[i])
    }
    par(new = TRUE)
    plot(0, 0, type = "n",  ylim = range(complete.cases(data)), yaxt = "n", ylab = "", xaxt = "n", xlab = "", frame.plot = FALSE)
    axis(side = 4, las = 2, tick = FALSE, line = 0.1, cex.axis = 1)
    suppressWarnings(par(xpd = xpd))
  }

# Prior No. Groups (DP & PY)
#' Plot Dirichlet / Pitman-Yor process Priors
#'
#' Plots the prior distribution of the number of clusters under a Dirichlet / Pitman-Yor process prior, for a sample of size \code{N} at given values of the concentration parameter \code{alpha} and optionally also the \code{discount} parameter. Useful for soliciting sensible priors for \code{alpha} or suitable fixed values for \code{alpha} or \code{discount} under the "\code{IMFA}" and "\code{IMIFA}" methods for \code{\link{mcmc_IMIFA}}. All arguments are vectorised. Requires use of the \code{Rmpfr} and \code{gmp} libraries. May encounter difficulty and slowness for large \code{N}, especially with non-zero \code{discount}. Users can also consult \code{\link{G_expected}} and \code{\link{G_variance}} in order to solicit sensible priors.
#' @param N The sample size.
#' @param alpha The concentration parameter. Must be specified and must be strictly greater than \code{-discount}.
#' @param discount The discount parameter for the Pitman-Yor process. Must lie in the interval [0, 1). Defaults to 0 (i.e. the Dirichlet process).
#' @param show.plot Logical indicating whether the plot should be displayed (default = \code{TRUE}).
#'
#' @return A plot of the prior distribution if \code{show.plot} is \code{TRUE}. Density values are returned invisibly. Note that the density values may not strictly sum to one in certain cases, as values small enough to be represented as zero may well be returned.
#' @export
#' @importFrom viridis "viridis"
#' @seealso \code{\link{G_expected}}, \code{\link{G_variance}}, \code{\link[Rmpfr]{Rmpfr}}
#'
#' @examples
#' # require("Rmpfr")
#'
#' # Plot Dirichlet process priors for different values of alpha
#' # DP <- G_priorDensity(N=50, alpha=c(3, 10, 25))
#' # DP
#'
#' # Verify that these alpha/discount values produce Pitman-Yor process priors with the same mean
#' # G_expected(N=50, alpha=c(19.23356, 6.47006, 1), discount=c(0, 0.47002, 0.7300045))
#'
#' # Now plot them to examine tail behaviour as discount increases
#' # PY <- G_priorDensity(N=50, alpha=c(19.23356, 6.47006, 1), discount=c(0, 0.47002, 0.7300045))
#' # PY
  G_priorDensity      <- function(N, alpha, discount = 0L, show.plot = TRUE) {
    firstex    <- suppressMessages(requireNamespace("Rmpfr", quietly=TRUE))
    if(isTRUE(firstex)) {
      on.exit(.detach_pkg("Rmpfr"))
      on.exit(.detach_pkg("gmp"), add=TRUE)
    } else                            stop("'Rmpfr' package not installed")
    on.exit(palette("default"),   add=!isTRUE(firstex))
    defopt     <- options()
    options(expressions = 500000)
    on.exit(options(defopt),      add=TRUE)
    if(any(c(length(N),
             length(show.plot)) > 1)) stop("Arguments 'N' and 'show.plot' must be strictly of length 1")
    if(!is.logical(show.plot))        stop("'show.plot' must be TRUE or FALSE")
    max.len    <- max(length(alpha),  length(discount))
    if(max.len  > 10)                 stop("Can't plot more than ten distributions simultaneously")
    if(!is.element(length(alpha),
       c(1, max.len)))                stop("'alpha' must be of length 1 or length(discount)")
    if(!is.element(length(discount),
       c(1, max.len)))                stop("'discount' must be of length 1 or length(alpha)")
    if(!all(is.numeric(discount), is.numeric(alpha),
            is.numeric(N)))           stop("'N', 'alpha', and 'discount' inputs must be numeric")
    if(any(discount < 0,
       discount >= 1))                stop("'discount' must lie in the interval [0,1)")
    if(any(alpha <= - discount))      stop("'alpha' must be strictly greater than -discount")
    if(length(alpha)    != max.len) {
      alpha    <- rep(alpha,    max.len)
    }
    if(length(discount) != max.len) {
      discount <- rep(discount, max.len)
    }
    rx         <- matrix(0, nrow=N, ncol=max.len)
    Nseq       <- seq_len(N)
    Nsq2       <- Rmpfr::mpfr(Nseq,        precBits=256)
    for(i in seq_len(max.len)) {
      alphi    <- Rmpfr::mpfr(alpha[i],    precBits=256)
      disci    <- Rmpfr::mpfr(discount[i], precBits=256)
      if(disci == 0) {
        vnk    <- exp(Nsq2 * log(alphi)     - log(Rmpfr::pochMpfr(alphi, N)))
        rx[,i] <- gmp::asNumeric(abs(vnk    * Rmpfr::.bigz2mpfr(gmp::Stirling1.all(N))))
      } else  {
        vnk    <- c(Rmpfr::mpfr(0,         precBits=256),  cumsum(log(alphi + Nseq[-N] * disci)))   -
                  log(Rmpfr::pochMpfr(alphi + 1, N - 1)) - Nsq2 * log(disci)
        lnkd   <- lapply(Nseq, function(g) Rmpfr::sumBinomMpfr(g, f=function(k)  Rmpfr::pochMpfr(-k * disci, N)))
        rx[,i] <- gmp::asNumeric(exp(vnk    - lfactorial(Nsq2)) * abs(methods::new("mpfr", unlist(lnkd))))
      }
    }
    if(isTRUE(show.plot))   {
      col      <- seq(from=2, to=max.len + 1)
      palette(adjustcolor(rep(col, 2), alpha.f=0.5))
      matplot(x=seq_len(N), y=rx, type="h", col=col, xlab="Clusters", ylim=c(0, max(rx)), ylab="Density",
              main=paste0("Prior Distribution of G\nN=", N), lwd=seq(3, 1, length.out=max.len), lty=seq_len(2))
    }
      invisible(if(max.len == 1) as.vector(rx) else rx)
  }
#
