#' Adds a figure caption number
#'
#' The function relies on \code{options("fig_caption_no")}
#' in order to keep track of the last number. If
#' you want to force the caption function to skip
#' captions while still using it in the knitr fig.cap option
#' then simply set \code{options(fig_caption_no = FALSE)}
#'
#' @param str The string that is to be prepended with string
#' @param roman Whether or not to use roman numbers instead
#'  of arabic. Can also be set through \code{options(fig_caption_no_roman = TRUE)}
#' @param sprintf_str An \code{\link[base]{sprintf}} formatted
#'  string where the first argument is reserved for the string
#'  generated by the counter and the second one is for the caption
#'  text. Can also be set through \code{options(fig_caption_no_sprintf = TRUE)}
#'
#' @export
#' @examples
#' \dontrun{
#' ```{r, fig.cap=figCapNo("My nice plot")}
#' plot(1:10 + rnorm(10), 1:10)
#' ```
#' }
#' org_opts <- options(fig_caption_no = 2,
#'                     fig_caption_no_sprintf = "Figure %s: %s")
#' figCapNo("A plot with caption number = 3")
#'
#' org_opts <- options(fig_caption_no = TRUE)
#' figCapNo("A plot with caption number =  1")
#'
#' # Use default setting
#' options(fig_caption_no_sprintf = NULL)
#' figCapNo("A plot with caption number =  2")
#'
#' # Return the original settings
#' options(org_opts)
#' @family figure caption functions
figCapNo <- function(str,
                     roman = getOption("fig_caption_no_roman",
                                       FALSE),
                     sprintf_str = getOption("fig_caption_no_sprintf",
                                               "Fig. %s: %s")){

  last_no <- getOption("fig_caption_no")
  if (is.logical(last_no)){
    if (last_no == FALSE)
      return(str)

    last_no <- 0
  }else if (is.null(last_no)){
    last_no <- 0
  }

  current_no <- last_no + 1
  options(fig_caption_no = current_no)

  if (roman)
    current_no <- as.character(as.roman(current_no))

  return(sprintf(sprintf_str,
                 current_no,
                 str))
}


#' Gets the last figure caption number
#'
#' The function relies on \code{options("fig_caption_no")}
#' in order to keep track of the last number.
#'
#' @inheritParams figCapNo
#' @export
#' @examples
#' org_opts <- options(fig_caption_no=1)
#' figCapNoLast()
#' options(org_opts)
#' @family figure caption functions
figCapNoLast <- function(roman = getOption("fig_caption_no_roman",
                                           FALSE)){
  last_no <- getOption("fig_caption_no")
  if (is.logical(last_no) ||
        is.null(last_no)){
    stop("You cannot call the get last figure number",
         " when there has been no prior figure registerd.",
         " In other words, you need to call the fiCapNo()",
         " on a figure before you call this function.",
         " If you want the next number then call figCapNoNext()",
         " instead of this function.")
  }

  if (roman)
    last_no <- as.character(as.roman(last_no))

  return(last_no)
}

#' Gets the next figure caption number
#'
#' The function relies on \code{options("fig_caption_no")}
#' in order to keep track of the last number.
#'
#' @inheritParams figCapNo
#' @export
#' @examples
#' org_opts <- options(fig_caption_no=1)
#' figCapNoNext()
#' options(org_opts)
#' @family figure caption functions
figCapNoNext <- function(roman = getOption("fig_caption_no_roman",
                                           FALSE)){
  last_no <- getOption("fig_caption_no")
  if (is.logical(last_no)){
    if (last_no == FALSE)
      stop("You cannot call the get last figure number",
           " when you have explicitly set the fig_cap_no",
           " option to false.")
    last_no <- 0

  }else if (is.null(last_no)){
    last_no <- 0
  }

  next_no <- last_no + 1

  if (roman)
    next_no <- as.character(as.roman(next_no))

  return(next_no)
}
