## ---- include = FALSE-------------------------------------------------------------------------------------------------------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)
options(width = 180)

## ---- setup-----------------------------------------------------------------------------------------------------------------------------------------------------------------------
library(GeoFIS)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
data(tolima)
fusion <- NewFusion(tolima)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
potassium <- NewFusionInput("K", NewMfTrapezoidalSup(0.2, 0.6))
phosphorus <- NewFusionInput("P", NewMfTrapezoidalSup(5, 15))
balance_gap <- NewFusionInput("Bal_Gap", NewMfTrapezoidalInf(0, 0.5), "BalanceGap")
n_gap <- NewFusionInput("N_Gap", NewMfTriangular(0.5, 1, 1.5), "Ngap_N_OpN")
base_sat <- NewFusionInput("Base_Sat", NewMfTrapezoidalSup(0.4, 0.6), "Base_S")
org_matter <- NewFusionInput("OM", NewMfTrapezoidalSup(3, 5))
ph <- NewFusionInput("pH", NewMfTrapezoidal(5, 5.5, 6.5, 7.5))
cadmium <- NewFusionInput("Cd", NewMfTrapezoidalInf(0, 0.43), "Cadmium")

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
macronutrients_fis <- NewFisFusion("MacN", # Fis name
  c("Bal_Gap", "K", "P", "N_Gap", "Base_Sat"), # Fis inputs names
  c(2, 2, 2, 2, 2), # Fis inputs granularities
  "MacN", # Fis output name
  c(0, 0.1, 0.15, 0.2, 0.25, 0.35, 0.4, 0.45,
    0.3, 0.4, 0.45, 0.5, 0.55, 0.65, 0.7, 0.75, 
    0.4, 0.3, 0.4, 0.45, 0.5, 0.55, 0.65, 0.7, 
    0.55, 0.6, 0.7, 0.75, 0.8, 0.85, 0.9, 1) # Fis conclusions
)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
print(macronutrients_fis)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
macronutrients_aggreg <- NewFusionAggreg("MacN", 
  NewAggregFis(macronutrients_fis), 
  balance_gap, potassium, phosphorus, n_gap, base_sat)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
nutri_balance_fis <- NewFisFusion("Nutri_Bal", # Fis name 
  c("pH", "OM", "MacN"), # Fis inputs names
  c(2, 2, 2), # Fis inputs granularities
  "Nutri_Bal", # Fis output name
  c(0, 0.2, 0.3, 0.5, 0.4, 0.6, 0.7, 1.0) # Fis conclusions
)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
print(nutri_balance_fis)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
nutri_balance_aggreg <- NewFusionAggreg("Nutri_Bal", 
  NewAggregFis(nutri_balance_fis), 
  ph, org_matter, macronutrients_aggreg)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
chemical_aggreg <- NewFusionAggreg("Chemical", 
  NewAggregWam(weights = c(0.3, 0.7)), 
  cadmium, nutri_balance_aggreg)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
print(chemical_aggreg, "aggreg", "mf")

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
fusion$aggregate <- chemical_aggreg

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
fusion$perform()
output <- fusion$output()

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
print(output, digits = 2)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
data(fusion_cars)
print(fusion_cars)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
a <- NewFusionInput("µA", NewMfTrapezoidalInf(4, 20), "A")
v <- NewFusionInput("µV", NewMfTrapezoidalSup(100, 500), "V")
s <- NewFusionInput("µS", NewMfTrapezoidalSup(120, 220), "S")
c <- NewFusionInput("µC", NewMfTrapezoidalInf(6, 16), "C")

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
fusion <- NewFusion(fusion_cars)
fusion$aggregate <- list(a, v, s, c)
fusion$perform()
degrees <- fusion$output()
print(degrees)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
target <- c(0.8, 0.6, 0.6, 0.6)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
target <- c(1, 0, 0, 0)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
wam_weights <- LearnWamWeights(degrees, target)
print(wam_weights)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
wam_aggreg <- NewFusionAggreg("wam", NewAggregWam(wam_weights), a, v, s, c)
fusion$aggregate <- wam_aggreg
fusion$perform()
wam_inferred <- fusion$output()["wam"]
print(wam_inferred)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
owa_weights <- LearnOwaWeights(degrees, target)
print(owa_weights)

## ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
owa_aggreg <- NewFusionAggreg("owa", NewAggregOwa(owa_weights), a, v, s, c)
fusion$aggregate <- owa_aggreg
fusion$perform()
owa_inferred <- fusion$output()["owa"]
print(owa_inferred)

