mixed_model <- function (fixed, random, data, family, na.action = na.exclude,
                         penalized = FALSE, n_phis = NULL, initial_values = NULL, 
                         control = list(), ...) {
    call <- match.call()
    # set family
    if (is.character(family))
        family <- get(family, mode = "function", envir = parent.frame())
    if (is.function(family))
        family <- family()
    if (is.null(family$family)) {
        print(family)
        stop("'family' not recognized.\n")
    }
    if (family$family == "gaussian")
        stop("use function lme() from package 'nlme' or function lmer() from ",
             "package 'lme4'.\n")
    if (length(grep("Negative Binomial", family$family))) {
        stop("Because the namespace of the MASS package seems also to be loaded\n",
             "  use 'family = GLMMadaptive::negative.binomial(xx)' with 'xx' ", 
             "denoting a value for the\n  'theta' parameter of the family.")
    }
    known_families <- c("binomial", "poisson", "negative binomial")
    # extract response vector, design matrices, offset
    mfX <- model.frame(terms(fixed, data = data), data = data, na.action = na.action)
    na_exclude <- attr(mfX, "na.action")
    termsX <- terms(mfX)
    y <- model.response(mfX)
    if (is.factor(y)) {
        if (family$family == "binomial")
            y <- as.numeric(y != levels(y)[1L])
        else
            stop("the response variable should not be a factor.\n")
    }
    X <- model.matrix(termsX, mfX)
    offset <- model.offset(mfX)
    form_random <- getRE_Formula(random)
    mfZ <- model.frame(terms(form_random, data = data), data = data)
    if (!is.null(na_exclude))
        mfZ <- mfZ[-na_exclude, ]
    termsZ <- terms(mfZ)
    Z <- model.matrix(termsZ, mfZ)
    id_nam <- all.vars(getID_Formula(random))
    id_orig <- model.frame(terms(getID_Formula(random)), data)[[1L]]
    if (!is.null(na_exclude))
        id_orig <- id_orig[-na_exclude]
    id <- match(id_orig, unique(id_orig))
    ###########################
    # control settings
    con <- list(iter_EM = 30, iter_qN_outer = 15, iter_qN = 10, iter_qN_incr = 10,
                optim_method = "BFGS", parscale_betas = 0.1, parscale_D = 0.01,
                parscale_phis = 0.01, tol1 = 1e-03, tol2 = 1e-04, tol3 = 1e-07,
                numeric_deriv = "fd", nAGQ = if (ncol(Z) < 3) 11 else 7, 
                update_GH_every = 10, verbose = FALSE)
    control <- c(control, list(...))
    namC <- names(con)
    con[(namc <- names(control))] <- control
    if (length(noNms <- namc[!namc %in% namC]) > 0)
        warning("unknown names in control: ", paste(noNms, collapse = ", "))
    ###########################
    # initial values
    diag_D <- (random[[length(random)]])[[1]] == as.name("||")
    inits <- if (family$family %in% known_families || (is.list(initial_values) &&
                 inherits(initial_values$betas, 'family'))) {
        betas <- if (family$family %in% known_families) {
            glm.fit(X, y, family = family)$coefficients
        } else {
            glm.fit(X, y, family = initial_values$betas)$coefficients
        }
        list(betas = betas * sqrt(1.346), D = if (diag_D) rep(1, ncol(Z)) else diag(ncol(Z)))
    } else {
        list(betas = rep(0, ncol(X)), D = if (diag_D) rep(1, ncol(Z)) else diag(ncol(Z)))
    }
    if (!is.null(initial_values) && is.list(initial_values) &&
        !inherits(initial_values$betas, 'family')) {
        lngths <- lapply(inits[(nams.initial_values <- names(initial_values))], length)
        if (!isTRUE(all.equal(lngths, lapply(initial_values, length)))) {
            warning("'initial_values' is not a list with elements of appropriate ",
                    "length; default initial_values are used instead.\n")
        } else {
            inits[nams.initial_values] <- initial_values
        }
    }
    ##########################
    # penalized
    penalized <- if (is.logical(penalized) && !penalized) {
        list(penalized = penalized)
    } else if (is.logical(penalized) && penalized) {
        list(penalized = penalized, pen_mu = 0, pen_sigma = 1, pen_df = 3)
    } else if (is.list(penalized)) {
        if (!all(names(penalized) %in% c("pen_mu", "pen_sigma", "pen_df")))
            stop("when argument 'penalized' is a list it needs to have the components ",
                 "'pen_mu', 'pen_sigma' and 'pen_df'.\n")
        c(list(penalized = TRUE), penalized)
    } else {
        stop("argument 'penalized' must be a logical or a list.\n")
    }
    ##########################
    # Functions
    Funs <- list(
        mu_fun = family$linkinv,
        var_fun = family$variance,
        mu.eta_fun = family$mu.eta
    )
    if (family$family %in% known_families && is.null(family$log_den)) {
        Funs$log_dens <- switch(family$family,
               'binomial' = binomial_log_dens,
               'poisson' = poisson_log_dens,
               'negative binomial' = negative.binomial_log_dens)
    } else if (!family$family %in% known_families && !is.null(family$log_dens)) {
        Funs$log_dens <- family$log_dens
    } else {
        stop("'log_dens' component of the 'family' argument is NULL with no default.\n")
    }
    if (!is.function(Funs$log_dens)) {
        stop("'log_dens' component of the 'family' argument must be a function.\n")
    }
    if (!is.function(Funs$mu_fun)) {
        stop("'linkinv' component of the 'family' argument must be a function.\n")
    }
    if (!is.function(Funs$mu_fun)) {
        stop("'linkinv' component of the 'family' argument must be a function.\n")
    }
    if (!is.null(family$score_eta_fun) && is.function(family$score_eta_fun)) {
        Funs$score_eta_fun <- family$score_eta_fun
    }
    if (!is.null(family$score_phis_fun) && is.function(family$score_phis_fun)) {
        Funs$score_phis_fun <- family$score_phis_fun
    }
    has_phis <- inherits(try(Funs$log_dens(y, 0, Funs$mu_fun, phis = NULL), TRUE),
                         "try-error")
    if (has_phis) {
        if (family$family == "negative binomial") {
            n_phis <- 1
        } else if (is.null(n_phis)) {
            stop("argument 'n_phis' needs to be specified.\n")
        }
        inits$phis <- rep(0.0, n_phis)
    }
    ###############
    # Fit the model
    out <- mixed_fit(y, X, Z, id, offset, family, inits, Funs, con, penalized)
    # fix names
    names(out$coefficients) <- colnames(X)
    dimnames(out$D) <- list(colnames(Z), colnames(Z))
    if (!is.null(out$phis))
        names(out$phis) <- paste0("phi_", seq_along(out$phis))
    all_nams <- if (diag_D) {
        nams_D <- paste0("D_", seq_len(ncol(Z)), seq_len(ncol(Z)))
        c(names(out$coefficients), nams_D, names(out$phis))
    } else {
        nams_D <- paste0("D_", apply(which(upper.tri(out$D, TRUE), arr.ind = TRUE), 1, 
                                     paste0, collapse = ""))
        c(names(out$coefficients), nams_D, names(out$phis))
    }
    dimnames(out$Hessian) <- list(all_nams, all_nams)
    out$id <- id_orig
    out$id_name <- id_nam 
    out$offset <- offset
    dimnames(out$post_modes) <- list(unique(id_orig), colnames(Z))
    names(out$post_vars) <- unique(id_orig)
    out$post_vars[] <- lapply(out$post_vars, function (v) {
        dimnames(v) <- list(colnames(Z), colnames(Z))
        v
    })
    out$Terms <- list(termsX = termsX, termsZ = termsZ)
    out$model_frames <- list(mfX = mfX, mfZ = mfZ)
    out$control <- con
    out$Funs <- Funs
    out$family <- family
    out$call <- call
    class(out) <- "MixMod"
    out
}
