% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dif.R, R/summary.GDINA.R
\name{dif}
\alias{dif}
\alias{summary.dif}
\title{Differential item functioning for cognitive diagnosis models}
\usage{
dif(
  dat,
  Q,
  group,
  model = "GDINA",
  method = "wald",
  anchor.items = NULL,
  dif.items = "all",
  p.adjust.methods = "holm",
  approx = FALSE,
  SE.type = 2,
  FS.args = list(on = FALSE, alpha.level = 0.05, maxit = 10, verbose = FALSE),
  ...
)

\method{summary}{dif}(object, ...)
}
\arguments{
\item{dat}{item responses from two groups; missing data need to be coded as \code{NA}}

\item{Q}{Q-matrix specifying the association between items and attributes}

\item{group}{a factor or a vector indicating the group each individual belongs to. Its length must be equal to the number of individuals.}

\item{model}{model for each item.}

\item{method}{DIF detection method; It can be \code{"wald"} for Hou, de la Torre, and Nandakumar's (2014)
Wald test method, and \code{"LR"} for likelihood ratio test (Ma, Terzi, Lee,& de la Torre, 2017).}

\item{anchor.items}{which items will be used as anchors? Default is \code{NULL}, which means none of the items are used as anchors.
For LR method, it can also be an integer vector giving the item numbers for anchors or \code{"all"}, which means all items are treated as anchor items.}

\item{dif.items}{which items are subject to DIF detection? Default is \code{"all"}. It can also be an integer vector giving the item numbers.}

\item{p.adjust.methods}{adjusted p-values for multiple hypothesis tests. This is conducted using \code{p.adjust} function in \pkg{stats},
and therefore all adjustment methods supported by \code{p.adjust} can be used, including \code{"holm"},
\code{"hochberg"}, \code{"hommel"}, \code{"bonferroni"}, \code{"BH"} and \code{"BY"}. See \code{p.adjust}
for more details. \code{"holm"} is the default.}

\item{approx}{Whether an approximated LR test is implemented? If TRUE, parameters of items except the studied one will not be re-estimated.}

\item{SE.type}{Type of standard error estimation methods for the Wald test.}

\item{FS.args}{arguments for the forward anchor item search procedure developed in Ma, Terzi, and de la Torre (2021). A list with the following elements:
\itemize{
  \item \code{on} - logical; \code{TRUE} if activate the forward anchor item search procedure. Default = \code{FALSE}.
  \item \code{alpha.level} - nominal level for Wald or LR test. Default = .05.
  \item \code{maxit} - maximum number of iterations allowed. Default = 10.
  \item \code{verbose} - logical; print information for each iteration or not? Default = \code{FALSE}.
  }}

\item{...}{arguments passed to GDINA function for model calibration}

\item{object}{dif object for S3 method}
}
\value{
A data frame giving the Wald statistics and associated p-values.
}
\description{
This function is used to detect differential item functioning using the Wald test (Hou, de la Torre, & Nandakumar, 2014; Ma, Terzi, & de la Torre, 2021) and the likelihood ratio
test (Ma, Terzi, & de la Torre, 2021). The forward anchor item search procedure developed in Ma, Terzi, and de la Torre (2021) was implemented. Note that it can only detect DIF for two groups currently.
}
\section{Methods (by generic)}{
\itemize{
\item \code{summary}: print summary information
}}

\examples{
\dontrun{
set.seed(123456)
N <- 3000
Q <- sim30GDINA$simQ
gs <- matrix(.2,ncol = 2, nrow = nrow(Q))
# By default, individuals are simulated from uniform distribution
# and deltas are simulated randomly
sim1 <- simGDINA(N,Q,gs.parm = gs,model="DINA")
sim2 <- simGDINA(N,Q,gs.parm = gs,model=c(rep("DINA",nrow(Q)-1),"DINO"))
dat <- rbind(extract(sim1,"dat"),extract(sim2,"dat"))
gr <- rep(c("G1","G2"),each=N)

# DIF using Wald test
dif.wald <- dif(dat, Q, group=gr, method = "Wald")
dif.wald
# DIF using LR test
dif.LR <- dif(dat, Q, group=gr, method="LR")
dif.LR
# DIF using Wald test + forward search algorithm
dif.wald.FS <- dif(dat, Q, group=gr, method = "Wald", FS.args = list(on = TRUE, verbose = TRUE))
dif.wald.FS
# DIF using LR test + forward search algorithm
dif.LR.FS <- dif(dat, Q, group=gr, method = "LR", FS.args = list(on = TRUE, verbose = TRUE))
dif.LR.FS
}
}
\references{
Hou, L., de la Torre, J., & Nandakumar, R. (2014). Differential item functioning assessment in cognitive diagnostic modeling: Application of the Wald test to
investigate DIF in the DINA model. \emph{Journal of Educational Measurement, 51}, 98-125.

Ma, W., Terzi, R., & de la Torre, J. (2021). Detecting differential item functioning using multiple-group cognitive diagnosis models. \emph{Applied Psychological Measurement}.
}
\seealso{
\code{\link{GDINA}}
}
\author{
{Wenchao Ma, The University of Alabama, \email{wenchao.ma@ua.edu} \cr Jimmy de la Torre, The University of Hong Kong}
}
