#' Euronext Paris ticker ISIN
#'
#' @details This function retrieves the ISIN (International Securities Identification Number)
#' of a company listed on the Euronext Paris exchange through the Euronext site. The function
#' takes a single parameter `ticker`. The function automatically formats the
#' ticker to uppercase using `toupper()`. It checks the existence of the ticker
#' and returns its corresponding ISIN.
#'
#' @param ticker A character string representing the company's ticker.
#'
#' @family IsiN Retrieval
#' @family Euronext
#'
#' @return A character
#' @export
#'
#' @examples
#'\donttest{
#' EN_P_Ticker_adn("abca")
#' result <- EN_P_Ticker_adn("AAA")
#' print(EN_P_Ticker_adn("afsf")) # Ticker is not existing
#'}


EN_P_Ticker_adn <- function(ticker) {

  ticker <- toupper(ticker)


  EN_paris_ticker <-  list(AAA = "FR0000062465-XPAR", AB = "FR0010557264-XPAR", ABCA = "FR0004040608-XPAR",
                           ABEO = "FR0013185857-XPAR", ABLD = "FR001400AHX6-XPAR", ABNX = "FR0012616852-XPAR",
                           ABO = "BE0974278104-XBRU", ABVX = "FR0012333284-XPAR", AC = "FR0000120404-XPAR",
                           ACA = "FR0000045072-XPAR", ACAN = "FR0000064602-XPAR", ADOC = "FR0011184241-XPAR",
                           ADP = "FR0010340141-XPAR", ADUX = "FR0012821890-XPAR", AELIS = "FR0014007ZB4-XPAR",
                           AF = "FR001400J770-XPAR", AFMBS = "FR001400FL38-XPAR", AFME = "FR0013333077-XPAR",
                           AI = "FR0000120073-XPAR", AIR = "NL0000235190-XPAR", AKE = "FR0010313833-XPAR",
                           AKOM = "US00774B2088-XPAR", AKW = "FR0000053027-XPAR", AL2SI = "FR0013341781-ALXP",
                           ALACT = "FR0014005OJ5-ALXP", ALADO = "FR001400JAL7-ALXP",
                           ALAFY = "FR0014005AC9-ALXP", ALAGO = "FR0014004339-ALXP",
                           ALAGP = "FR0013452281-ALXP", ALAGR = "FR0010641449-ALXP",
                           ALAIR = "FR0014003V77-ALXP", ALALO = "FR001400JAP8-ALXP",
                           ALAMA = "GB00BNKGZC51-ALXP", ALAMG = "FR0013410370-ALXP",
                           ALAQU = "FR0010340711-ALXP", ALARF = "FR0013284627-ALXP",
                           ALAST = "FR0000076887-ALXP", ALATA = "FR0010478248-ALXP",
                           ALATI = "FR0000076655-ALXP", ALAUD = "FR00140059B5-ALXP",
                           ALAUR = "FR0013183589-ALXP", ALAVI = "FR0000053043-ALXP",
                           ALBDM = "FR0000035305-ALXP", ALBFR = "FR0010202606-ALXP",
                           ALBI = "FR0000124414-ALXP", ALBIO = "FR0011005933-ALXP",
                           ALBIZ = "FR0014003711-ALXP", ALBKK = "FR0013384369-ALXP",
                           ALBLD = "FR0004174233-ALXP", ALBLU = "FR0011041011-ALXP",
                           ALBOA = "FR0011365907-ALXP", ALBON = "FR0000121295-ALXP",
                           ALBOO = "FR001400IAM7-ALXP", ALBOU = "FR0000054421-ALXP",
                           ALBPK = "FR001400AJZ7-ALXP", ALBPS = "FR0012816825-ALXP",
                           ALCAB = "FR001400DIY6-ALXP", ALCAP = "FR0012969095-ALXP",
                           ALCAR = "FR0010907956-ALXP", ALCBI = "FR0014007LW0-ALXP",
                           ALCBX = "FR0000054322-ALXP", ALCHI = "FR0014000JX7-ALXP",
                           ALCIS = "FR0000064446-ALXP", ALCJ = "FR0011716265-ALXP",
                           ALCLA = "FR0004152882-ALXP", ALCLS = "FR0010425595-ALXP",
                           ALCOF = "FR0013257409-ALXP", ALCOG = "FR0011071570-ALXP",
                           ALCOI = "BE0160342011-ALXP", ALCOX = "FR0013018124-ALXP",
                           ALCRB = "FR0011648716-ALXP", ALCUR = "FR0013398997-ALXP",
                           ALCWE = "FR001400AJ60-ALXP", ALCYB = "FR0014004QR6-ALXP",
                           ALD = "FR0013258662-XPAR", ALDAR = "FR0000185423-ALXP", ALDBL = "FR0000066961-ALXP",
                           ALDBT = "FR001400LO86-ALXP", ALDEL = "FR0000054132-ALXP",
                           ALDEV = "FR0000060840-ALXP", ALDLS = "FR0010404368-ALXP",
                           ALDLT = "FR0013283108-ALXP", ALDMS = "FR0012202497-ALXP",
                           ALDNE = "FR0013331212-ALXP", ALDNX = "FR0010436584-ALXP",
                           ALDOL = "FR0014004QZ9-ALXP", ALDRV = "FR0013088606-ALXP",
                           ALDUB = "FR0004030708-ALXP", ALDV = "BE0974269012-XPAR",
                           ALDVI = "FR0013296746-ALXP", ALEAC = "IT0005351504-ALXP",
                           ALECO = "FR0013534617-ALXP", ALECP = "FR0010490920-ALXP",
                           ALECR = "FR0013256518-ALXP", ALEMG = "FR0000075343-ALXP",
                           ALEMV = "FR0013356755-ALXP", ALENE = "FR0011915339-ALXP",
                           ALENI = "FR001400C2Z4-ALXP", ALENO = "FR0014004974-ALXP",
                           ALENR = "FR0000045122-ALXP", ALENT = "FR0010424697-ALXP",
                           ALEO2 = "FR0010465534-ALXP", ALERS = "FR0013240934-ALXP",
                           ALESA = "FR0011490648-ALXP", ALESE = "FR0014004362-ALXP",
                           ALESK = "FR0000035818-ALXP", ALEUA = "FR0010844001-ALXP",
                           ALEUP = "FR001400CF13-ALXP", ALEXA = "FR0000064164-ALXP",
                           ALEXP = "FR0013266772-ALXP", ALFBA = "FR0004034593-ALXP",
                           ALFLE = "FR0000074759-ALXP", ALFLO = "FR001400GO75-ALXP",
                           ALFOC = "FR0012419307-ALXP", ALFPC = "FR0010485268-ALXP",
                           ALFRE = "FR0004187367-ALXP", ALFUM = "FR001400AEM6-ALXP",
                           ALGAU = "FR0013495298-ALXP", ALGBE = "FR0011052257-ALXP",
                           ALGEC = "FR001400M1R1-ALXP", ALGEN = "FR0004053510-ALXP",
                           ALGEV = "FR0000033888-ALXP", ALGID = "FR0000072373-ALXP",
                           ALGIL = "FR0012819381-ALXP", ALGIR = "FR0000060790-ALXP",
                           ALGLD = "FR0011208693-ALXP", ALGRE = "FR001400IV58-ALXP",
                           ALGRO = "FR0014005ZM5-ALXP", ALGTR = "FR0013429404-ALXP",
                           ALHAF = "FR0014007ND6-ALXP", ALHEX = "FR0004159473-ALXP",
                           ALHF = "FR0000038531-ALXP", ALHGO = "GB00BMDXQ672-ALXP",
                           ALHGR = "FR0013451044-ALXP", ALHIT = "FR0010396309-ALXP",
                           ALHOP = "FR0000065278-ALXP", ALHPI = "FR0014000U63-ALXP",
                           ALHRG = "FR0000066540-ALXP", ALHRS = "FR0014001PM5-ALXP",
                           ALHUN = "FR0014007LQ2-ALXP", ALHYG = "FR0004153930-ALXP",
                           ALHYP = "FR0012821916-ALXP", ALI2S = "FR0005854700-ALXP",
                           ALICA = "FR001400A3Q3-ALXP", ALICR = "FR0014005IU4-ALXP",
                           ALIDS = "FR0000062184-ALXP", ALIE = "IT0005380602-ALXP",
                           ALIKO = "FR00140048X2-ALXP", ALIMG = "BE0974289218-ALXB",
                           ALIMO = "FR0014000RP6-ALXP", ALIMP = "FR0013470168-ALXP",
                           ALIMR = "FR0013060100-ALXP", ALINN = "FR0000064297-ALXP",
                           ALINS = "FR0011179886-ALXP", ALINT = "FR0010908723-ALXP",
                           ALINV = "BE0974299316-ALXP", ALISP = "ES0109429037-ALXP",
                           ALITL = "FR0000072597-ALXP", ALJXR = "FR001400KO61-ALXP",
                           ALKAL = "FR0010722819-ALXP", ALKEM = "FR00140069V2-ALXP",
                           ALKER = "IT0005380438-ALXP", ALKEY = "FR0004029411-ALXP",
                           ALKKO = "FR0013374667-ALXP", ALKLA = "FR0011038348-ALXP",
                           ALKLK = "FR0013156007-ALXP", ALKOM = "ES0105425005-ALXP",
                           ALLAM = "BE0974334667-ALXP", ALLAN = "FR0004027068-ALXP",
                           ALLDL = "FR0000075442-ALXP", ALLEC = "FR0013335742-ALXP",
                           ALLEX = "FR0000033599-ALXP", ALLGO = "FR0013308582-ALXP",
                           ALLHB = "FR0007080254-ALXP", ALLIX = "FR0010131409-ALXP",
                           ALLLN = "ES0105089009-ALXP", ALLOG = "FR0000044943-ALXP",
                           ALLPL = "FR001400F2Z1-ALXP", ALLUX = "FR0000060451-ALXP",
                           ALMAR = "FR0013400835-ALXP", ALMAS = "FR0004155687-ALXP",
                           ALMCE = "FR001400H3A0-ALXP", ALMDG = "FR0010353888-ALXP",
                           ALMDP = "FR0010844464-ALXP", ALMDT = "FR0011049824-ALXP",
                           ALMER = "FR0010776617-ALXP", ALMET = "FR0011217710-ALXP",
                           ALMEX = "IT0005450819-ALXP", ALMGI = "FR0010204453-ALXP",
                           ALMIB = "FR0011051598-ALXP", ALMIC = "FR0000077570-ALXP",
                           ALMII = "FR0013270626-ALXP", ALMIL = "FR0010285965-ALXP",
                           ALMIN = "FR0004172450-ALXP", ALMKS = "ES0105463006-ALXP",
                           ALMKT = "FR0010609263-ALXP", ALMLB = "FR0013053535-ALXP",
                           ALMND = "FR00140050Q2-ALXP", ALMOU = "FR0011033083-ALXP",
                           ALMRB = "FR0004034320-ALXP", ALMUN = "FR0013462231-ALXP",
                           ALNEO = "FR0013421286-ALXP", ALNEV = "FR001400HDX0-ALXP",
                           ALNFL = "FR0014003XT0-ALXP", ALNLF = "FR0011636083-ALXP",
                           ALNMG = "FR0000072993-ALXP", ALNMR = "FR0014003J32-ALXP",
                           ALNN6 = "FR0013330792-ALXP", ALNOV = "FR0010397232-ALXP",
                           ALNRG = "FR0013399359-ALXP", ALNSC = "FR0000064529-ALXP",
                           ALNSE = "FR0004065639-ALXP", ALNTG = "FR0004154060-ALXP",
                           ALNXT = "FR0004171346-ALXP", ALO = "FR0010220475-XPAR", ALODC = "FR0014003T71-ALXP",
                           ALOKW = "FR0013439627-ALXP", ALOPM = "FR001400CM63-ALXP",
                           ALORA = "FR0000061244-ALXP", ALORD = "FR0013318052-ALXP",
                           ALOSM = "FR0013231180-ALXP", ALPAR = "FR0013204070-ALXP",
                           ALPAT = "FR0010785790-ALXP", ALPAU = "FR0013479730-ALXP",
                           ALPCV = "FR0013178712-ALXP", ALPDX = "FR0000061608-ALXP",
                           ALPER = "NL0012191662-ALXP", ALPHA = "FR001400JXB0-ALXP",
                           ALPHI = "ES0105029005-ALXP", ALPHS = "BE0974429624-ALXP",
                           ALPIX = "FR001400JX97-ALXP", ALPJT = "FR0000066441-ALXP",
                           ALPLA = "FR0010211037-ALXP", ALPM = "FR0014004EC4-ALXP",
                           ALPOU = "FR0013015583-ALXP", ALPRE = "FR0010169920-ALXP",
                           ALPRG = "FR0010380626-ALXP", ALPRI = "FR0004044600-ALXP",
                           ALPRO = "FR0010313486-ALXP", ALQGC = "FR0011648971-ALXP",
                           ALQP = "ES0105118006-ALXP", ALQWA = "FR0010889386-ALXP",
                           ALREA = "FR0011858190-ALXP", ALREW = "FR0010820274-ALXP",
                           ALRFG = "IT0005466963-ALXP", ALRGR = "FR0000037640-ALXP",
                           ALRIB = "FR0000075954-ALXP", ALROC = "FR0010523167-ALXP",
                           ALRPD = "GB00BM9PTW47-ALXP", ALSAF = "FR001400F1V2-ALXP",
                           ALSAS = "FR0000074775-ALXP", ALSEI = "IT0005353484-ALXP",
                           ALSEN = "FR0012596468-ALXP", ALSGD = "FR0011464452-ALXP",
                           ALSMA = "FR0014005I80-ALXP", ALSOG = "FR0000065864-ALXP",
                           ALSPT = "FR00140043Y1-ALXP", ALSPW = "FR001400BVK2-ALXP",
                           ALSRS = "FR0010812230-ALXP", ALSTI = "FR001400MDW2-ALXP",
                           ALSTW = "FR0010528059-ALXP", ALTA = "FR0000033219-XPAR",
                           ALTBG = "FR0011053636-ALXP", ALTD = "FR001400H2X4-ALXP",
                           ALTHE = "FR0010120402-ALXP", ALTHO = "FR00140066X4-ALXP",
                           ALTHX = "FR0013286259-ALXP", ALTLX = "LU0394945660-ALXP",
                           ALTME = "NL0015000YE1-ALXP", ALTOO = "FR00140062B9-ALXP",
                           ALTPC = "FR0004016699-ALXP", ALTRI = "FR0010397901-ALXP",
                           ALTRO = "FR0004175099-ALXP", ALTTI = "FR0010383877-ALXP",
                           ALTTU = "IT0005507857-ALXP", ALTUR = "FR0010395681-XPAR",
                           ALTUV = "FR0013345493-ALXP", ALTXC = "FR0010654087-ALXP",
                           ALU10 = "FR0000079147-ALXP", ALUCI = "FR0011884378-ALXP",
                           ALUNT = "FR0012709160-ALXP", ALUPG = "FR0010337865-ALXP",
                           ALUVI = "FR0011898584-ALXP", ALVAL = "FR0013254851-ALXP",
                           ALVAP = "FR0013419876-ALXP", ALVDM = "FR0004045847-ALXP",
                           ALVER = "FR001400JXA2-ALXP", ALVET = "BE0974387194-ALXP",
                           ALVG = "FR0013530102-ALXP", ALVGO = "FR0011532225-ALXP",
                           ALVIA = "FR0010326090-ALXP", ALVIN = "FR001400AXT1-ALXP",
                           ALVIO = "FR0010095596-ALXP", ALVIV = "FR0004029478-ALXP",
                           ALVMG = "FR0013481835-ALXP", ALVU = "FR0010766667-ALXP",
                           ALWEC = "FR0013079092-ALXP", ALWED = "FR0010688440-ALXP",
                           ALWF = "FR0014000P11-ALXP", ALWIT = "FR0013143872-ALXP",
                           ALWTR = "FR001400IUV6-ALXP", AM = "FR0014004L86-XPAR", AMPLI = "FR0012789667-XPAR",
                           AMUN = "FR0004125920-XPAR", ANTIN = "FR0014005AL0-XPAR",
                           APAM = "LU0569974404-XAMS", ARAMI = "FR0014003U94-XPAR",
                           ARDO = "FR0004070795-XMLI", AREIT = "FR0000039216-XPAR",
                           ARG = "FR0010481960-XPAR", ARTE = "FR0012185536-XPAR", ARTO = "FR0000076952-XPAR",
                           ARVBS = "FR0014003AC4-XPAR", ARVEN = "FR001400JWR8-XPAR",
                           ASY = "FR0000074148-XPAR", ATE = "FR0000071946-XPAR", ATEME = "FR0011992700-XPAR",
                           ATLD = "FR0013455482-XPAR", ATO = "FR0000051732-XPAR", AUB = "FR0000063737-XPAR",
                           AUGR = "FR0000061780-XPAR", AURE = "FR0000039232-XPAR", AVT = "FR001400CFI7-XPAR",
                           AXW = "FR0011040500-XPAR", BAIN = "MC0000031187-XPAR", BALYO = "FR0013258399-XPAR",
                           BASS = "FR0004023208-XPAR", BB = "FR0000120966-XPAR", BEN = "FR0000035164-XPAR",
                           BERR = "FR0000062341-XPAR", BIG = "FR0000074072-XPAR", BIM = "FR0013280286-XPAR",
                           BIOS = "BE0974280126-XBRU", BLC = "FR0000035370-XPAR", BLEE = "FR0000062150-XPAR",
                           BLV = "FR0014003FE9-XPAR", BN = "FR0000120644-XPAR", BNBS = "FR0014007951-ALXP",
                           BNP = "FR0000131104-XPAR", BOI = "FR0000061129-XPAR", BOL = "FR0000039299-XPAR",
                           BON = "FR0000063935-XPAR", BPBS = "FR001400LN79-ALXP", BPSBS = "FR0013507290-ALXP",
                           BSD = "FR0000074254-XPAR", BUI = "FR0000062788-XPAR", BUR = "FR0000061137-XPAR",
                           BVI = "FR0006174348-XPAR", CA = "FR0000120172-XPAR", CAF = "FR0000045528-XPAR",
                           CAFO = "FR0010151589-XPAR", CAP = "FR0000125338-XPAR", CARM = "FR0010828137-XPAR",
                           CARP = "FR0000064156-XPAR", CAT31 = "FR0000045544-XPAR",
                           CATG = "FR0010193052-XPAR", CATR = "US1491231015-XPAR", CBDG = "FR0000079659-XPAR",
                           CBE = "FR0000045601-XPAR", CBIBS = "FR001400D0X2-ALXP", CBOT = "FR0010193979-XPAR",
                           CBR = "FR0000045619-XPAR", CBSM = "FR0006239109-XPAR", CCN = "FR0000044364-XPAR",
                           CDA = "FR0000053324-XPAR", CDI = "FR0000130403-XPAR", CEN = "FR0000036675-XPAR",
                           CFI = "FR0000037475-XPAR", CGG = "FR0013181864-XPAR", CGM = "FR0000053506-XPAR",
                           CHSR = "FR0000060907-XPAR", CIV = "FR0000045213-XPAR", CLA = "FR0013426004-XPAR",
                           CLARI = "FR0010386334-XPAR", CMO = "FR0000045551-XPAR", CNDF = "FR0000185514-XPAR",
                           CNV = "NL0010949392-XPAR", CO = "FR0000125585-XPAR", COFA = "FR0010667147-XPAR",
                           COH = "FR0004031763-XPAR", COTY = "US2220702037-XPAR", COUR = "FR0000065393-XPAR",
                           COV = "FR0000064578-XPAR", COVH = "FR0000060303-XPAR", CRAP = "FR0000044323-XPAR",
                           CRAV = "FR0000185506-XPAR", CRBP2 = "FR0010483768-XPAR",
                           CRI = "FR0000130692-XPAR", CRLA = "FR0010461053-XPAR", CRLO = "FR0000045239-XPAR",
                           CROS = "FR0000050395-XPAR", CRSU = "FR0000045346-XPAR", CRTO = "FR0000045304-XPAR",
                           CS = "FR0000120628-XPAR", CSNV = "FR001400M7B2-XPAR", CV = "BMG9887P1068-XPAR",
                           CYAD = "BE0974260896-XBRU", CYBK1 = "FR0013507977-ALXP",
                           CYBKA = "FR0013507985-ALXP", CYBKB = "FR0013508009-ALXP",
                           DBG = "FR0000053381-XPAR", DBV = "FR0010417345-XPAR", DEC = "FR0000077919-XPAR",
                           DEEZR = "FR001400AYG6-XPAR", DEEZW = "FR0014004JF6-XPAR",
                           DG = "FR0000125486-XPAR", DIM = "FR0013154002-XPAR", DKUPL = "FR0000062978-XPAR",
                           DMSBS = "FR001400IAQ8-ALXP", DPAM = "FR0000065260-XPAR",
                           DPT = "FR0000054199-XPAR", DPTDS = "FR001400N0G5-XPAR", DSY = "FR0014003TT8-XPAR",
                           EAPI = "FR0014008VX5-XPAR", EC = "GA0000121459-XPAR", EDEN = "FR0010908533-XPAR",
                           EDI = "FR0000064404-XPAR", EEM = "FR0000035719-XPAR", EFI = "FR0000061475-XPAR",
                           EGR = "FR0000035784-XPAR", EIFF = "FR0000036816-XPAR", EKI = "FR0011466069-XPAR",
                           EL = "FR0000121667-XPAR", ELEC = "FR0000031023-XPAR", ELIOR = "FR0011950732-XPAR",
                           ELIS = "FR0012435121-XPAR", EN = "FR0000120503-XPAR", ENGI = "FR0010208488-XPAR",
                           ENNV = "FR001400M3D7-XPAR", ENX = "NL0006294274-XPAR", EOS = "FR0000076861-XPAR",
                           EQS = "FR0012882389-XPAR", ERA = "FR0000131757-XPAR", ERF = "FR0014000MR3-XPAR",
                           ERFNV = "FR001400M7C0-XPAR", ES = "FR0000120669-XPAR", ETL = "FR0010221234-XPAR",
                           EUR = "FR0000054678-XPAR", EURS = "FR0000038499-XPAR", EXA = "FR0000062671-XPAR",
                           EXE = "FR0004527638-XPAR", EXN = "FR0014005DA7-XPAR", EXPL = "FR0014003AQ4-XPAR",
                           FALG = "FR0011271600-XPAR", FAYE = "FR0000031973-XPAR", FCMC = "FR0000062101-XPAR",
                           FDE = "FR0013030152-XPAR", FDJ = "FR0013451333-XPAR", FGA = "FR0011665280-XPAR",
                           FGR = "FR0000130452-XPAR", FII = "FR0000050353-XPAR", FINM = "FR0000060824-XPAR",
                           FIPP = "FR0000038184-XPAR", FLY = "FR0000033409-XPAR", FMONC = "FR0000076986-XPAR",
                           FNAC = "FR0011476928-XPAR", FNTS = "FR0000035123-XPAR", FOAF = "SN0000033192-XPAR",
                           FORE = "CI0000053161-XPAR", FORSE = "FR0014005SB3-XPAR",
                           FPG = "FR0000074197-XPAR", FR = "FR0013176526-XPAR", FREY = "FR0010588079-XPAR",
                           FRVIA = "FR0000121147-XPAR", GALIM = "FR0000030611-XPAR",
                           GAM = "FR0000034894-XPAR", GBT = "FR0000032526-XPAR", GDS = "FR0000044471-XPAR",
                           GEA = "FR0000053035-XPAR", GET = "FR0010533075-XPAR", GFC = "FR0010040865-XPAR",
                           GJAJ = "FR0004010338-XPAR", GLE = "FR0000130809-XPAR", GLO = "FR0000066672-XPAR",
                           GNE = "US3696043013-XPAR", GNFT = "FR0004163111-XPAR", GNRO = "CH0308403085-XPAR",
                           GPE = "FR0010214064-XPAR", GRVO = "FR0000065971-XPAR", GTBP = "US36254L2097-XPAR",
                           GTT = "FR0011726835-XPAR", GUI = "FR0000066722-XPAR", GV = "FR0011799907-XPAR",
                           HCO = "FR0000054231-XPAR", HDF = "FR0014003VY4-XPAR", HDP = "FR0004165801-XPAR",
                           HO = "FR0000121329-XPAR", IAM = "MA0000011488-XPAR", ICAD = "FR0000035081-XPAR",
                           IDIP = "FR0000051393-XPAR", IDL = "FR0010929125-XPAR", IMDA = "FR0000033243-XPAR",
                           IMPDS = "FR001400MDQ4-ALXP", INEA = "FR0010341032-XPAR",
                           INF = "FR0000071797-XPAR", INFE = "FR0000066219-XPAR", IPH = "FR0010331421-XPAR",
                           IPN = "FR0010259150-XPAR", IPS = "FR0000073298-XPAR", ITE = "FR0004026151-XPAR",
                           ITP = "FR0004024222-XPAR", ITXT = "FR0000064958-XPAR", IVA = "FR0013233012-XPAR",
                           JBOG = "FR0012872141-XPAR", JCQ = "FR0000033904-XPAR", KER = "FR0000121485-XPAR",
                           KOF = "FR0004007813-XPAR", LACR = "FR0000066607-XPAR", LAT = "FR001400JY13-XPAR",
                           LBIRD = "FR0000038242-XPAR", LEBL = "FR0000065930-XPAR",
                           LHYFE = "FR0014009YQ1-XPAR", LI = "FR0000121964-XPAR", LIN = "FR0004156297-XPAR",
                           LNA = "FR0004170017-XPAR", LOCAL = "FR00140006O9-XPAR", LOUP = "FR0013204336-XPAR",
                           LPE = "FR0006864484-XPAR", LR = "FR0010307819-XPAR", LSS = "FR0000065484-XPAR",
                           LSSNV = "FR001400MMI2-XPAR", LTA = "FR0000053837-XPAR", MAAT = "FR0012634822-XPAR",
                           MALT = "FR0000030074-XPAR", MASBS = "FR0013472446-ALXP",
                           MAU = "FR0000051070-XPAR", MBWS = "FR0000060873-XPAR", MC = "FR0000121014-XPAR",
                           MCPHY = "FR0011742329-XPAR", MDM = "FR0013153541-XPAR", MEDCL = "FR0004065605-XPAR",
                           MEMS = "FR0010298620-XPAR", MERY = "FR0010241638-XPAR", METEX = "FR0004177046-XPAR",
                           MF = "FR0000121204-XPAR", MHM = "FR001400IE67-XPAR", ML = "FR001400AJ45-XPAR",
                           MLAA = "FR0014005WE9-XMLI", MLAAH = "NL0010273694-XMLI",
                           MLAAT = "PTAZR0AM0006-XMLI", MLABC = "ES0105551008-XMLI",
                           MLACT = "FR0010979377-XMLI", MLAEM = "FR0012968485-XMLI",
                           MLAGI = "FR0011908045-XMLI", MLAGP = "ES0105478004-XMLI",
                           MLAIG = "ES0105744009-XMLI", MLALE = "GB00BJ9M4V82-XMLI",
                           MLALV = "FR0013253812-XMLI", MLARD = "FR0005057635-XMLI",
                           MLARE = "ES0105661005-XMLI", MLARI = "ES0105601001-XMLI",
                           MLARO = "ES0105486007-XMLI", MLASO = "ES0105658001-XMLI",
                           MLAST = "ES0105708004-XMLI", MLAUD = "FR0011896463-XMLI",
                           MLAZL = "IT0004812258-XMLI", MLAZR = "GRS528003007-XMLI",
                           MLBAR = "ES0105362000-XMLI", MLBAT = "FR0010436170-XMLI",
                           MLBBO = "CH1148983609-XMLI", MLBIO = "ES0105719001-XMLI",
                           MLBMD = "CH0451123589-XMLI", MLBON = "BE6333353298-XMLI",
                           MLBSP = "FR0013340973-XMLI", MLCAC = "FR0006205019-XMLI",
                           MLCAN = "ES0105586004-XMLI", MLCFD = "FR0000037871-XMLI",
                           MLCFM = "MC0010000826-XMLI", MLCHE = "FR0010447086-XMLI",
                           MLCIO = "FR0012384907-XMLI", MLCLI = "FR0011092089-XMLI",
                           MLCLP = "FR0010959684-XMLI", MLCMB = "FR0000060428-XMLI",
                           MLCMG = "FR0013406881-XMLI", MLCMI = "FR0010972091-XMLI",
                           MLCNT = "FR0000077828-XMLI", MLCOE = "ES0105660007-XMLI",
                           MLCOR = "FR0010035816-XMLI", MLCOT = "FR0013371507-XMLI",
                           MLCOU = "FR0004998318-XMLI", MLCVG = "FR0000051567-XMLI",
                           MLDAM = "FR0000077885-XMLI", MLDYN = "FR0000052920-XMLI",
                           MLDYX = "LU0881232630-XMLI", MLEAS = "HK0000038783-XMLI",
                           MLEAV = "FR0010536185-XMLI", MLECE = "ES0105553004-XMLI",
                           MLECO = "DE000A0XYM45-XMLI", MLEDR = "FR0007200100-XMLI",
                           MLEDS = "FR0000052755-XMLI", MLEDU = "FR0010439265-XMLI",
                           MLEFA = "FR0010882886-XMLI", MLERH = "ES0105726006-XMLI",
                           MLERO = "FR0010157115-XMLI", MLETA = "FR0010492181-XMLI",
                           MLFDV = "FR0012300424-XMLI", MLFIR = "CH0120879058-XMLI",
                           MLFNP = "FR0013230067-XMLI", MLFSG = "FR0013222346-XMLI",
                           MLFTI = "FR0010823724-XMLI", MLFXO = "FR0010487272-XMLI",
                           MLGAI = "FR0000053415-XMLI", MLGAL = "FR0011100759-XMLI",
                           MLGDI = "LU1840650458-XMLI", MLGEQ = "PTGVE1AE0009-XMLI",
                           MLGLA = "IT0005454167-XMLI", MLGLB = "IT0005454175-XMLI",
                           MLGLW = "IT0005454134-XMLI", MLGRC = "FR0000076960-XMLI",
                           MLHAY = "ES0105498002-XMLI", MLHBB = "FR0000051302-XMLI",
                           MLHBP = "ES0105664009-XMLI", MLHCF = "FR0012336691-XMLI",
                           MLHIN = "FR0006563904-XMLI", MLHK = "DE000A11Q133-XMLI",
                           MLHMC = "FR0006226791-XMLI", MLHOP = "FR0006823092-XMLI",
                           MLHOT = "FR0000053738-XMLI", MLHPE = "FR0010312181-XMLI",
                           MLHYD = "FR0000064735-XMLI", MLHYE = "FR0005843125-XMLI",
                           MLIDS = "FR0000079691-XMLI", MLIFC = "FR0011158823-XMLI",
                           MLIFS = "ES0105590006-XMLI", MLIML = "FR0010086371-XMLI",
                           MLIMP = "FR0000065773-XMLI", MLINM = "ES0105511002-XMLI",
                           MLINT = "ES0105479002-XMLI", MLIOC = "BE6200101556-XMLI",
                           MLIPO = "ES0105417002-XMLI", MLIPP = "FR0006859039-XMLI",
                           MLIRF = "IT0005391161-XMLI", MLISP = "ES0105473005-XMLI",
                           MLITN = "IT0005336521-XMLI", MLJ21 = "ES0105636007-XMLI",
                           MLJDL = "ES0105550000-XMLI", MLJSA = "FR0010680033-XMLI",
                           MLLAB = "IT0005324105-XMLI", MLLCB = "FR00140007I9-XMLI",
                           MLLOI = "FR0004155208-XMLI", MLMAB = "FR0000061657-XMLI",
                           MLMAD = "FR0010328302-XMLI", MLMAQ = "ES0105447009-XMLI",
                           MLMAT = "CI0000000832-XMLI", MLMCE = "IT0004844848-XMLI",
                           MLMDV = "FR0014006PT9-XMLI", MLMFI = "BE6252013725-XMLI",
                           MLMGL = "FR0010827741-XMLI", MLMIB = "BE0974328602-XMLI",
                           MLMIV = "ES0105559001-XMLI", MLMTP = "ES0105549002-XMLI",
                           MLMUT = "ES0105697009-XMLI", MLNDG = "ES0105537007-XMLI",
                           MLNMA = "FR0010500363-XMLI", MLNOV = "FR0000185464-XMLI",
                           MLOCT = "FR0013310281-XMLI", MLOKP = "ES0105698007-XMLI",
                           MLONE = "FR0010106039-XMLI", MLONL = "FR0004174712-XMLI",
                           MLORB = "ES0105490009-XMLI", MLORQ = "ES0105534004-XMLI",
                           MLPAC = "FR0000077992-XMLI", MLPER = "ES0105612008-XMLI",
                           MLPET = "NL0015001HZ9-XMLI", MLPHO = "BE0948608451-XMLI",
                           MLPHW = "FR0000185480-XMLI", MLPLC = "FR0000030769-XMLI",
                           MLPRI = "FR0000061376-XMLI", MLPRX = "NL0012650535-XMLI",
                           MLPVG = "FR0010529719-XMLI", MLSAG = "ES0105651006-XMLI",
                           MLSCI = "ES0105592002-XMLI", MLSDN = "FR0013155975-XMLI",
                           MLSEQ = "NL0010623518-XMLI", MLSGT = "FR001400LBS6-XMLI",
                           MLSIL = "FR0010679365-XMLI", MLSML = "FR0011131788-XMLI",
                           MLSMP = "IT0005072811-XMLI", MLSRP = "FR0000054371-XMLI",
                           MLSTR = "FR0000063976-XMLI", MLSUM = "GB00B8GJBS16-XMLI",
                           MLTRA = "FR0000031866-XMLI", MLTRO = "FR0000031106-XMLI",
                           MLUAV = "ES0105639001-XMLI", MLUMG = "FR0011776889-XMLI",
                           MLVAZ = "FR0014007T10-XMLI", MLVER = "FR0006174496-XMLI",
                           MLVIE = "FR0014003FN0-XMLI", MLVIN = "FR0011605617-XMLI",
                           MLVIR = "ES0105704003-XMLI", MLVRE = "ES0105623005-XMLI",
                           MLVRF = "ES0105492005-XMLI", MLVST = "FR0011076595-XMLI",
                           MLVSY = "FR0014003I41-XMLI", MLWEA = "FR0010688465-XMLI",
                           MLWEL = "FR0004152700-XMLI", MLWEY = "FR0010768770-XMLI",
                           MLWIZ = "FR00140047H7-XMLI", MLWRC = "ES0105399002-XMLI",
                           MLZAM = "ZM0000000037-XMLI", MMB = "FR0000130213-XPAR", MMT = "FR0000053225-XPAR",
                           MONT = "BE0003853703-XBRU", MRM = "FR00140085W6-XPAR", MRN = "FR0000039620-XPAR",
                           MT = "LU1598757687-XAMS", MTU = "FR0000038606-XPAR", NACON = "FR0013482791-XPAR",
                           NANO = "FR0011341205-XPAR", NEOEN = "FR0011675362-XPAR",
                           NEX = "FR0000044448-XPAR", NHOA = "FR0012650166-XPAR", NK = "FR0000120859-XPAR",
                           NOKIA = "FI0009000681-XPAR", NR21 = "FR0014001PV6-XPAR",
                           NRG = "FR0000121691-XPAR", NRO = "FR0004050250-XPAR", NXI = "FR0010112524-XPAR",
                           ODET = "FR0000062234-XPAR", OLG = "FR0010428771-XPAR", OR = "FR0000120321-XPAR",
                           ORA = "FR0000133308-XPAR", ORAP = "FR0000075392-XPAR", OREGE = "FR0010609206-XPAR",
                           ORIA = "FR0000060535-XPAR", ORP = "FR0000184798-XPAR", ORPDS = "FR001400MXO7-XPAR",
                           OSE = "FR0012127173-XPAR", OVH = "FR0014005HJ9-XPAR", PAR = "FR0010263202-XPAR",
                           PARP = "FR0012612646-XPAR", PARRO = "FR0004038263-XPAR",
                           PAT = "FR0011027135-XPAR", PATBS = "FR0011844067-ALXP", PERR = "FR0000061459-XPAR",
                           PEUG = "FR0000064784-XPAR", PHXM = "FR001400K4B1-XPAR", PIG = "FR0000066755-XPAR",
                           POM = "FR0000124570-XPAR", POXEL = "FR0012432516-XPAR", PRC = "FR0000074783-XPAR",
                           PROAC = "FR0004052561-XPAR", PROBT = "FR0013398617-ALXP",
                           PSAT = "FR0000038465-XPAR", PUB = "FR0000130577-XPAR", PVL = "FR0013252186-XPAR",
                           PWG = "FR0012613610-XPAR", QDT = "FR0000120560-XPAR", RAL = "FR0000060618-XPAR",
                           RBO = "FR0013344173-XPAR", RBT = "FR0000039091-XPAR", RCO = "FR0000130395-XPAR",
                           RF = "FR0000121121-XPAR", RI = "FR0000120693-XPAR", RMS = "FR0000052292-XPAR",
                           RNO = "FR0000131906-XPAR", ROCBT = "FR0013477593-ALXP", RUI = "FR0013269123-XPAR",
                           RXL = "FR0010451203-XPAR", S30 = "FR0013379484-XPAR", SABE = "FR0000060121-XPAR",
                           SACI = "FR0000061418-XPAR", SAF = "FR0000073272-XPAR", SAMS = "FR0000060071-XPAR",
                           SAN = "FR0000120578-XPAR", SANNV = "FR001400M6Z3-XPAR", SAVE = "FR0000120107-XPAR",
                           SBT = "FR0000052680-XPAR", SCHP = "FR0000039109-XPAR", SCR = "FR0010411983-XPAR",
                           SDG = "FR0000032658-XPAR", SEC = "FR0000078321-XPAR", SEFER = "FR0011950682-XPAR",
                           SELER = "FR0004175842-XPAR", SESG = "LU0088087324-XPAR",
                           SFCA = "FR0010209809-XPAR", SFPI = "FR0004155000-XPAR", SGO = "FR0000125007-XPAR",
                           SGONV = "FR001400M7D8-XPAR", SGRO = "GB00B5ZN1N88-XPAR",
                           SIGHT = "FR0013183985-XPAR", SII = "FR0000074122-XPAR", SK = "FR0000121709-XPAR",
                           SLB = "AN8068571086-XPAR", SLCO = "FR0000065492-XPAR", SMCP = "FR0013214145-XPAR",
                           SMLBS = "FR0010649228-XMLI", SOI = "FR0013227113-XPAR", SOLB = "BE0003470755-XBRU",
                           SOP = "FR0000050809-XPAR", SPEL = "FR0000053944-XPAR", SPIE = "FR0012757854-XPAR",
                           SQI = "FR0011289040-XPAR", SRP = "FR0013006558-XPAR", STF = "FR0000064271-XPAR",
                           STLAP = "NL00150001Q9-XPAR", STMPA = "NL0000226223-XPAR",
                           STRBS = "FR001400DGA0-ALXP", STWBS = "FR001400DG11-ALXP",
                           SU = "FR0000121972-XPAR", SW = "FR0000121220-XPAR", SWP = "FR0004180578-XPAR",
                           SYENS = "BE0974464977-XBRU", TAYN = "FR0000063307-XPAR",
                           TCHCS = "FR001400I939-XPAR", TE = "NL0014559478-XPAR", TEP = "FR0000051807-XPAR",
                           TERBS = "FR0014000TB2-XPAR", TFF = "FR0013295789-XPAR", TFI = "FR0000054900-XPAR",
                           TFINV = "FR001400M3E5-XPAR", THEP = "FR0013333432-XPAR",
                           TIPI = "FR0000066482-XPAR", TITC = "BE0974338700-XBRU", TKO = "FR0013230612-XPAR",
                           TKTT = "FR0004188670-XPAR", TMBSY = "NL0015001SS1-ALXP",
                           TMBSZ = "NL0015001SR3-ALXP", TNG = "FR0005175080-XPAR", TOUP = "FR0000033003-XPAR",
                           TRACT = "FR001400BMH7-XPAR", TRI = "FR0005691656-XPAR", TTE = "FR0000120271-XPAR",
                           TVRB = "CH0008175645-XPAR", UBI = "FR0000054470-XPAR", UNBL = "FR0000054215-XPAR",
                           URW = "FR0013326246-XPAR", VAC = "FR0000073041-XPAR", VACBS = "FR001400B4H9-XPAR",
                           VACBT = "FR001400B4G1-XPAR", VANBS = "FR0013526225-XPAR",
                           VANTI = "FR0013505062-XPAR", VCT = "FR0000031775-XPAR", VETO = "FR0004186856-XPAR",
                           VIE = "FR0000124141-XPAR", VIL = "FR0000050049-XPAR", VIRP = "FR0000031577-XPAR",
                           VIV = "FR0000127771-XPAR", VIVBS = "FR0014003O76-ALXP", VK = "FR0013506730-XPAR",
                           VKBS = "FR00140030K7-XPAR", VLA = "FR0004056851-XPAR", VLTSA = "FR0011995588-XPAR",
                           VMX = "FR0010291245-XPAR", VRAP = "FR0000062796-XPAR", VRLA = "FR0013447729-XPAR",
                           VTR = "FR0010309096-XPAR", VU = "FR0010282822-XPAR", WAGA = "FR0012532810-XPAR",
                           WAVE = "FR0013357621-XPAR", WLN = "FR0011981968-XPAR", XFAB = "BE0974310428-XPAR",
                           XIL = "FR0004034072-XPAR")

  if (ticker %in% names(EN_paris_ticker)) {
    return(EN_paris_ticker[[ticker]])
  } else {
    return("Ticker not found")
  }
}

