# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ #
#                                 Ease CLASS                                   #
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ #

## Functions ----

#' Stop condition reached
#'
#' Generation of the vector describing for each simulation which stop
#' condition(s) have been validated or if the simulation has not been stopped
#' (unstopped).
#'
#' @param easeObj an \code{Ease} object
#' @param addAllStop logical indicating whether all stop modalities should be
#' taken into account (if they have not been reached by any of the simulations,
#' they will be in the levels of the output factor vector)
#'
#' @return A vector of factors
#'
#' @author Ehouarn Le Faou
#'
stopVect <- function(easeObj, addAllStop = TRUE) {
  nbStop <- length(easeObj@stopCondition) + 1
  namesStop <- colnames(easeObj@results$stopCondition)
  stops <- as.factor(apply(
    easeObj@results$stopCondition, 1,
    function(x) {
      Reduce(
        function(x, y) paste(x, "&", y), namesStop[x]
      )
    }
  ))
  if (addAllStop) levels(stops) <- union(levels(stops), namesStop)
  return(stops)
}



#' Proportion of stop condition reached
#'
#' Data frame summarizing for each stop mode of a simulation (stop condition(s)
#' and reaching the threshold) the proportion of simulations that have reached
#' them.
#'
#' @param easeObj an \code{Ease} object
#'
#' @return A data.frame with two columns and as many rows as there are
#' levels in the vector generated by the stopVect function. The first column
#' corresponds to the different levels mentioned above and the second to the
#' proportions of all the simulations that led to these types of simulation
#' stops.
#'
#' @author Ehouarn Le Faou
#'
stopProportions <- function(easeObj) {
  stops <- stopVect(easeObj)
  tabStop <- table(stops) / length(stops)
  df <- data.frame(
    finish = levels(stops),
    prop = as.vector(tabStop)
  )
  return(df)
}


#' Average number of generations to reach stop conditions
#'
#' Data frame summarizing for each stop mode of a simulation (stop condition(s)
#' and reaching the threshold) the average number of generations to reach stop
#' conditions
#'
#' @param easeObj an \code{Ease} object
#'
#' @return A data.frame with two columns and as many rows as there are
#' levels in the vector generated by the stopVect function. The first column
#' corresponds to the different levels mentioned above and the second to the
#' verage number of generations to reach stop conditions calculated over all
#' simulations.
#'
#' @author Ehouarn Le Faou
#'
stopGenerations <- function(easeObj) {
  stops <- stopVect(easeObj)
  stopsDf <- t(sapply(
    stops,
    function(x) {
      y <- as.numeric(x == levels(stops))
      y[which(y == 0)] <- NA
      y
    }
  ))
  if (length(levels(stops)) == 1) {
    stopsDf <- t(stopsDf)
  }
  colnames(stopsDf) <- levels(stops)
  stopsDf <- as.data.frame(stopsDf)
  stopsDf <-
    genStopsDf <- as.data.frame(lapply(
      stopsDf,
      function(k) {
        k * easeObj@results$gen
      }
    ))
  df <- data.frame(
    finish = levels(stops),
    mean = sapply(genStopsDf, mean, na.rm = TRUE)
  )
  return(df)
}

## Validity check ----

#' The validity check for the \code{Ease} class
#'
#' @param object \code{Ease} object
#'
#' @return A logical corresponding to whether the object is a correct
#' \code{Ease} object.
#'
#' @author Ehouarn Le Faou
#'
check.ease <- function(object) {
  if (object@N < 0 | object@N != as.integer(object@N)) {
    stop("The population size must be a positive integer.")
  }
  if (object@threshold < 0 | object@threshold != as.integer(object@threshold)) {
    stop("The threshold must be a positive integer.")
  }
  if (object@selfRate < 0 | object@selfRate > 1) {
    stop("The selfing rate should be between 0 and 1.")
  }

  return(TRUE)
}

## Class definition ----

#' \code{Ease} class
#'
#' The \code{Ease} class is used to manage the simulations by handling the
#' objects needed to build the model. Thus to build an object of class
#' \code{Ease}, it is necessary to have defined an object \code{Genome},
#' as well as an object \code{MutationMatrix} and an object \code{Selection}
#' (even if it is neutral, see \link[Ease]{setSelectNeutral}).
#'
#' @slot N the population size
#' @slot threshold the maximum number of generations
#' @slot dioecy logical indicating whether the simulated population is
#' dioecious or hermaphroditic
#' @slot selfRate the selfing rate
#' @slot genome a \code{Genome} object
#' @slot mutMat a \code{MutationMatrix} object
#' @slot recMat a two-by-two recombination rate vector
#' @slot meiosisMat a meiosis matrix
#' @slot haploCrossMat an haplotype crossing matrix
#' @slot haploCrossMatNamed an haplotype crossing matrix with names of
#' genotypes instead of their indices
#' @slot gametogenesisMat a gametogenesis matrix
#' @slot alleleFreqMat a matrix for calculating allelic frequencies
#' @slot initGenoFreq A row matrix of the size of the genotype number
#' describing the initial allele frequencies common to all simulations
#' @slot stopCondition list of vectors that each describe the alleles that
#' must be fixed to define a stop condition. Each of these stop conditions
#' will therefore be associated with a stop condition
#' @slot IDstopCondition IDs of stop conditions
#' @slot selection a \code{Selection} object
#' @slot nsim the number of simulation to perform
#' @slot results a list of data.frames that describe the results of the
#' simulations
#' @slot records a list of the records of the simulations
#'
#' @author Ehouarn Le Faou
#'
#' @export
setClass("Ease",
  representation(
    N = "numeric",
    threshold = "numeric",
    dioecy = "logical",
    selfRate = "numeric",
    genome = "Genome",
    mutMat = "MutationMatrix",
    recMat = "matrix",
    meiosisMat = "matrix",
    haploCrossMat = "matrix",
    haploCrossMatNamed = "data.frame",
    gametogenesisMat = "matrix",
    alleleFreqMat = "matrix",
    initGenoFreq = "matrix",
    stopCondition = "list",
    IDstopCondition = "character",
    selection = "Selection",
    nsim = "numeric",
    results = "list",
    records = "list"
  ),
  validity = check.ease
)

## Initialize method ----

#' Initialize method for the \code{Ease} class
#'
#' @param .Object a \code{Ease} object
#' @param N the population size
#' @param threshold the maximum number of generations
#' @param dioecy logical indicating whether the simulated population is
#' dioecious or hermaphroditic
#' @param mutMatrixObj a \code{MutationMatrix} object
#' @param genomeObj a \code{Genome} object
#' @param selectionObj a \code{Selection} object
#' @param stopCondition list of vectors that each describe the alleles that
#' must be fixed to define a stop condition. Each of these stop conditions
#' will therefore be associated with a stop condition
#' @param selfRate the selfing rate
#' @param initGenoFreq A vector of the size of the genotype number
#' describing the initial allele frequencies common to all simulations
#'
#' @return An \code{Ease} object
#'
#' @author Ehouarn Le Faou
#'
setMethod("initialize", "Ease", function(.Object, N, threshold, dioecy,
                                         mutMatrixObj, genomeObj,
                                         selectionObj, stopCondition,
                                         selfRate, initGenoFreq) {

  # Population parameters
  .Object@N <- N
  .Object@threshold <- threshold
  .Object@dioecy <- dioecy
  .Object@selfRate <- selfRate

  # Genome
  .Object@genome <- genomeObj

  # Matrices
  .Object@mutMat <- mutMatrixObj
  .Object@recMat <- recombinationMatrix(genomeObj)
  .Object@meiosisMat <- meiosisMatrix(genomeObj)
  .Object@haploCrossMat <- haploCrossMatrix(genomeObj)

  haploCrossMatNamed <- .Object@haploCrossMat
  haploCrossMatNamed <-
    apply(.Object@haploCrossMat, 2, function(x) {
      .Object@genome@IDgenotypes[x]
    })
  haploCrossMatNamed <- as.data.frame(haploCrossMatNamed, 2, as.factor)
  rownames(haploCrossMatNamed) <- rownames(.Object@haploCrossMat)
  .Object@haploCrossMatNamed <- haploCrossMatNamed

  .Object@gametogenesisMat <- .Object@recMat %*% .Object@meiosisMat %*%
    .Object@mutMat@mutationMatrix
  .Object@alleleFreqMat <- alleleFreqMatGeneration(genomeObj)

  # Simulation vectors
  if (identical(initGenoFreq, matrix())) {
    initGenoFreq <- matrix(c(1, rep(0, .Object@genome@nbGeno - 1)), 1)
  } else {
    if (length(initGenoFreq) != .Object@genome@nbGeno) {
      stop(paste0(
        "The vector of initial genotype frequencies must be the ",
        "same length as the number of genotypes in the input ",
        "genome (", .Object@genome@nbGeno, ")."
      ))
    }
    if (sum(initGenoFreq) != 1) {
      stop(paste(
        "The sum of the initial genotype frequencies must be",
        "equal to 1."
      ))
    }
    initGenoFreq <- t(as.matrix(initGenoFreq))
  }
  .Object@initGenoFreq <- initGenoFreq
  colnames(.Object@initGenoFreq) <- .Object@genome@IDgenotypes

  # Stop condition -> THE ALLELE NAMES MUST ALL BE DIFFERENT
  if (!identical(stopCondition, list())) {
    IDstopCondition <- names(stopCondition)
    if (identical(IDstopCondition, NULL)) {
      IDstopCondition <- paste0("stop", 1:length(stopCondition))
    }
    .Object@IDstopCondition <- IDstopCondition
    .Object@stopCondition <- rep(
      list(rep(NA, length(genomeObj@alleles))),
      length(stopCondition)
    )
    for (i in 1:length(stopCondition)) {
      .Object@stopCondition[[i]][sapply(stopCondition[[i]], function(x) {
        which(genomeObj@alleles == x)
      })] <- 1
    }
    names(.Object@stopCondition) <- IDstopCondition
  } else {
    .Object@stopCondition <- list()
  }

  # Selection object
  .Object@selection <- selectionObj

  # Warnings
  if (dioecy) {
    if (!is.na(selfRate)) {
      warning("The selfing rate in dioecy is necessarily 0.")
    }
    .Object@selfRate <- 0
  }
  if (!dioecy & is.na(selfRate)) {
    warning("The selfing rate has been set to 0.")
    .Object@selfRate <- 0
  }

  validObject(.Object)

  return(.Object)
})


## Simulate method ----

#' Simulate method for the \code{Ease} class
#'
#' Performing simulations of an Ease object. The returned object is the same
#' Ease object completed with the results and records if they have been
#' activated.
#'
#' @param object a \code{Ease} object
#' @param nsim the number of simulation to perform
#' @param seed the probabilist seed to be fixed (allows exact reproduction of
#' results)
#' @param includefreqGeno a logical indicating whether to include genotype
#' frequencies in the results
#' @param recording a logical indicating whether to record all mutations, i.e.
#' to record allelic and genotypic frequencies along the simulations
#' @param recordGenGap the number of generations between two records during
#' simulation, if the record parameter is TRUE. Whatever the value of this
#' parameter, both the first and the last generation will be included in
#' the record
#' @param drift a logical indicating whether genetic drift should be
#' considered (i.e. whether deterministic simulations are performed or not)
#' @param includeParams a logical indicating whether the parameters should be
#' included in the result data.frame (can be useful when compiling multiple
#' result tables)
#' @param includeFitness a logical indicating whether the mean fitness should
#' be included in the result data.frame (can be useful when compiling multiple
#' result tables)
#' @param verbose logical determining if the progress of the simulations should
#' be displayed or not (useful in case of many simulations)
#'
#' @return An \code{Ease} object from which we can now extract the results
#' (or the records if recording = TRUE) with the getResults and getRecords
#' functions.
#'
#' @author Ehouarn Le Faou
#'
#' @export
#'
setMethod("simulate", "Ease", function(object, nsim = 1, seed = NULL,
                                       includefreqGeno = TRUE,
                                       recording = FALSE,
                                       recordGenGap = 1, drift = TRUE,
                                       includeParams = TRUE,
                                       includeFitness = TRUE,
                                       verbose = FALSE) {
  # Warnings
  if (!drift & nsim > 1) {
    warning(paste(
      "It is useless to repeat simulations without the drift",
      "(they will all be identical). The number of simulations",
      "has therefore been fixed at 1."
    ))
    nsim <- 1
  }

  if (verbose) {
    message("-=-=-=-=-=-=-=-=-=-=-= MODEL SIMULATION =-=-=-=-=-=-=-=-=-=-=-")
    message("Model simulation:")
    message(paste0(
      "Population of ", object@N, " ",
      c("hermaphroditic", "dioecious")[as.integer(object@dioecy) + 1],
      " individuals"
    ))
    if (!object@dioecy) {
      tbp <- paste0("with a ", round(object@selfRate * 100, 2))
      if (tbp == object@selfRate * 100) {
        message(paste0(tbp, "% selfing rate"))
      } else {
        message(paste0("~", tbp, "% selfing rate"))
      }
    }
    message(paste0("Threshold: ", object@threshold))
    message("..............................................................")
  }
  set.seed(seed)
  res <- SIMULATION_MULTIPLE(
    nsim, recording, recordGenGap, drift,
    object@genome@nbHaplo,
    object@genome@nbGeno,
    object@genome@nbAlleles,
    object@initGenoFreq,
    object@gametogenesisMat,
    object@N,
    object@threshold,
    object@dioecy,
    object@selfRate,
    object@stopCondition,
    object@haploCrossMat,
    object@alleleFreqMat,
    object@selection@femgamFit,
    object@selection@malegamFit,
    object@selection@femindFit,
    object@selection@maleindFit,
    object@selection@indFit,
    object@selection@femProdFit,
    object@selection@maleProdFit,
    verbose
  )
  if (verbose) {
    message("''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''")
    message("Done.")
    message("=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-")
    message()
  }

  # Type modifications
  res$stopCondition <- matrix(as.logical(res$stopCondition),
    ncol = ncol(res$stopCondition)
  )

  # Outputs from the simulations
  if (!object@dioecy) {
    mats <- res[which(names(res) %in%
      c("genoFreq", "allelesFreq", "gen", "stopCondition"))]
    dfs <- lapply(mats, as.data.frame)
    if (includeFitness) {
      dfs$meanFitness <- data.frame(
        meanIndFit = apply(
          dfs$genoFreq, 1,
          function(x) sum(x * object@selection@indFit)
        ),
        meanFemProdFit = apply(
          dfs$genoFreq, 1,
          function(x) sum(x * object@selection@femProdFit)
        ),
        meanMaleProdFit = apply(
          dfs$genoFreq, 1,
          function(x) sum(x * object@selection@maleProdFit)
        )
      )
    }
  } else {
    mats <- res[which(names(res) %in%
      c(
        "genoFreqFemale", "genoFreqMale", "allelesFreq",
        "gen", "stopCondition"
      ))]
    dfs <- lapply(mats, as.data.frame)
    if (includeFitness) {
      dfs$meanFitness <- data.frame(
        meanFemFit = apply(
          dfs$genoFreqFemale, 1,
          function(x) sum(x * object@selection@femindFit)
        ),
        meanMaleFit = apply(
          dfs$genoFreqMale, 1,
          function(x) sum(x * object@selection@maleindFit)
        ),
        meanFemProdFit = apply(
          dfs$genoFreqFemale, 1,
          function(x) sum(x * object@selection@femProdFit)
        ),
        meanMaleProdFit = apply(
          dfs$genoFreqMale, 1,
          function(x) sum(x * object@selection@maleProdFit)
        )
      )
    }
  }
  if (!identical(object@stopCondition, list())) {
    unstopped <- !apply(dfs$stopCondition, 1, any)
    dfs$stopCondition$unstopped <- unstopped
  } else {
    dfs$stopCondition <- data.frame(unstopped = rep(TRUE, nsim))
  }

  # Records of the simulations
  if (!object@dioecy) {
    records <- res[which(names(res) %in%
      c(
        "moniFreqGeno", "moniFreqAlleles",
        "moniGenerations"
      ))]
  } else {
    records <- res[which(names(res) %in%
      c(
        "moniFreqGenoFemale", "moniFreqGenoMale",
        "moniFreqAlleles", "moniGenerations"
      ))]
  }

  # Colnames
  if (!object@dioecy) {
    names(dfs$genoFreq) <- object@genome@IDgenotypes
  } else {
    names(dfs$genoFreqFemale) <- paste0(object@genome@IDgenotypes, "(fem)")
    names(dfs$genoFreqMale) <- paste0(object@genome@IDgenotypes, "(mal)")
  }
  colnames(dfs$allelesFreq) <- object@genome@alleles
  colnames(dfs$gen) <- "genStop"
  colnames(dfs$stopCondition) <- c(
    object@IDstopCondition,
    "unstopped"
  )

  # Excluding genotype frequencies if requested
  if (!includefreqGeno) {
    if (!object@dioecy) {
      dfs <- dfs[-which(names(dfs) == "genoFreq")]
    } else {
      dfs <- dfs[-which(names(dfs) == "genoFreqMale")]
      dfs <- dfs[-which(names(dfs) == "genoFreqFemale")]
    }
  }

  # Including (or not) parameters
  if (includeParams) {
    parameters <- data.frame(
      N = object@N,
      threshold = object@threshold,
      dioecy = factor(c("herma.", "dioecy")[as.numeric(object@dioecy) + 1],
        levels = as.factor(c("herma.", "dioecy"))
      ),
      selfRate = object@selfRate
    )

    dfs <- c(parameters = list(parameters), dfs)
  }

  # Recording colnames
  if (recording) {
    listRecords <- list()
    for (i in 1:nsim) {
      freqAlleles <- records$moniFreqAlleles[[i]]
      genRecord <- data.frame(gen = records$moniGenerations[[i]])

      colnames(freqAlleles) <- object@genome@alleles
      freqAlleles <- as.data.frame(freqAlleles)

      if (!object@dioecy) {
        freqGeno <- records$moniFreqGeno[[i]]
        colnames(freqGeno) <- object@genome@IDgenotypes
        freqGeno <- as.data.frame(freqGeno)
        meanFitness <- data.frame(
          meanIndFit = apply(
            freqGeno, 1,
            function(x) sum(x * object@selection@indFit)
          ),
          meanFemProdFit = apply(
            freqGeno, 1,
            function(x) sum(x * object@selection@femProdFit)
          ),
          meanMaleProdFit = apply(
            freqGeno, 1,
            function(x) sum(x * object@selection@maleProdFit)
          )
        )
      } else {
        freqGenoFemale <- records$moniFreqGenoFemale[[i]]
        colnames(freqGenoFemale) <- paste0(object@genome@IDgenotypes, "(fem)")
        freqGenoMale <- records$moniFreqGenoMale[[i]]
        colnames(freqGenoMale) <- paste0(object@genome@IDgenotypes, "(mal)")
        freqGeno <- as.data.frame(cbind(freqGenoFemale, freqGenoMale))
        meanFitness <- data.frame(
          meanFemFit = apply(
            freqGenoFemale, 1,
            function(x) sum(x * object@selection@femindFit)
          ),
          meanMaleFit = apply(
            freqGenoMale, 1,
            function(x) sum(x * object@selection@maleindFit)
          ),
          meanFemProdFit = apply(
            freqGenoFemale, 1,
            function(x) sum(x * object@selection@femProdFit)
          ),
          meanMaleProdFit = apply(
            freqGenoMale, 1,
            function(x) sum(x * object@selection@maleProdFit)
          )
        )
      }
      listRecords <- c(listRecords, list(cbind(
        genRecord, freqGeno,
        freqAlleles, meanFitness
      )))
    }
  }

  object@nsim <- nsim
  object@results <- dfs
  if (recording) {
    object@records <- listRecords
  } else {
    object@records <- list()
  }
  return(object)
})

## Summary method ----

#' Summary method for the \code{Ease} class
#'
#' Generattion of a small summary of the simulation results.
#'
#' @param object a \code{Ease} object
#' @param ... Ignored.
#'
#' @return Some statistics summarising the results obtained by simulating:
#' the proportions of simulations having reached each of the defined stop
#' conditions or having reached the threshold and the average duration of
#' the simulations according to these stop conditions.
#'
#' @export
setMethod("summary", "Ease", function(object, ...) {
  catn("-=-=-=- EASE OJBECT SUMMARY -=-=-=-")
  catn(paste0("# Size: ", object@N))
  catn(paste0(
    "# System: ",
    c("hermaphodism", "dioecy")[as.integer(object@dioecy) + 1]
  ))
  if (!object@dioecy) {
    cat("# Selfing rate: ")
    tbp <- round(object@selfRate * 100, 2)
    if (tbp == object@selfRate * 100) {
      catn(paste0(tbp, "%"))
    } else {
      catn(paste0("~", tbp, "%"))
    }
  }

  if (identical(object@results, list())) {
    catn("-=-=-=-=-=-=-=-=-=-=--=-=-=-=-=-=-=")
    catn("For a summary of the results, use 'simulate' to generate them.")
  } else {
    catn()
    catn("-=-=-=-=-=-=- RESULTS -=-=-=-=-=-=-")
    df <- stopProportions(object)
    vectProp <- df$prop
    names(vectProp) <- df$finish

    df <- stopGenerations(object)
    vectGen <- df$mean
    names(vectGen) <- df$finish

    catn("# Outcome distribution of simuls.")
    print(vectProp)
    catn()
    catn("# Mean stop generation")
    print(vectGen)
    catn("-=-=-=-=-=-=-=-=-=-=--=-=-=-=-=-=-=")
  }
})

## Show method ----

#' Show method for the \code{Ease} class
#'
#' @param object a \code{Ease} object
#'
#' @return No return value, only a display.
#'
#' @author Ehouarn Le Faou
#'
#' @export
setMethod("show", "Ease", function(object) {
  catn("-=-=- EXPLICIT POPULATION GENETICS MODEL -=-=-")
  catn("Settings:")
  cat(paste0(
    " # ", object@N, " ",
    c("hermaphroditic", "dioecious")[as.integer(object@dioecy) + 1],
    " individuals"
  ))
  if (!object@dioecy) {
    cat("\n   with a ")
    tbp <- round(object@selfRate * 100, 2)
    if (tbp == object@selfRate * 100) {
      catn(paste0(tbp, "% selfing rate."))
    } else {
      catn(paste0("~", tbp, "% selfing rate."))
    }
  } else {
    catn()
  }
  catn(" # Threshold:", object@threshold, "generations.")

  nbAlleleHL <- sapply(object@genome@listHapLoci, length)
  nbAlleleDL <- sapply(object@genome@listDipLoci, length)
  catn("Genome:")
  cat(" # ", object@genome@nbHL)
  if (object@genome@nbHL == 1) {
    catn(" haploid locus, with ", nbAlleleHL, " allele(s)", sep = "")
  } else {
    catn(" haploid loci, with respectively ",
      Reduce(function(x, y) {
        paste0(x, ", ", y)
      }, nbAlleleHL[-object@genome@nbHL]),
      " and ", nbAlleleHL[object@genome@nbHL], " allele(s)",
      sep = ""
    )
  }
  cat(" # ", object@genome@nbDL)
  if (object@genome@nbDL == 1) {
    catn(" diploid locus, with ", nbAlleleDL, " allele(s)", sep = "")
  } else {
    catn(" diploid loci, with respectively ",
      Reduce(function(x, y) {
        paste0(x, ", ", y)
      }, nbAlleleDL[-object@genome@nbDL]),
      " and ", nbAlleleDL[object@genome@nbDL], " allele(s)",
      sep = ""
    )
  }
  catn(" # ", object@genome@nbHaplo, "haplotypes")
  catn(" # ", object@genome@nbGeno, "genotypes")

  catn("Selection:")
  catn(" #  On individuals:", c("NO", "YES")[
    as.integer(object@selection@sOnInds) + 1
  ])
  catn(" #  On gametes:", c("NO", "YES")[
    as.integer(object@selection@sOnGams) + 1
  ])
  catn(" #  On gamete production:", c("NO", "YES")[
    as.integer(object@selection@sOnGamsProd) + 1
  ])

  catn("Simulation:")
  if (identical(object@results, list())) {
    catn(" #  No simulation carried out yet")
  } else {
    catn(" #  Nb sim.:", object@nsim)
    catn(" #  Recording:", c("NO", "YES")[as.integer(!identical(
      object@records,
      list()
    )) + 1])
  }

  catn("-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=")
})

## Print method ----

#' Print method for the \code{Ease} class
#'
#' @param x a \code{Ease} object
#' @param ... there are no more parameters.
#'
#' @return No return value, only a display.
#'
#' @author Ehouarn Le Faou
#'
#' @export
setMethod("print", "Ease", function(x, ...) {
  catn("-=-=-=-=-=-=- EXPLICIT POPULATION GENETICS MODEL -=-=-=-=-=-=-")
  catn("                         in details")
  catn()
  catn(" #  Population: ")
  catn()
  catn(paste0("   - Size: ", x@N))
  catn(paste0(
    "   - System: ",
    c("hermaphodism", "dioecy")[as.integer(x@dioecy) + 1]
  ))
  if (!x@dioecy) {
    cat("   - Selfing rate: ")
    tbp <- round(x@selfRate * 100, 2)
    if (tbp == x@selfRate * 100) {
      catn(paste0(tbp, "%"))
    } else {
      catn(paste0("~", tbp, "%"))
    }
  }

  catn()
  catn(" #  Simulations: ")
  catn()
  catn(paste0("   - Threshold: ", x@threshold))
  catn("   - Initial genotypes frequency: ")
  tbp <- as.vector(x@initGenoFreq)
  names(tbp) <- x@genome@IDgenotypes
  print(tbp)

  catn()
  catn(" #  Haploptypes: ")
  catn()
  haplo <- x@genome@IDhaplotypes
  names(haplo) <- 1:length(haplo)
  print(haplo)

  catn()
  catn(" #  Genotypes: ")
  catn()
  geno <- x@genome@IDgenotypes
  names(geno) <- 1:length(geno)
  print(geno)

  catn()
  catn(" #  Matrices involved in gametogenesis: ")
  catn()
  catn("   - Mutation matrix: ")
  print(x@mutMat@mutationMatrix)
  catn()
  catn("   - Recombination matrix: ")
  print(x@recMat)
  catn()
  catn("   - Meiosis matrix: ")
  print(x@meiosisMat)
  catn()
  catn("   - Final gametogenesis matrix: ")
  print(x@gametogenesisMat)
  catn()

  catn(" #  Haplotypes crossing matrix: ")
  catn()
  print(x@haploCrossMatNamed)
  catn()

  catn(" #  Allele frequencies from genotype frequencies matrix: ")
  catn()
  print(x@alleleFreqMat)
  catn()

  catn(" #  Allele frequencies from genotype frequencies matrix: ")
  catn()
  print(x@alleleFreqMat)
  catn()

  catn(" #  Stop condition: ")
  catn()
  if (!identical(x@stopCondition, list())) {
    le <- length(x@stopCondition)
    numero <- c(c("1st", "2nd", "3rd"), paste0(min(4, le - 1):max(4, le), "th"))
    tbp <- paste0("   - ", numero[1:le], " condition: ")
    for (i in 1:le) {
      catn(paste0(
        tbp[i],
        "fixation of allele(s) ",
        listing(x@genome@alleles[which(x@stopCondition[[i]] == 1)])
      ))
      catn()
    }
  } else {
    catn("No stop condition defined.")
  }
  catn()

  catn(" #  Selection: ")

  catn()
  if (!x@selection@sOnInds & !x@selection@sOnGams & !x@selection@sOnGamsProd) {
    catn("No selection defined.")
    catn()
  }
  if (x@selection@sOnInds) {
    if (length(x@selection@indFit) > 0) {
      catn("   - On individuals: ")
      tbp <- t(t(x@selection@indFit))
      colnames(tbp) <- "Fitness"
      print(tbp)
      catn()
    } else {
      tbp <- cbind(t(t(x@selection@femindFit)), t(t(x@selection@maleindFit)))
      colnames(tbp) <- c("Female", "Male")
      print(tbp)
      catn()
    }
  }
  if (x@selection@sOnGams) {
    catn("   - On gametes: ")
    tbp <- cbind(t(t(x@selection@femgamFit)), t(t(x@selection@malegamFit)))
    colnames(tbp) <- c("Female gamete", "Male gamete")
    print(tbp)
    catn()
  }
  if (x@selection@sOnGamsProd) {
    catn("   - On gamete production: ")
    tbp <- cbind(t(t(x@selection@femProdFit)), t(t(x@selection@maleProdFit)))
    colnames(tbp) <- c("Female gamete", "Male gamete")
    print(tbp)
    catn()
  }

  catn("-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=")
})

## Plot method ----

#' Plot method for the \code{Ease} class
#'
#' Generation of a some simple plots that summaries the simulation
#' results.
#'
#' @param x a \code{Ease} object
#' @param y Ignored.
#' @param ... Ignored.
#' @param n number of the graph to be displayed (between 1 and 3 if no
#' simulations were recorded, from 1 to 4 otherwise).
#'
#' @return The graphics are displayed one after the other and require user
#' interaction to switch between them. Three graphs concern the results of
#' the simulations: (i) the proportion of stop conditions reached, (ii)
#' the distribution of simulation durations according to the stop conditions
#' reached, and finally (iii) the average time to reach stop conditions. If the
#' simulations were recorded (parameter \code{recording} of the method
#' \code{simulate} for the class {Ease}), an additional graph is available,
#' and represents the evolution of allelic frequencies as a function of
#' generations for each of the simulations. For each simulation, the vector
#' of generations is standardised (by dividing it by its final value) in
#' order to have an overview of the ways in which the simulations ended.
#'
#' @author Ehouarn Le Faou
#'
#' @export
#'
#' @import ggplot2
setMethod("plot", "Ease", function(x, y, ..., n = NULL) {
  easeObj <- x


  recorded <- !identical(easeObj@records, list())
  if (is.null(n)) {
    n <- 1:4
    if (!recorded) {
      n <- 1:3
    }
  }

  if (identical(easeObj@results, list())) {
    stop(paste(
      "No graphics available until you have simulated the",
      "configuration of this object. To do this, use the",
      "'simulate' function."
    ))
  } else {

    # Save of the current theme (it will me restored when all plots have
    # been shown)
    ggThemeSave <- theme_get()
    theme_set(theme_classic() +
      theme(
        panel.grid.major.y = element_line(
          colour = "gray",
          size = 0.5,
          linetype = "solid"
        ),
        panel.grid.major.x = element_blank()
      ))

    if (1 %in% n) {
      # Stop proportions
      df <- stopProportions(easeObj)

      g1 <-
        ggplot(data = df, aes_string(x = "finish", fill = "finish", y = "prop")) +
        geom_bar(stat = "identity", color = "black") +
        scale_fill_brewer(palette = "Blues") +
        ylab("") +
        xlab("") +
        labs(
          title = paste(
            "Distribution of the simulations according to the",
            "way they were stopped"
          ),
          subtitle = paste(
            "(without being stopped, by reaching the",
            "threshold, or by being stopped by a stop",
            "condition)"
          ),
          fill = "Stop conditions"
        )
    } else {
      g1 <- NULL
    }

    if (2 %in% n) {
      # Stop generations
      df <- stopGenerations(easeObj)

      g2 <-
        ggplot(data = df, aes_string(x = "finish", fill = "finish", y = "mean")) +
        geom_bar(stat = "identity", color = "black") +
        scale_fill_brewer(palette = "Blues") +
        ylab("Generations") +
        xlab("") +
        labs(
          title = paste(
            "Mean number of generations before the stop",
            "condition is reached"
          ),
          fill = "Stop conditions"
        )
    } else {
      g2 <- NULL
    }

    if (3 %in% n) {
      df <- cbind(
        easeObj@results$gen,
        data.frame(finish = stopVect(x, addAllStop = FALSE))
      )

      g3 <-
        ggplot(data = df, aes_string(x = "genStop", fill = "finish")) +
        geom_histogram(
          color = "black",
          binwidth = range(df$genStop)[2] / 50
        ) +
        scale_fill_brewer(palette = "Blues") +
        ylab("Generations") +
        xlab("") +
        labs(
          title = paste(
            "Distribution of the number of generations",
            "before the stop condition is reached"
          ),
          fill = "Stop conditions"
        ) +
        facet_grid(finish ~ ., scales = "free") +
        theme(
          strip.background = element_blank(),
          strip.text.y = element_blank()
        )
    } else {
      g3 <- NULL
    }

    if (4 %in% n) {
      if (recorded) {
        idColAlleles <- which(colnames(easeObj@records[[1]]) %in% easeObj@genome@alleles)
        listDfAlleles <- lapply(easeObj@records, function(df) {
          df[, idColAlleles]
        })
        listDf <- lapply(
          as.list(1:easeObj@nsim),
          function(i) {
            data.frame(
              gen = rep(easeObj@records[[i]]$gen, easeObj@genome@nbAlleles),
              freq = unlist(listDfAlleles[[i]], use.names = FALSE),
              allele = rep(
                easeObj@genome@alleles,
                rep(
                  nrow(listDfAlleles[[i]]),
                  easeObj@genome@nbAlleles
                )
              )
            )
          }
        )


        df <- Reduce(rbind, listDf)
        df$id <- rep(1:length(listDf), unlist(lapply(listDf, nrow)))
        df$genMax <- rep(
          unlist(lapply(listDf, function(x) max(x$gen))),
          unlist(lapply(listDf, nrow))
        )
        lociId <- rep(
          names(easeObj@genome@listLoci),
          unlist(lapply(easeObj@genome@listLoci, length))
        )
        df$loci <- sapply(df$allele, function(a) {
          lociId[which(a ==
            easeObj@genome@alleles)]
        })
        df$genStandard <- df$gen / df$genMax

        listPlot <- lapply(
          as.list(names(easeObj@genome@listLoci)),
          function(l) {
            ggplot(
              data = df[which(df$loci == l), ],
              aes_string(
                x = "genStandard", y = "freq",
                group = "id", color = "id"
              )
            ) +
              geom_line() +
              facet_grid(. ~ allele) +
              scale_fill_brewer(palette = "Blues") +
              xlab("Generations (standardised)") +
              labs(color = "Simuls.") +
              ylim(0, 1)
          }
        )

        g4 <- ggarrange(
          plotlist = listPlot,
          labels = names(easeObj@genome@listLoci),
          ncol = 1, nrow = length(easeObj@genome@listLoci)
        )
      } else {
        g4 <- NULL
      }
    } else {
      g4 <- NULL
    }

    gList <- list(g1, g2, g3)
    if (recorded) gList <- c(gList, list(g4))

    if (length(n) == 1) {
      print(gList[[n]])
    } else {
      print(gList[[n[1]]])
      for (i in 2:length(n)) {
        readline("<Press enter to go to the next graph>")
        print(gList[[n[i]]])
      }
    }
    theme_set(ggThemeSave)
  }
})
