\name{plot}
\alias{plot.DTRreg}
\title{Diagnostic Plots for DTR Estimation}
\description{
Diagnostic plots for assessment of treatment, treatment-free and blip models following DTR estimation using DTRreg.
}
\usage{
\method{plot}{DTRreg}(x, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{A model object generated by the function DTRreg.}
  \item{...}{Space for additional arguments (not currently used by DTRreg)}
}
\details{
DTR estimation using G-estimation and dWOLS requires the specification of three models: the treatment, treatment-free and blip.  The treatment model may be assessed via standard diagnostics, whereas the treatment-free and blip models may be simultaneously assessed using diagnostic plots introduced by Rich et al.  The plot() function first presents diagnostic plots that assess the latter, plotting fitted values against residuals and covariates following DTR estimation.  If there is any evidence of a relationship between the variables in these plots, this is evidence that at least one of the blip or treatment-free models is mis-specified.

Following these plots, the plot() function will present standard diagnostic plots for the treatment model.  These are produced directly by the standard plot() command applied to the models that were fit.  For example, if treatment is binary, the resulting plots are the same as those that are generated by the plot() command applied to a glm object for logistic regression.
}

\references{
Chakraborty, B., Moodie, E. E. M. (2013) \emph{Statistical Methods for Dynamic Treatment Regimes}. New York: Springer.

Rich B., Moodie E. E. M., Stephens D. A., Platt R. W. (2010) Model Checking with Residuals for G-estimation of Optimal Dynamic Treatment Regimes. \emph{International Journal of Biostatistics} \bold{6}(2), Article 12.

Robins, J. M. (2004) \emph{Optimal structural nested models for optimal sequential decisions}. In Proceedings of the Second Seattle Symposium on Biostatistics, D. Y. Lin and P. J. Heagerty (eds), 189--326. New York: Springer.

Wallace, M. P., Moodie, E. M. (2015) Doubly-Robust Dynamic Treatment Regimen Estimation Via Weighted Least Squares. \emph{Biometrics} \bold{71}(3), 636--644 (doi:10.1111/biom.12306.)
}
\author{Michael Wallace}

\examples{
##################
# example single run of a 2-stage g-estimation analysis
set.seed(1)
# expit function
expit <- function(x) {1 / (1 + exp(-x))}
# sample size
n <- 10000
# variables (X = patient information, A = treatment)
X1 <- rnorm(n)
A1 <- rbinom(n, 1, expit(X1))
X2 <- rnorm(n)
A2 <- rbinom(n, 1, expit(X2))
# blip functions
gamma1 <- A1 * (1 + X1)
gamma2 <- A2 * (1 + X2)
# observed outcome: treatment-free outcome plus blip functions
Y <- exp(X1) + exp(X2) + gamma1 + gamma2 + rnorm(n)
# models to be passed to DTRreg
# blip model
blip.mod <- list(~X1, ~X2)
# treatment model (correctly specified)
treat.mod <- list(A1~X1, A2~X2)
# treatment-free model (incorrectly specified)
tf.mod <- list(~X1, ~X2)

# perform G-estimation
mod1 <- DTRreg(Y, blip.mod, treat.mod, tf.mod, method = "gest")

# model diagnostics: note treatment-free model is mis-specified
plot(mod1)
##################
}

\keyword{dynamic treatment regimens,adaptive treatment strategies,personalized medicine,g-estimation,dynamic weighted ordinary least squares}
