% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modid.R
\name{modid}
\alias{modid}
\title{D3mirt Model Identification}
\usage{
modid(x, lower = 0.5, upper = 0.1, fac.order = NULL)
}
\arguments{
\item{x}{A data frame with factor loadings}

\item{lower}{The lower bound for the item pool is calculated using the standard deviation of scaled item factor loadings. The default is \code{lower = 0.5}.}

\item{upper}{The upper bound for filtering of absolute sum scores less than or equal to the indicated value. The default is \code{upper = .10}}

\item{fac.order}{Optional. Users can override the automatic sorting of factors by manually indicating factor order with integer values, e.g., \code{c(2,1,3)} to use the second factor (or column) in data frame x first, first factor (or column) in x second, and the third factor (or column) is left untouched.
The default is \code{fac.order = NULL}.}
}
\value{
S3 object of class \code{modid} with lists of items and absolute sum scores, sorted by the latter, and sum of squared factor loadings and frame with raw factor loadings with columns ordered on explained variance (high to low) or according to user settings.
}
\description{
\code{modid()} performs model identification for descriptive multidimensional item response theory (DMIRT) models by indicating what items, from a set or scale, to use to maximize the utility of the DMIRT model.
}
\details{
Before performing DMIRT analysis it is necessary to identify the compensatory model (Reckase, 2009).
For a three-dimensional model, this entails that two items must be chosen.
If improper items are chosen the model will be hard to interpret.
The \code{modid()} function was designed to maximize the former analytically.
Briefly, the function first order factors by the sum of squares, and from this select the strongest loading items that meet the statistical assumptions described above.
This orders the model so that the strongest loading item, from the strongest factor, always aligns perfectly with the x-axis and that the other items follow thereon.

Of the two items that must be chosen for \code{D3mirt}analysis, the first item is constrained not to load on the second and third axes (y and z),
while the second item is only constrained not to load on the third axis (z).
This will identify an orthogonal three-dimensional structure with the first item fixed on the x-axis.
The model identification process is briefly explained below. For further details, please see the package vignette.
\subsection{Step 1: Explore Data Structure}{

To begin the factor structure must be explored with exploratory factor analysis (EFA).
However, because \code{D3mirt} analysis is based on item response theory, and uses the multidimensional two-parameter logistic model or the multidimensional graded response model, it is preferable to use item response theory EFA methods,
such as the EFA option in \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012) with \code{ìtemtype = 'graded'}, or \code{'2PL'} and number of factors set to 3.
Observe that the EFA is only used to find the appropriate items for the model identification and the EFA itself is discarded after this step in the procedure.
}

\subsection{Step 2: Item Selection}{

The \code{modid()} takes in the factor solution from the EFA, assigned to a data frame \emph{x}, and outputs lists (denoted \emph{id1]...idn}) with suggestions of items (\emph{item.1....item.n}) to use for the model identification.
These lists contain one column with the highest factor loadings (equal to or higher than the limit set by the lower bound) from each item on the factor of interest and one column with absolute sum scores for the items calculated based on the remaining factor loadings in the model.
Each list is sorted with the lowest absolute sum score highest up.
Accordingly, the top items in each list are the items that best meet the assumption of orthogonality in the model.
Therefore, for a three-dimensional model, all else equal, the item highest up in the first list should be used to identify the x-axis, and the item highest up in the second list should be used to identify the y-axis, and so on.
}

\subsection{Criteria}{

Model identification items should preferably (a) have an absolute sum score of less than or equal to .10 and (b) have the highest factor loading scores on the factor of interest.
Of these two criteria, (a) should be given the strongest weight in the selection decision.
If these conditions cannot be met, the user is advised to proceed with caution since the loading scores, therefore, imply that an adequate orthogonal structure may not be empirically attainable.
If problems occur, try the following:

\enumerate{
\item Change the rotation method in the EFA, e.g., to change from \emph{oblimin} to \emph{varimax}.
\item Adjust the \code{lower} bound in \code{modid()}.
\item Override factor order with \code{fac.order}.
\item Adjust the \code{upper} bound in \code{modid()}.
}

The latter (point 4) should, however, only be used as a last resort.
This is because the upper bound sets the upper limit for item inclusion.
Adjusting this limit too high means that the necessary statistical requirements are compromised.
The lower limit (point 2), however, only increases the size of the item pool used for the item selection.
It is, therefore, recommended to adjust the lower limit up and down to see if the output differs, and from that make the final decision.

The user also has the option of overriding the automatic sorting of factor order (point 4) with the argument \code{fac.order} (see examples section).
This can be useful when there is only a very small difference between the squared factor loadings that in turn can
causes problems (often only observable at later stages) when trying to find the best items for the model identification.

Note, the \code{modid()} function is not limited to three-dimensional analysis and can be used on any number of factors.
Moreover, although based on suggestions on model identification given by Reckase (2009) for this type of analysis, the function offers some expansions that introduce more precision.
The latter foremost consists in incorporating sum of squares in the item selection process (unless the user has not specified otherwise).
Experience tells that this is good practice that often leads to better results compared to other options.
However, it is important to recognize that the \code{modid()} function only gives suggestions to the model specification, and there could be situations where the researcher should consider other methods.
}
}
\examples{
\donttest{
# Load data
data("anes0809offwaves")
x <- anes0809offwaves
x <- x[,3:22] # Remove columns for age and gender

# Fit a three-factor EFA model with the mirt package
e <- mirt::mirt(x, 3, itemtype = 'graded')

# Assign data frame with factor loadings with oblimin rotation
f <- summary(e, rotate= 'oblimin')
h <- data.frame(f$rotF)

# Call to modid()
modid(h)

# Call to modid with increased lower and higher bound
modid(h, lower = 1, upper = .12 )

# Override factor order by reversing columns in the original data frame
modid(h, fac.order = c(3,2,1))
}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.

Reckase, M. D. (2009). \emph{Multidimensional Item Response Theory}. Springer.

Reckase, M. D., & McKinley, R. L. (1991). The Discriminating Power of Items That Measure More Than One Dimension. \emph{Applied Psychological Measurement, 15}(4), 361-373–373. https://doi-org.ezp.sub.su.se/10.1177/014662169101500407
}
\author{
Erik Forsberg
}
