#' Add a color scale
#'
#' Adds a color scale to plots. Default behavior set for bathymetry. May also be used to 
#' place a \code{\link[graphics]{legend}}.
#' 
#' @param pos character, indicating the vertical position of the color scale (which is always on the 
#' right side of plots). if \code{pos="1/1"}, the color scale will be centered. 
#' if \code{pos="1/2"}, the color scale will in the top half of the plotting regions.
#' if \code{pos="2/2"}, the color scale will in the bottom half of the plotting regions.
#' @param title character, title of the color scale.
#' @param width width of the color scale box, expressed in \% of the width of the plotting region.
#' @param height height of the color scale box, expressed in \% of the height of the plotting region.
#' @param cuts numeric vector of color classes. May be generated via \code{\link{add_col}}.
#' @param cols character vector of color names. May be generated via \code{\link{add_col}}.
#' @param minVal If desired, the color scale may be generated starting from the value \code{minVal}. See examples.
#' @param maxVal If desired, the color scale may be generated up to the value \code{maxVal}. See examples.
#' @param fontsize Size of the text in the color scale.
#' @param offset Controls the horizontal position of the color scale. Increase to distance from the plotting region.
#' @param lwd thickness of lines.
#' @param mode if 'Cscale', the default, the function builds a color scale. if 'Legend', the function
#' gives you the location of a \code{\link[graphics]{legend}}, arguments \code{pos}, \code{offset} and \code{height}
#' may be used for adjustments. See examples.
#' 
#' @seealso 
#' \code{\link{SmallBathy}}, \code{\link{Depth_cuts}}, \code{\link{Depth_cols}}, \code{\link{add_col}},
#' \code{\link{Depth_cuts2}}, \code{\link{Depth_cols2}},
#' \href{http://www.stat.columbia.edu/~tzheng/files/Rcolor.pdf}{R colors}, \code{\link[graphics]{legend}}.
#' 
#' @examples
#' 
#' #Example 1: simple bathymetry plot with color scale
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right)
#' plot(SmallBathy, breaks=Depth_cuts, col=Depth_cols, legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(height=95)
#' par(Mypar)
#' 
#' #' #Example 2: simple bathymetry plot with Fishable Depth range highlight and color scale
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right)
#' plot(SmallBathy, breaks=Depth_cuts2, col=Depth_cols2, legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(height=95,cuts=Depth_cuts2,cols=Depth_cols2)
#' par(Mypar)
#' 
#' #Example 3: Show only values greater than 'minVal'
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right),
#' plot(SmallBathy, breaks=Depth_cuts, col=Depth_cols, legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(minVal=-3200)
#' par(Mypar)
#' 
#' #Example 4: Show only values between 'minVal' and 'maxVal'
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right)
#' plot(SmallBathy, breaks=Depth_cuts, col=Depth_cols, legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(minVal=-3200,maxVal=-400)
#' par(Mypar)
#' 
#' #Example 5: Adding two color scales
#' 
#' #Bathymetry
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right)
#' 
#' plot(SmallBathy,breaks=Depth_cuts,col=Depth_cols,legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(pos='1/2',height=45,maxVal=-1,minVal=-4000,fontsize=0.8)
#' #Some gridded data
#' MyGrid=create_PolyGrids(GridData,dlon=2,dlat=1)
#' Gridcol=add_col(MyGrid$Catch_sum,cuts=10)
#' plot(MyGrid,col=Gridcol$varcol,add=TRUE)
#' #Add color scale using cuts and cols generated by add_col, note the use of 'round'
#' add_Cscale(pos='2/2',height=45,title='Catch (t)',
#'            cuts=round(Gridcol$cuts,1),cols=Gridcol$cols,fontsize=0.8)
#' par(Mypar)
#' 
#' #Example 6: Adding a color scale and a legend
#' 
#' #Create some point data
#' MyPoints=create_Points(PointData)
#' 
#' #Crop the bathymetry to match the extent of MyPoints (extended extent)
#' BathyCr=raster::crop(SmallBathy,raster::extend(raster::extent(MyPoints),100000))
#' Mypar=par(mai=c(0,0,0,1)) #plot margins as c(bottom, left, top, right)
#' plot(BathyCr,breaks=Depth_cuts,col=Depth_cols,legend=FALSE,axes=FALSE,box=FALSE)
#' add_Cscale(pos='1/2',height=45,maxVal=-1,minVal=-4000,fontsize=0.8)
#' 
#' #Plot points with different symbols and colors (see ?points)
#' Psymbols=c(21,22,23,24)
#' Pcolors=c('red','green','blue','yellow')
#' plot(MyPoints[MyPoints$name=='one',],pch=Psymbols[1],bg=Pcolors[1],add=TRUE)
#' plot(MyPoints[MyPoints$name=='two',],pch=Psymbols[2],bg=Pcolors[2],add=TRUE)
#' plot(MyPoints[MyPoints$name=='three',],pch=Psymbols[3],bg=Pcolors[3],add=TRUE)
#' plot(MyPoints[MyPoints$name=='four',],pch=Psymbols[4],bg=Pcolors[4],add=TRUE)
#' 
#' #Add legend with position determined by add_Cscale
#' Loc=add_Cscale(pos='2/2',height=45,mode='Legend')
#' legend(Loc,legend=c('one','two','three','four'),title='Vessel',pch=Psymbols,pt.bg=Pcolors,xpd=TRUE)
#' par(Mypar)
#' 
#' @export

add_Cscale=function(pos='1/1',title='Depth (m)',width=18,height=70,
              cuts=Depth_cuts,cols=Depth_cols,
              minVal=NA,maxVal=NA,fontsize=1,offset=100,lwd=1,mode="Cscale"){
  offset=offset*1000
  #Get plot boundaries
  ls=par("usr")
  xmin=ls[1]
  xmax=ls[2]
  ymin=ls[3]
  ymax=ls[4]
  xdist=xmax-xmin
  ydist=ymax-ymin
  
  #Midpoint
  n=as.numeric(strsplit(pos,'/')[[1]])
  N=n[2]
  n=n[1]
  ymid=seq(ymax,ymin,length.out=2*N+1)[seq(2,n+N,by=2)[n]]
  
  #Overall box
  bxmin=xmax+0.005*xdist+offset
  bxmax=xmax+(width/100)*xdist+offset
  bymin=ymid-(height/200)*ydist
  bymax=ymid+(height/200)*ydist
  
  if(mode=='Legend'){
    out=cbind(x=bxmin,y=bymax)
    return(out)
  }else{
  
  #constrain colors and breaks
  cuts=cuts
  if(is.na(minVal)==FALSE){
    indx=which(cuts>=minVal)
    if(min(indx)>1){indx=c(min(indx)-1,indx)}
    cuts=cuts[indx]
  }
  if(is.na(maxVal)==FALSE){
    indx=which(cuts<=maxVal)
    if(max(indx)<length(cuts)){indx=c(indx,max(indx+1))}
    cuts=cuts[indx]
  }
  cutsTo=cuts
  colsTo=cols
  
  if(any(is.na(c(minVal,maxVal))==FALSE)){
    indx=match(cuts,cutsTo)
    cutsTo=cutsTo[indx]
    colsTo=colsTo[seq(min(indx),max(indx)-1)]
  }
  
  if(all(cutsTo<0)){
    cutsTo=-cutsTo
  }
  
  #plot Overall box
  rect(xleft=bxmin,
       ybottom=bymin,
       xright=bxmax,
       ytop=bymax,xpd=TRUE,lwd=lwd,col='white')
  #Col box
  cxmin=bxmin+0.01*xdist
  cxmax=bxmin+0.05*xdist
  cymin=bymin+0.02*ydist
  cymax=bymax-0.07*ydist
  Ys=seq(cymin,cymax,length.out=length(colsTo)+1)
  rect(xleft=cxmin,
       ybottom=Ys[1:(length(Ys)-1)],
       xright=cxmax,
       ytop=Ys[2:length(Ys)],xpd=TRUE,lwd=0,col=colsTo)
  rect(xleft=cxmin,
       ybottom=cymin,
       xright=cxmax,
       ytop=cymax,xpd=TRUE,lwd=lwd)
  #Ticks
  segments(x0=cxmax,
           y0=Ys,
           x1=cxmax+0.01*xdist,
           y1=Ys,lwd=lwd,xpd=TRUE,lend=1)
  text(cxmax+0.02*xdist,Ys,
       cutsTo,adj=c(0,0.5),xpd=TRUE,cex=fontsize)
  #Title
  text(cxmin,cymax+0.04*ydist,title,
       cex=1.2*fontsize,adj=c(0,0.5),xpd=TRUE)
  }
}