% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main_func.R
\name{power.glm.fixed.a0}
\alias{power.glm.fixed.a0}
\title{Power/type I error calculation for generalized linear models with fixed a0}
\usage{
power.glm.fixed.a0(
  data.type,
  data.link = "",
  data.size,
  n = 1,
  historical = list(),
  x.samples = matrix(),
  samp.prior.beta,
  samp.prior.var = 0,
  lower.limits = rep(-100, 50),
  upper.limits = rep(100, 50),
  slice.widths = rep(1, 50),
  delta = 0,
  gamma = 0.95,
  nMC = 10000,
  nBI = 250,
  N = 10000,
  approximate = FALSE,
  nNR = 10000,
  tol = 1e-05
)
}
\arguments{
\item{data.type}{Character string specifying the type of response. The options are "Normal", "Bernoulli", "Binomial", "Poisson" and "Exponential".}

\item{data.link}{Character string specifying the link function. The options are "Logistic", "Probit", "Log", "Identity-Positive", "Identity-Probability" and "Complementary Log-Log". Does not apply if \code{data.type} is "Normal".}

\item{data.size}{Sample size of the simulated datasets.}

\item{n}{(For binomial data only) vector of integers specifying the number of subjects who have a particular value of the covariate vector. If the data is binary and all covariates are discrete, collapsing Bernoulli data into a binomial structure can make the slice sampler much faster.}

\item{historical}{(Optional) list of historical dataset(s). East historical dataset is stored in a list which contains three \emph{named} elements: \code{y0}, \code{x0} and \code{a0}.
\itemize{
\item \code{y0} is a vector of responses.
\item \code{x0} is a matrix of covariates. \code{x0} should NOT have the treatment indicator. Apart from missing the treatment indicator, \code{x0} should have the same set of covariates in the same order as \code{x}.
\item \code{a0} is a number between 0 and 1 indicating the discounting parameter value for that historical dataset.
}
For binomial data, an additional element \code{n0} is required.
\itemize{
\item \code{n0} is vector of integers specifying the number of subjects who have a particular value of the covariate vector.
}}

\item{x.samples}{Matrix of possible values of covariates from which covariate vectors are sampled with replacement. Only applies when there is no historical dataset. The matrix should not include the treatment indicator.}

\item{samp.prior.beta}{Matrix of possible values of \eqn{\beta} to sample (with replacement) from. Each row is a possible \eqn{\beta} vector (a realization from the sampling prior for \eqn{\beta}), where the first element is the coefficient for the intercept and the second element is the coefficient for the treatment indicator.
The length of the vector should be equal to the total number of parameters, i.e. P+2 where P is the number of columns of \code{x0} in \code{historical}.}

\item{samp.prior.var}{Vector of possible values of \eqn{\sigma^2} to sample (with replacement) from. Only applies if \code{data.type} is "Normal". The vector contains realizations from the sampling prior (e.g. inverse-gamma distribution) for \eqn{\sigma^2}.}

\item{lower.limits}{Vector of lower limits for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is -100 for all parameters (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{upper.limits}{Vector of upper limits for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is 100 for all parameters (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{slice.widths}{Vector of initial slice widths for parameters to be used by the slice sampler. The length of the vector should be equal to the total number of parameters, i.e. P+1 where P is the number of covariates. The default is 1 for all parameter (may not be appropriate for all situations). Does not apply if \code{data.type} is "Normal".}

\item{delta}{Prespecified constant that defines the boundary of the null hypothesis. The default is zero.}

\item{gamma}{Posterior probability threshold for rejecting the null. The null hypothesis is rejected if posterior probability is greater \code{gamma}. The default is 0.95.}

\item{nMC}{Number of iterations (excluding burn-in samples) for the slice sampler or Gibbs sampler. The default is 10,000.}

\item{nBI}{Number of burn-in samples for the slice sampler or Gibbs sampler. The default is 250.}

\item{N}{Number of simulated datasets to generate. The default is 10,000.}

\item{approximate}{Logical value indicating whether the approximation method based on asymptotic theory is used. The default is FALSE. If TRUE, an approximation method based on the Newton-Raphson algorithm (assuming canonical links) is used.
This feature helps users quickly obtain a rough estimate of the sample size required for the desired level of power or type I error rate.}

\item{nNR}{(Only applies if \code{approximate=TRUE}) number of iterations of the Newton-Raphson algorithm. The default value is 10,000.}

\item{tol}{(Only applies if \code{approximate=TRUE}) absolute tolerance of the Newton-Raphson algorithm. The default value is 0.00001.}
}
\value{
Power or type I error is returned, depending on the sampling prior used. If \code{data.type} is "Normal", average posterior means of \eqn{\beta}, \eqn{\tau} and \eqn{\tau_k}'s (if historical data is given) are also returned.
For all other data types, the average posterior mean of \eqn{\beta} is also returned. The first column of \eqn{\beta} contains posterior samples of the intercept. The second column contains posterior samples of \eqn{\beta_1}, the parameter for the treatment indicator.
}
\description{
Power/type I error calculation for generalized linear models with fixed \eqn{a_0} using power priors
}
\details{
If historical datasets are provided, the algorithm samples with replacement from the historical covariates to construct the simulated datasets.
Otherwise, the algorithm samples with replacement from \code{x.samples}. One of the arguments \code{historical} and \code{x.samples} must be provided.

\code{samp.prior.beta} can be generated using the sampling priors (see example).
\code{samp.prior.var} is necessary for generating normally distributed data.

If \code{data.type} is "Normal", the response \eqn{y_i} is assumed to follow \eqn{N(x_i'\beta, \tau^{-1})} where \eqn{x_i} is the vector of covariates for subject \eqn{i}.
Each historical dataset \eqn{D_{0k}} is assumed to have a different precision parameter \eqn{\tau_k}.
The initial prior for \eqn{\tau} is the Jeffery's prior, \eqn{\tau^{-1}}, and the initial prior for \eqn{\tau_k} is \eqn{\tau_k^{-1}}.
The initial prior for \eqn{\beta} is the uniform improper prior. Posterior samples are obtained through Gibbs sampling.

For all other data types, posterior samples are obtained through slice sampling.
The default lower limits for the parameters are -100. The default upper limits
for the parameters are 100. The default slice widths for the parameters are 1.
The defaults may not be appropriate for all situations, and the user can specify the appropriate limits
and slice width for each parameter.

If a sampling prior with support in the null space is used, the value returned is a Bayesian type I error rate.
If a sampling prior with support in the alternative space is used, the value returned is a Bayesian power.

Because running \code{power.glm.fixed.a0()} and \code{power.glm.random.a0()} is potentially time-consuming,
an approximation method based on asymptotic theory has been implemented for the model with fixed \eqn{a_0}.
In order to attain the exact sample size needed for the desired power, the user can start with the approximation
to get a rough estimate of the sample size required, using \code{power.glm.fixed.a0()} with \code{approximate=TRUE}.
}
\examples{

data.type <- "Bernoulli"
data.link <- "Logistic"
data.size <- 100

# Simulate two historical datasets
p <- 3
historical <- list(list(y0=rbinom(data.size,size=1,prob=0.2),
                        x0=matrix(rnorm(p*data.size),ncol=p,nrow=data.size), a0=0.2),
                   list(y0=rbinom(data.size, size=1, prob=0.5),
                        x0=matrix(rnorm(p*data.size),ncol=p,nrow=data.size), a0=0.3))

# Generate sampling priors

# The null hypothesis here is H0: beta_1 >= 0. To calculate power,
# we can provide samples of beta_1 such that the mass of beta_1 < 0.
# To calculate type I error, we can provide samples of beta_1 such that
# the mass of beta_1 >= 0.
samp.prior.beta1 <- rnorm(100, mean=-3, sd=1)
# Here, mass is put on the alternative region, so power is calculated.
samp.prior.beta <- cbind(rnorm(100), samp.prior.beta1, matrix(rnorm(100*p), 100, p))

nMC <- 100 # nMC should be larger in practice
nBI <- 50
N <- 5 # N should be larger in practice
result <- power.glm.fixed.a0(data.type=data.type, data.link=data.link,
                             data.size=data.size, historical=historical,
                             samp.prior.beta=samp.prior.beta,
                             delta=0, nMC=nMC, nBI=nBI, N=N)

}
\references{
Chen, Ming-Hui, et al. "Bayesian design of noninferiority trials for medical devices using historical data." Biometrics 67.3 (2011): 1163-1170.

Neal, Radford M. Slice sampling. Ann. Statist. 31 (2003), no. 3, 705--767.
}
\seealso{
\code{\link{glm.fixed.a0}}
}
