% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_cp.R
\name{detect_cp}
\alias{detect_cp}
\title{Detect change points on time series.}
\usage{
detect_cp(
  data,
  n_iterations,
  n_burnin = 0,
  params = list(),
  print_progress = TRUE,
  user_seed = 1234
)
}
\arguments{
\item{data}{a vector or a matrix. If a vector the algorithm for
univariate time series is used. If a matrix, where rows are the observations
and columns are the times, then the algorithm for multivariate time series is used.}

\item{n_iterations}{number of MCMC iterations.}

\item{n_burnin}{number of iterations that must be excluded when computing the posterior estimate.}

\item{params}{a list of parameters:

If the time series is univariate the following must be specified:

\itemize{
\item \code{q} probability of performing a split at each iteration.
\item \code{phi} parameter \eqn{\phi} of the integrated likelihood function.
\item \code{a}, \code{b}, \code{c} parameters of the Normal-Gamma prior for \eqn{\mu} and \eqn{\lambda}.
\item \code{par_theta_c}, \code{par_theta_d} parameters of the shifted Gamma prior for \eqn{\theta}.
}

If the time series is multivariate the following must be specified:

\itemize{
\item \code{q} probability of performing a split at each iteration.
\item \code{k_0}, \code{nu_0}, \code{phi_0}, \code{m_0} parameters for the Normal-Inverse-Wishart prior for \eqn{(\mu,\lambda)}.
\item \code{par_theta_c}, \code{par_theta_d} parameters for the shifted Gamma prior for \eqn{\theta}.
\item \code{prior_var_gamma} parameters for the Gamma prior for \eqn{\gamma}.
\item \code{print_progress} If TRUE (default) print the progress bar.
\item \code{user_seed} seed for random distribution generation.
}}

\item{print_progress}{If TRUE (default) print the progress bar.}

\item{user_seed}{seed for random distribution generation.}
}
\value{
A \code{DetectCpObj} class object containing

\itemize{
\item \code{$data} vector or matrix with the data.
\item \code{$n_iterations} number of iterations.
\item \code{$n_burnin} number of burn-in iterations.
\item \code{$orders} matrix where each entries is the assignment of the realization to a block. Rows are the iterations and columns the times.
\item \code{$time} computational time.
\item \code{$gammaMCMC} traceplot for \eqn{\gamma}.
\item \code{$gamma_MCMC_01} a \eqn{0/1} vector, the \eqn{n}-th element is equal to \eqn{1} if the proposed \eqn{\gamma} was accepted, \eqn{0} otherwise.
\item \code{$sigma_MCMC} traceplot for \eqn{\sigma}.
\item \code{$sigma_MCMC_01} a \eqn{0/1} vector, the \eqn{n}-th element is equal to \eqn{1} if the proposed \eqn{\sigma} was accepted, \eqn{0} otherwise.
\item \code{$theta_MCMC} traceplot for \eqn{\theta}.
\item \code{$univariate_ts} TRUE if data is an univariate time series, FALSE if it is a multivariate time series.
}
}
\description{
The \code{detect_cp} function detect change points on univariate and multivariate time series.
}
\examples{

## Univariate time series

data_vec <- as.numeric(c(rnorm(50,0,0.1), rnorm(50,1,0.25)))


out <- detect_cp(data = data_vec, n_iterations = 2500, n_burnin = 500,
                 params = list(q = 0.25, phi = 0.1, a = 1, b = 1, c = 0.1))

print(out)

## Multivariate time series

data_mat <- matrix(NA, nrow = 3, ncol = 100)

data_mat[1,] <- as.numeric(c(rnorm(50,0,0.100), rnorm(50,1,0.250)))
data_mat[2,] <- as.numeric(c(rnorm(50,0,0.125), rnorm(50,1,0.225)))
data_mat[3,] <- as.numeric(c(rnorm(50,0,0.175), rnorm(50,1,0.280)))


out <- detect_cp(data = data_mat, n_iterations = 2500, n_burnin = 500,
                 params = list(q = 0.25, k_0 = 0.25, nu_0 = 4, phi_0 = diag(1,3,3), m_0 = rep(0,3),
                               par_theta_c = 2, par_theta_d = 0.2, prior_var_gamma = 0.1))

print(out)

}
\references{
Martínez, A. F., & Mena, R. H. (2014). On a Nonparametric Change Point Detection Model in Markovian Regimes. \emph{Bayesian Analysis}, 9(4), 823–858. \doi{10.1214/14-BA878}

Corradin, R., Danese, L., & Ongaro, A. (2022). Bayesian nonparametric change point detection for multivariate time series with missing observations. \emph{International Journal of Approximate Reasoning}, 143, 26--43. \doi{10.1016/j.ijar.2021.12.019}
}
