% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_DKP.R
\name{fit.DKP}
\alias{fit.DKP}
\title{Fit a Dirichlet Kernel Process (DKP) Model}
\usage{
fit.DKP(
  X,
  Y,
  Xbounds = NULL,
  prior = c("noninformative", "fixed", "adaptive"),
  r0 = 2,
  p0 = NULL,
  kernel = c("gaussian", "matern52", "matern32"),
  loss = c("brier", "log_loss"),
  n_multi_start = NULL
)
}
\arguments{
\item{X}{A numeric input matrix of size \eqn{n \times d}, where each row
represents a covariate vector.}

\item{Y}{Matrix of observed multinomial counts, with dimension \eqn{n \times
q}.}

\item{Xbounds}{Optional \eqn{d \times 2} matrix specifying the lower and
upper bounds of each input dimension. Used to normalize inputs to
\eqn{[0,1]^d}. If \code{Xbounds} is \code{NULL}, the input is assumed to
have already been normalized, and the default bounds are set to \eqn{[0,1]^d}.}

\item{prior}{Type of prior to use. One of \code{"noninformative"},
\code{"fixed"}, or \code{"adaptive"}.}

\item{r0}{Global prior precision (only used when \code{prior = "fixed"} or
\code{"adaptive"}).}

\item{p0}{Global prior mean vector (only used when \code{prior = "fixed"}).
Must be of length \eqn{q}.}

\item{kernel}{Kernel function for local weighting. Choose from
\code{"gaussian"}, \code{"matern52"}, or \code{"matern32"}.}

\item{loss}{Loss function for kernel hyperparameter tuning. One of
\code{"brier"} (default) or \code{"log_loss"}.}

\item{n_multi_start}{Number of random initializations for multi-start
optimization. Default is \code{10 × d}.}
}
\value{
A list of class \code{"DKP"} representing the fitted DKP model, with
the following components:
\describe{
\item{\code{theta_opt}}{Optimized kernel hyperparameters (lengthscales).}
\item{\code{kernel}}{Kernel function used, as a string.}
\item{\code{loss}}{Loss function used for hyperparameter tuning.}
\item{\code{loss_min}}{Minimum loss value achieved during optimization.}
\item{\code{X}}{Original (unnormalized) input matrix of size \code{n × d}.}
\item{\code{Xnorm}}{Normalized input matrix scaled to \eqn{[0,1]^d}.}
\item{\code{Xbounds}}{Matrix specifying normalization bounds for each input dimension.}
\item{\code{Y}}{Observed multinomial counts of size \code{n × q}.}
\item{\code{prior}}{Type of prior used.}
\item{\code{r0}}{Prior precision parameter.}
\item{\code{p0}}{Prior mean (for fixed priors).}
\item{\code{alpha0}}{Prior Dirichlet parameters at each input location (scalar or matrix).}
\item{\code{alpha_n}}{Posterior Dirichlet parameters after kernel smoothing.}
}
}
\description{
Fits a DKP model for multinomial response data by locally
smoothing observed counts to estimate latent Dirichlet parameters.
}
\examples{
#-------------------------- 1D Example ---------------------------
set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model1 <- fit.DKP(X, Y, Xbounds = Xbounds)
print(model1)


#-------------------------- 2D Example ---------------------------
set.seed(123)

# Define latent function and transform to 3-class probabilities
true_pi_fun <- function(X) {
  if (is.null(nrow(X))) X <- matrix(X, nrow = 1)
  m <- 8.6928; s <- 2.4269
  x1 <- 4 * X[,1] - 2
  x2 <- 4 * X[,2] - 2
  a <- 1 + (x1 + x2 + 1)^2 *
    (19 - 14*x1 + 3*x1^2 - 14*x2 + 6*x1*x2 + 3*x2^2)
  b <- 30 + (2*x1 - 3*x2)^2 *
    (18 - 32*x1 + 12*x1^2 + 48*x2 - 36*x1*x2 + 27*x2^2)
  f <- (log(a * b) - m) / s
  p <- pnorm(f)
  return(matrix(c(p/2, p/2, 1 - p), nrow = length(p)))
}

n <- 100
Xbounds <- matrix(c(0, 0, 1, 1), nrow = 2)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model2 <- fit.DKP(X, Y, Xbounds = Xbounds)
print(model2)

}
\seealso{
\code{\link{fit.BKP}} for modeling binomial responses using the Beta
Kernel Process. \code{\link{predict.DKP}}, \code{\link{plot.DKP}},
\code{\link{simulate.DKP}} for making predictions, visualizing results, and
generating simulations from a fitted DKP model. \code{\link{summary.DKP}},
\code{\link{print.DKP}} for inspecting fitted model summaries.
}
