% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fevd.R
\name{fevd}
\alias{fevd}
\alias{fevd.bgvar.irf}
\title{Forecast Error Variance Decomposition}
\usage{
fevd(x, rotation.matrix=NULL, var.slct=NULL, verbose=TRUE)
}
\arguments{
\item{x}{an object of class \code{bgvar.irf}.}

\item{rotation.matrix}{If \code{NULL} and the \code{x} has been fitted via sign restrictions, the rotation matrix is used that minimizes the distance to the median impulse responses at the posterior median.}

\item{var.slct}{character vector that contains the variables for which forecast error variance decomposition should be performed. If \code{NULL} the FEVD is computed for the whole system, which is very time consuming.}

\item{verbose}{If set to \code{FALSE} it suppresses printing messages to the console.}
}
\value{
Returns a list with two elements \itemize{
\item{\code{FEVD}}{  an array of size (K times horizon times N), where K are all variables in the system, horizon is the specified impulse response horizon and N is the size of the decomposed structural variables (if \code{var.slct=NULL} then K=N).}
\item{\code{xglobal}}{ used data of the model.}
}
}
\description{
This function calculates the forecast error variance decomposition (FEVDs) for Cholesky and sign-identified shocks.
}
\details{
Since the calculations are very time consuming, the FEVDs are based on the posterior median only (as opposed to calculating FEVDs for each MCMC sweep). In case the underlying shock has been identified via sign restrictions, the rotation matrix corresponds to the one that fulfills the sign restrictions at the posterior median of the estimated coefficients. More precisely, the algorithm searches for 50 rotation matrices that fulfill the sign restrictions at the \emph{posterior median} of the coefficients and then singles out the rotation matrix that minimizes the distance to the median of the impulse responses as suggested in Fry and Pagan (2011).
}
\examples{
\dontshow{
set.seed(123)
library(BGVAR)
data(testdata)
model.eer<-bgvar(Data=testdata,W=W.test,prior="MN",
                 draws=100,burnin=50,plag=1,eigen=TRUE)
                      
# US monetary policy shock
shockinfo <- get_shockinfo("chol")
shockinfo$shock <- "US.stir"; shockinfo$scale <- -100
irf.chol.us.mp<-irf(model.eer,n.ahead=48,shockinfo=shockinfo)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.chol.us.mp,var.slct=c("US.Dp","EA.y"))

# US monetary policy shock with sign restrictions
shockinfo <- get_shockinfo("sign")
shockinfo <- add_shockinfo(shockinfo, shock="US.stir", 
                           restriction=c("US.y","US.Dp"), 
                           sign=c("<","<"), horizon=c(1,1), 1, 100)
irf.sign.us.mp<-irf(model.eer,n.ahead=24,shockinfo=shockinfo)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.sign.us.mp,var.slct=c("US.Dp","EA.y"))
}
}
\seealso{
\code{\link{bgvar}}, \code{\link{irf}}
}
\author{
Maximilian Boeck, Martin Feldkircher, Florian Huber
}
