#' The bivariate von Mises sine model
#'
#' @param n number of observations. Ignored if at least one of the other parameters have length k > 1, in which
#' case, all the parameters are recycled to length k to produce k random variates.
#' @param x  bivariate vector  or a two-column matrix with each row being a bivariate vector of angles
#' (in radians)  where the densities are to be evaluated.
#' @param mu1,mu2 vectors of mean parameters.
#' @param kappa1,kappa2,kappa3 vectors of concentration parameters; \code{kappa1, kappa2 > 0}.
#'
#' @details
#' The bivariate von Mises sine model density at the point \eqn{x = (x_1, x_2)} is given by
#' \deqn{f(x) = C_s (\kappa_1, \kappa_2, \kappa_3) \exp(\kappa_1 \cos(T_1) + \kappa_2 \cos(T_2) + \kappa_3 \sin(T_1) \sin(T_2))}
#' where
#' \deqn{T_1 = x_1 - \mu_1;  T_2 = x_2 - \mu_2}
#' and \eqn{C_s (\kappa_1, \kappa_2, \kappa_3)} denotes the normalizing constant for the sine model.
#'
#'
#' @return \code{dvmsin} gives the density  and \code{rvmsin} generates random deviates.
#'
#' @examples
#' kappa1 <- c(1, 2, 3)
#' kappa2 <- c(1, 6, 5)
#' kappa3 <- c(0, 1, 2)
#' mu1 <- c(1, 2, 5)
#' mu2 <- c(0, 1, 3)
#' x <- diag(2, 2)
#' n <- 10
#'
#' # when x is a bivariate vector and parameters are all scalars,
#' # dvmsin returns single density
#' dvmsin(x[1, ], kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # when x is a two column matrix and parameters are all scalars,
#' # dmvsin returns a vector of densities calculated at the rows of
#' # x with the same parameters
#' dvmsin(x, kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # if x is a bivariate vector and at least one of the parameters is
#' # a vector, all parameters are recycled to the same length, and
#' # dvmsin returns a vector of with ith element being the density
#' # evaluated at x with parameter values kappa1[i], kappa2[i],
#' # kappa3[i], mu1[i] and mu2[i]
#' dvmsin(x[1, ], kappa1, kappa2, kappa3, mu1, mu2)
#'
#' # if x is a two column matrix and at least one of the parameters is
#' # a vector, rows of x and the parameters are recycled to the same
#' # length, and dvmsin returns a vector of with ith element being the
#' # density evaluated at ith row of x with parameter values kappa1[i],
#' # kappa2[i], # kappa3[i], mu1[i] and mu2[i]
#' dvmsin(x[1, ], kappa1, kappa2, kappa3, mu1, mu2)
#'
#' # when parameters are all scalars, number of observations generated
#' # by rvmsin is n
#' rvmsin(n, kappa1[1], kappa2[1], kappa3[1], mu1[1], mu2[1])
#'
#' # when at least one of the parameters is a vector, all parameters are
#' # recycled to the same length, n is ignored, and the number of
#' # observations generated by rvmsin is the same as the length of the
#' # recycled vectors
#' rvmsin(n, kappa1, kappa2, kappa3, mu1, mu2)
#'
#' @export

# rvmsin <- function(n, kappa1=1, kappa2=1, kappa3=0, mu1=0, mu2=0)
# {
#   if(any(c(kappa1, kappa2) < 0)) stop("kappa1 and kappa2 must be non-negative")
#   if(any(mu1 < 0 | mu1 >= 2*pi)) mu1 <- prncp_reg(mu1)
#   if(any(mu2 < 0 | mu2 >= 2*pi)) mu2 <- prncp_reg(mu2)
#
#   opt_I_a <- function(k1, k2, k3, mu1, mu2) {
#     I_a <- function(x) BESSI0_C(sqrt(k2^2 + k3^2 * (sin(x - mu1))^2))
#     optimize(I_a, c(0, 2*pi), maximum = TRUE)$maximum
#   }
#
#   if(max(length(kappa1), length(kappa2), length(kappa3), length(mu1), length(mu2)) > 1) {
#     expanded <- expand_args(kappa1, kappa2, kappa3, mu1, mu2)
#     kappa1 <- expanded[[1]]; kappa2 <- expanded[[2]]; kappa3 <- expanded[[3]]
#     mu1 <- expanded[[4]]; mu2 <- expanded[[5]]
#     upper_bd_all <- vapply(1:length(kappa1),
#                            function(h) opt_I_a(kappa1[h], kappa2[h], kappa3[h], mu1[h], mu2[h]),
#                            0)
#     rsin_manypar(kappa1, kappa2, kappa3, mu1, mu2, upper_bd_all)
#   } else {
#     upper_bd <- opt_I_a(kappa1, kappa2, kappa3, mu1, mu2)
#     rsin_onepar(n, kappa1, kappa2, kappa3, mu1, mu2, upper_bd)
#   }
# }


rvmsin <- function(n, kappa1=1, kappa2=1, kappa3=0, mu1=0, mu2=0) {
  if(any(c(kappa1, kappa2) < 0)) stop("kappa1 and kappa2 must be nonnegative")
  if(any(mu1 < 0 | mu1 >= 2*pi)) mu1 <- prncp_reg(mu1)
  if(any(mu2 < 0 | mu2 >= 2*pi)) mu2 <- prncp_reg(mu2)

  opt_obj <- function(k1=1, k2=1, k3=0, mu1=0, mu2=0) {
    # for numerical stability, if k3 < 0, fabs(k1+k3) < 1e-5 or fabs(k2+k3) < 1e-5
    # make k3 = k3 * (1+1e-5)
    if (k3 < 0) {
      while (abs(k1 + k3) < 1e-5 || abs(k2 + k3) < 1e-5) {
        k3 = k3*(1+1e-5)
      }
    }
    obj <- optim(c(0,0), fn = function(x) -(k1*cos(x[1]-mu1)+k2*cos(x[2]-mu2)+k3*sin(x[1]-mu1)*sin(x[2]-mu2)),
                 gr = function(x)  -c(-k1*sin(x[1]-mu1)+k3*cos(x[1]-mu1)*sin(x[2]-mu2),
                                      -k2*sin(x[2]-mu2)+k3*sin(x[1]-mu1)*cos(x[2]-mu2)))
    -obj$value

  }

  if(max(length(kappa1), length(kappa2), length(kappa3), length(mu1), length(mu2)) > 1) {
    expanded <- expand_args(kappa1, kappa2, kappa3, mu1, mu2)
    k1 <- expanded[[1]]; k2 <- expanded[[2]]; k3 <- expanded[[3]]
    mu1 <- expanded[[4]]; mu2 <- expanded[[5]]
    upper_bd_all <- vapply(1:length(k1),
                           function(h) opt_obj(k1[h], k2[h], k3[h], mu1[h], mu2[h]),
                           0)
    rsin_manypar(k1, k2, k3, mu1, mu2, upper_bd_all)
  } else {
    upper_bd <- opt_obj(kappa1, kappa2, kappa3, mu1, mu2)
    rsin_onepar(n, kappa1, kappa2, kappa3, mu1, mu2, upper_bd)
  }
}

#' @rdname rvmsin
#' @export
dvmsin <- function(x, kappa1=1, kappa2=1, kappa3=0, mu1=0, mu2=0)
{
  if(any(c(kappa1, kappa2) < 0)) stop("kappa1 and kappa2 must be non-negative")
  if(any(mu1 < 0 | mu1 >= 2*pi)) mu1 <- prncp_reg(mu1)
  if(any(mu2 < 0 | mu2 >= 2*pi)) mu2 <- prncp_reg(mu2)
  if((length(dim(x)) < 2 && length(x) != 2) || (length(dim(x)) == 2 && tail(dim(x), 1) != 2)
     || (length(dim(x)) > 2)) stop("x must either be a bivariate vector or a two-column matrix")

  if(max(length(kappa1), length(kappa2), length(kappa3), length(mu1), length(mu2)) > 1) {
    expanded <- expand_args(kappa1, kappa2, kappa3, mu1, mu2)
    kappa1 <- expanded[[1]]; kappa2 <- expanded[[2]]; kappa3 <- expanded[[3]]
    mu1 <- expanded[[4]]; mu2 <- expanded[[5]]

    if(length(x) != 2) {
      x_set <- 1:nrow(x)
      par_set <- 1:length(kappa1)
      expndn_set <- expand_args(x_set, par_set)
      x_set <- expndn_set[[1]]
      par_set <- expndn_set[[2]]
      as.vector(dsin_manyx_manypar(x[x_set, ], kappa1[par_set], kappa2[par_set], kappa3[par_set], mu1[par_set], mu2[par_set]))
    } else{
      as.vector(dsin_onex_manypar(x, kappa1, kappa2, kappa3, mu1, mu2))
    }

  } else {
    if(length(x) != 2){
      as.vector(dsin_manyx_onepar(x, kappa1, kappa2, kappa3, mu1, mu2))
    } else{
      exp(ldsinnum(x[1], x[2], c(kappa1, kappa2, kappa3, mu1, mu2)))/const_vmsin(kappa1, kappa2, kappa3)
    }
  }
}

#' The bivariate von Mises sine model mixtures
#' @param n number of observations.
#' @param x matrix of angles (in radians) where the density is to be evaluated, with each row being a
#' single bivariate vector of angles.
#'
#' @param pmix vector of mixture proportions.
#' @param mu1,mu2 vectors of mean parameters.
#' @param kappa1,kappa2,kappa3 vectors of concentration parameters; \code{kappa1, kappa2 > 0} for each component.
#'
#' @details All the argument vectors \code{pmix, kappa1, kappa2, kappa3, mu1} and \code{mu2} must be of
#' the same length ( = component size of the mixture model), with \eqn{j}-th element corresponding to the
#' \eqn{j}-th component of the mixture distribution.
#' @details The bivariate von Mises sine model mixture distribution with component size \code{K = \link{length}(p.mix)} has density
#' \deqn{g(x) = \sum p[j] * f(x; \kappa_1[j], \kappa_2[j], \kappa_3[j], \mu_1[j], \mu_2[j])}
#' where the sum extends over \eqn{j}; \eqn{p[j]; \kappa_1[j], \kappa_2[j], \kappa_3[j]}; and \eqn{\mu_1[j], \mu_2[j]} respectively denote the mixing proportion,
#' the three concentration parameters and the two mean parameter for the \eqn{j}-th component, \eqn{j = 1, ..., K},
#' and \eqn{f(. ; \kappa_1, \kappa_2, \kappa_3, \mu_1, \mu_2)} denotes the density function of the von Mises sine model
#' with concentration parameters \eqn{\kappa_1, \kappa_2, \kappa_3} and  mean parameters \eqn{\mu_1, \mu_2}.
#'
#' @return \code{dvmsinmix} computes the density (vector if x is a two column matrix with more than one row)
#' and \code{rvmsinmix} generates random deviates from the mixture density.
#'
#' @examples
#' kappa1 <- c(1, 2, 3)
#' kappa2 <- c(1, 6, 5)
#' kappa3 <- c(0, 1, 2)
#' mu1 <- c(1, 2, 5)
#' mu2 <- c(0, 1, 3)
#' pmix <- c(0.3, 0.4, 0.3)
#' x <- diag(2, 2)
#' n <- 10
#'
#' # mixture densities calculated at the rows of x
#' dvmsinmix(x, kappa1, kappa2, kappa3, mu1, mu2, pmix)
#'
#' # number of observations generated from the mixture distribution is n
#' rvmsinmix(n, kappa1, kappa2, kappa3, mu1, mu2, pmix)
#'
#' @export

rvmsinmix <- function(n, kappa1, kappa2, kappa3, mu1, mu2, pmix)
{
  allpar <- list(kappa1=kappa1, kappa2=kappa2, kappa3=kappa3,
                 mu1=mu1, mu2=mu2, pmix=pmix)

  allpar_len <- listLen(allpar)
  if(min(allpar_len) != max(allpar_len))
    stop("component size mismatch: number of components of the input parameter vectors differ")

  if(any(allpar$pmix < 0)) stop("\'pmix\' must be non-negative")
  sum_pmix <- sum(allpar$pmix)
  if(signif(sum_pmix, 5) != 1) {
    if(sum_pmix <= 0) stop("\'pmix\' must have at least one positive element")
    allpar$pmix <- allpar$pmix/sum_pmix
    warning("\'pmix\' is rescaled to add up to 1")
  }

  if(any(c(allpar$kappa1, allpar$kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(allpar$mu1 < 0 | allpar$mu1 >= 2*pi)) allpar$mu1 <- prncp_reg(allpar$mu1)
  if(any(allpar$mu2 < 0 | allpar$mu2 >= 2*pi)) allpar$mu2 <- prncp_reg(allpar$mu2)

  # opt_I_a <- function(k1, k2, k3, mu1, mu2) {
  #   I_a <- function(x) BESSI0_C(sqrt(k2^2 + k3^2 * (sin(x - mu1))^2))
  #   optimize(I_a, c(0, 2*pi), maximum = TRUE)$maximum
  # }
  # upper_bd_all <- vapply(1:length(kappa1),
  #                        function(h) opt_I_a(kappa1[h], kappa2[h], kappa3[h], mu1[h], mu2[h]),
  #                        0)
  # clus_label <- cID(t(replicate(allpar$pmix, n = n)), length(allpar$pmix), runif(n))
  # rsin_manypar(allpar$kappa1[clus_label], allpar$kappa2[clus_label], allpar$kappa3[clus_label],
  #              allpar$mu1[clus_label], allpar$mu2[clus_label], upper_bd_all)

  # browser()
  out <- matrix(0, n, 2)
  ncomp <- allpar_len[1] # number of components
  comp_ind <- cID(t(replicate(n, pmix)), ncomp, runif(n))
  # n samples from multinom(ncomp, pmix)
  for(j in seq_len(ncomp)) {
    obs_ind_j <- which(comp_ind == j)
    n_j <- length(obs_ind_j)
    if(n_j > 0) {
      out[obs_ind_j, ] <- rvmsin(n_j, kappa1[j], kappa2[j],
                                 kappa3[j], mu1[j], mu2[j])
    }
  }

  out
}


#' @rdname rvmsinmix
#' @export
dvmsinmix <- function(x, kappa1, kappa2, kappa3, mu1, mu2, pmix)
{
  allpar <- list("kappa1"=kappa1, "kappa2"=kappa2, "kappa3"=kappa3,
                 "mu1"=mu1, "mu2"=mu2, "pmix"=pmix)

  allpar_len <- listLen(allpar)
  if(min(allpar_len) != max(allpar_len)) stop("component size mismatch: number of components of the input parameter vectors differ")

  if(any(allpar$pmix < 0)) stop("\'pmix\' must be non-negative")
  sum_pmix <- sum(allpar$pmix)
  if(signif(sum_pmix, 5) != 1) {
    if(sum_pmix <= 0) stop("\'pmix\' must have at least one positive element")
    allpar$pmix <- allpar$pmix/sum_pmix
    warning("\'pmix\' is rescaled to add up to 1")
  }

  if(any(c(allpar$kappa1, allpar$kappa2) <= 0)) stop("kappa1 and kappa2 must be positive")
  if(any(allpar$mu1 < 0 | allpar$mu1 >= 2*pi)) allpar$mu1 <- prncp_reg(allpar$mu1)
  if(any(allpar$mu2 < 0 | allpar$mu2 >= 2*pi)) allpar$mu2 <- prncp_reg(allpar$mu2)
  if((length(dim(x)) < 2 && length(x) != 2) || (length(dim(x)) == 2 && tail(dim(x), 1) != 2)
     || (length(dim(x)) > 2)) stop("x must either be a bivariate vector or a two-column matrix")


  par_mat <- rbind(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2)
  pi_mix <- allpar$pmix
  log_c_von = log_const_vmsin_all(par_mat)

  if(length(x) == 2) {
    vmsinmix(x[1], x[2],par_mat, pi_mix, log_c_von)
  } else {
    as.vector(vmsinmix_manyx(x, par_mat, pi_mix, log_c_von))
  }
}


#' Fitting bivariate von Mises sine model mixtures using MCMC
#'
#' @param data vector of observations. If outside, the values are transformed into the scale \eqn{[0, 2\pi)}.
#' @param ncomp number of components in the mixture model. If \code{comp == 1}, a single component model is fitted.
#' @param start_par list with elements \code{pmix} (ignored if \code{comp == 1}), \code{kappa1, kappa2, mu1} and \code{mu2},
#' all being vectors of length same as \code{ncomp},
#' providing the starting values; with \eqn{j}-th component of each vector corresponding to the \eqn{j}-th component of the
#' mixture distribution. If missing, moment estimators based on random components are used.
#' @param method MCMC strategy to be used for the model paramters:  \code{"hmc"} or \code{"rwmh"}.
#' @param epsilon,L  tuning parameters for HMC; ignored if \code{method = "rwmh"}. \code{epsilon} (step-size) is a quantity in
#' \eqn{[0, 1)} and \code{L} (leapfrog steps) is a positive integer.
#' @param epsilon.random logical. Should a random value from  Uniform(0, \code{epsilon}) be used for \code{epsilon}
#' at each iteration? Ignored if \code{method = "rwmh"}.
#' @param L.random logical. Should a random value from discrete Uniform(1,..., \code{L}) be used for \code{L}
#' at each iteration? Ignored if \code{method = "rwmh"}.
#' @param propscale tuning parameters for RWMH; a vector of size 5 representing the variances for the proposal normal densities
#' for \eqn{\kappa_1, \kappa_2, \kappa_3, \mu_1} and \eqn{\mu_2} respectively. Ignored if \code{method = "hmc"}.
#' @param n.iter number of iterations for the Markov Chain.
#' @param gam.loc,gam.scale location and scale (hyper-) parameters for the gamma prior for \code{kappa1} and \code{kappa2}. See
#' \link{dgamma}. Defaults are \code{gam.loc = 0, gam.scale = 1000} that makes the prior non-informative.
#' @param pmix.alpha concentration parameter(s) for the Dirichlet prior for \code{pmix}. Must either be a positive real number, or a vector
#' with positive entries and of the same size as \code{pmix}. The default is 1/2 which corresponds to the Jeffreys prior.
#' @param norm.var variance (hyper-) parameter in the normal prior for \code{kappa3}. (Prior mean is zero).
#' Default is 1000 that makes the prior non-informative.
#' @param  autotune logical. Should the Markov chain auto-tune the parameters (\code{epsilon} in HMC and
#' \code{propscale} in RWMH) on the basis of acceptances in \code{iter.tune}? (The iterations used to tune the
#' parameters are discarded.) Set to \code{FALSE} by default.
#' @param iter.tune number of initial iterations used to tune the parameters (\code{epsilon} in HMC and
#' \code{propscale} in RWMH). Default is 20. Ignored if \code{autotune == FALSE}.
#' @param ncores number of CPU cores to be used for computing the likelihood, the posterior weight matrix for
#' the Gibbs Sampler of mixture proportions and the gradient in HMC in parallel. Default is all of the available cores
#' (obtained via \link{detectCores}). Note that parallelization is implemented using OpenMP.  This argument is ignored and the
#' computations are done serially if OpenMP is not available.
#' @param show.progress logical. Should a progress bar be included?
#'
#' @details
#' \code{fit_vmsinmix} generates MCMC samples of vmsin mixture model parameters, and returns an
#' angmcmc object as the output, which can be used as an argument for diagnostics and estimation
#' functions.
#'
#' Default \code{method} is \code{"hmc"}.
#'
#' If the acceptance rate drops below 5\% after 100 or more HMC iterations, \code{epsilon} is automatically lowered, and the
#' Markov chain is restarted at the current parameter values.
#'
#' @return returns an angmcmc object.
#'
#' @examples
#' # illustration only - more iterations needed for convergence
#' fit.vmsin.20 <- fit_vmsinmix(tim8, ncomp = 3, n.iter =  20,
#'                              ncores = 1)
#' fit.vmsin.20
#'
#' @export

fit_vmsinmix <- function(data, ncomp, start_par = list(), method="hmc", epsilon=0.01, L=10, epsilon.random=TRUE,
                         L.random=FALSE, propscale = rep(0.01, 5), n.iter=500, gam.loc=0, gam.scale=1000, pmix.alpha = 1/2,
                         norm.var=1000, autotune = FALSE, iter.tune = 10, ncores, show.progress = TRUE) {

  if(is.null(dim(data)) | !(mode(data) %in% c("list", "numeric") && ncol(data) == 2)) stop("non-compatible data")

  kappa_upper <- 150

  curr.model <- "vmsin"
  data.rad <- rm_NA_rad(data)
  n.data <- nrow(data.rad)

  if(missing(ncores)) {
    ncores <- floor(parallel::detectCores())
  }

  if(ncomp == 1) {

    if(missing(start_par)) {
      starting <- start_clus_kmeans_vmsin(data.rad, ncomp, nstart=5)
      starting$par.mat <- matrix(starting$par.mat, ncol=1)
    } else {
      allpar <- start_par
      if(any(is.null(allpar$kappa1), is.null(allpar$kappa2), is.null(allpar$kappa3),
             is.null(allpar$mu1), is.null(allpar$mu2)) ) {
        stop("too few elements in start_par, with no default")
      }
      allpar1 <- list(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2)
      allpar_len <- listLen(allpar1)
      if(min(allpar_len) != max(allpar_len)){
        stop("component size mismatch: number of components of the input parameter vectors differ")
      }
      starting <- list("par.mat" = rbind(start_par$kappa1, start_par$kappa2, start_par$kappa3,
                                         start_par$mu1, start_par$mu2), "pi.mix" = 1)
    }
  } else if(ncomp > 1) {
    if(missing(start_par)) {
      starting <- start_clus_kmeans_vmsin(data.rad, ncomp, nstart=5)
    } else {
      allpar <- start_par
      if(any(is.null(allpar$kappa1), is.null(allpar$kappa2), is.null(allpar$kappa3),
             is.null(allpar$mu1), is.null(allpar$mu2), is.null(allpar$pmix)) ) {
        stop("too few elements in start_par, with no default")
      }
      allpar1 <- list(allpar$kappa1, allpar$kappa2, allpar$kappa3, allpar$mu1, allpar$mu2, allpar$pmix)
      allpar_len <- listLen(allpar1)
      allpar_len <- listLen(allpar)
      if(min(allpar_len) != max(allpar_len)){
        stop("component size mismatch: number of components of the input parameter vectors differ")
      }
      starting <- list("par.mat" = rbind(start_par$kappa1, start_par$kappa2, start_par$kappa3,
                                         start_par$mu1, start_par$mu2), "pi.mix" = start_par$pmix)
    }
  }

  starting$par.mat[abs(starting$par.mat) >= kappa_upper/2] <- kappa_upper/2
  starting$l.c.vmsin <- as.numeric(log_const_vmsin_all(starting$par.mat))
  starting$llik <- llik_vmsin_full(data.rad, starting$par.mat, starting$pi.mix, starting$l.c.vmsin, ncores)
  starting$lprior <- sum((pmix.alpha-1)*log(starting$pi.mix)) + sum(ldgamanum(starting$par.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((starting$par.mat[3,]/norm.var)^2)
  starting$lpd <- starting$llik + starting$lprior

  par.mat.all <- array(0, dim = c(5, ncomp, n.iter+1))
  pi.mix.all <- matrix(1, nrow = ncomp, ncol = n.iter+1)
  llik.all <- lprior.all <- lpd.all <- rep(-Inf, (n.iter+1))
  accpt.par.mat.all <- accpt.kappa.all <- accpt.mu.all <- rep(0, (n.iter+1))
  modelpar.names <- c("kappa1", "kappa2", "kappa3", "mu1", "mu2")

  MC <- starting  #simulation results list, 1st entry = method of moments on kmeans output

  par.mat.all[,,1] <- MC$par.mat
  pi.mix.all[,1] <- MC$pi.mix
  llik.all[1] <- MC$llik
  lprior.all[1] <- MC$lprior
  lpd.all[1] <- MC$lpd

  epsilon_ave <- NULL
  L_ave <- NULL
  propscale_final <- NULL

  clus.ind <- matrix(1, nrow = n.data, ncol = n.iter+2)

  iter <- 2
  ntune <- 0

  if(show.progress) pb <- txtProgressBar(min = 2, max = n.iter+1, style = 3)

  #******************************************************************************************
  # single component model
  #******************************************************************************************

  if(ncomp == 1 && grepl(method, "hmc")) # using hmc
  {
    if(epsilon.random) {
      epsilon_vec <- runif(n.iter, min = 0.9*epsilon, max = 1.1*epsilon)
      # epsilon.0 <- epsilon
    }
    if(L.random) {
      L_vec <- sample(1:L, n.iter, replace = TRUE)
      # L.0 <- L
    }

    while(iter <= (n.iter+1)) {
      broken <- FALSE
      kappa.large <- FALSE

      pi.mix.1 <- 1
      par.mat.old <- MC$par.mat
      l.c.vmsin.old <- MC$l.c.vmsin

      llik_new.pi <- MC$llik

      #----------------------------------------------------------------------------------
      #generating par.mat by HMC
      #----------------------------------------------------------------------------------

      par.mat.1 <- par.mat.old
      lprior.1 <- MC$lprior
      llik.1 <- llik_new.pi
      lpd.1 <- llik.1 + lprior.1
      l.c.vmsin.1 <- MC$l.c.vmsin
      accpt.par.mat <- 0


      current_q <- par.mat.1
      current_p <- matrix(rnorm(5*ncomp,0,1), nrow = 5)  # independent standard normal variates
      p <- current_p
      q <- current_q

      if(L.random) {
        L <- L_vec[iter-1]
        # L <- sample.int(1:L.0, n.iter, replace = TRUE)
      }
      if(epsilon.random) {
        epsilon <- epsilon_vec[iter-1]
        # epsilon <- runif(n.iter, min = 0.9*epsilon.0, max = 1.1*epsilon.0)
      }


      # Do leapfrog with L and epsilon
      {
        # Make a half step for momentum at the beginning

        p <- p - (epsilon/2) * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1)
        ) # the second term in the bracket arises from prior
        # Alternate full steps for position and momentum

        for (i in 1:L)
        {
          # Make a full step for the position

          q <- q + epsilon * p

          if(all(!is.nan(q)) && any(abs(q[1:3, ]) >= kappa_upper)) {
            kappa.large <- TRUE
            break
          }

          if(any(is.nan(c(q,p))))  {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          }

          # Make sure the components of q1 are in the proper ranges
          {
            q1 <- q; p1 <- p


            for(j in 1:ncomp) {

              while(q1[1,j] <= 0) {
                q1[1,j] <- -q1[1,j]; p1[1,j] <- -p1[1,j]
              }

              while(q1[2,j] <= 0) {
                q1[2,j] <- -q1[2,j]; p1[2,j] <- -p1[2,j]
              }

              # q3 is unbounded in vmsin

              while(q1[4,j] < 0 || q1[4,j] >= 2*pi) {
                if(q1[4,j] < 0) {
                  q1[4,j] <- - q1[4,j]; p1[4,j] <- -p1[4,j]
                } else {
                  q1[4,j] <- 4*pi - q1[4,j]; p1[4,j] <- -p1[4,j]
                }
              }
              while(q1[5,j] < 0 || q1[5,j] >= 2*pi) {
                if(q1[5,j] < 0) {
                  q1[5,j] <- - q1[5,j]; p1[5,j] <- -p1[5,j]
                } else {
                  q1[5,j] <- 4*pi - q1[5,j]; p1[5,j] <- -p1[5,j]
                }
              }
            }


            p <- p1; q <- q1
          }
          # Make a full step for the momentum, except at end of trajectory

          if(any(is.nan(c(p, q))))  {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          } else if(i!=L) {
            p <- p - epsilon * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1 )) # the second term in the bracket arises from prior
          }
        }

        # Make a half step for momentum at the end.
        if(all(!broken, !kappa.large)){
          if(any(is.nan(c(p, q))))  {
            broken <- TRUE
          } else {
            p <- p - (epsilon/2) * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                    + matrix(c(1/gam.scale + (1- 1/gam.scale)/q[1:2,], q[3,], rep(0,2)), ncol=1 ) ) # the second term in the bracket arises from prior

          }
        }
        # Negate momentum at end of trajectory to make the proposal symmetric

        if(any(is.nan(c(p, q))))  {
          broken <- TRUE
        } else {
          p <-  -p
        }
      }


      if (iter > 100 && mean(accpt.par.mat.all[1:iter]) < 0.05) {
        broken <- TRUE
      }

      if (broken) {
        print("Acceptance rate too low. Automatically restarting with a smaller \'epsilon\'.")
        iter <- 2
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }


        par.mat.all[,,iter] <- par.mat.1
        pi.mix.all[,iter] <- pi.mix.1
        llik.all[iter] <- llik.1
        lprior.all[iter] <- lprior.1
        lpd.all[iter] <- lpd.1
        accpt.par.mat.all[iter] <- accpt.par.mat


        next
      }



      # Evaluate potential and kinetic energies at start and end of trajectory
      current_U <- -lpd.1
      current_K <- sum(current_p^2) / 2

      if(kappa.large) {
        proposed_U <- proposed_K <- Inf
      } else {

        par.mat.prop <- q
        l.c.vmsin.prop <- log_const_vmsin_all(par.mat.prop)

        lprior.prop <- sum((pmix.alpha-1) * log(pi.mix.1)) + sum(ldgamanum(q[1:2,], gam.loc, gam.scale)) - 0.5*sum((q[3,]/norm.var)^2)

        llik.prop <- llik_vmsin_full(data.rad, q, pi.mix.1, l.c.vmsin.prop, ncores)

        proposed_U <- -(llik.prop + lprior.prop)
        proposed_K <- sum(p^2) / 2
      }

      exp(current_U-proposed_U+current_K-proposed_K)
      # Accept or reject the state at end of trajectory, returning either
      # the position at the end of the trajectory or the initial position

      if (runif(1) < exp(current_U-proposed_U+current_K-proposed_K))
      {
        # return (q)  # accept
        # accpt = 1
        par.mat.1 <- signif(par.mat.prop, 8)
        lprior.1 <- signif(lprior.prop, 8)
        llik.1 <- signif(llik.prop, 8)
        lpd.1 <- signif(-proposed_U, 8)
        accpt.par.mat <- 1
        l.c.vmsin.1 <- signif(l.c.vmsin.prop, 8)
      }


      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.vmsin" = l.c.vmsin.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.par.mat" = accpt.par.mat)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.par.mat.all[iter] <- accpt.par.mat


      # tuning epsilon with first 20 draws
      if(autotune && iter == iter.tune && mean(accpt.par.mat.all[2:(iter.tune+1)]) < 0.6) {
        iter <- 2
        ntune <- ntune + 1
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter + 1

    }
  }

  if(ncomp == 1 && grepl(method, "rwmh")) # using rwmh
  {
    while(iter <= (n.iter+1)) {
      pi.mix.1 <- 1
      par.mat.old <- MC$par.mat
      l.c.vmsin.old <- MC$l.c.vmsin

      llik_new.pi <- MC$llik
      #----------------------------------------------------------------------------------
      #generating presicion parameters
      #----------------------------------------------------------------------------------

      k1.1.old <- par.mat.old[1, ]
      k2.1.old <- par.mat.old[2, ]
      k3.1.old <- par.mat.old[3, ]

      k1.1.prop <- pmax(k1.1.old + rnorm(ncomp,0,propscale[1]), 1e-6)
      k2.1.prop <- pmax(k2.1.old + rnorm(ncomp,0,propscale[2]), 1e-6)
      k3.1.prop <- k3.1.old + rnorm(ncomp,0,propscale[3])

      prop.mat <- unname(matrix(c(k1.1.prop,k2.1.prop,k3.1.prop, par.mat.old[4:5, ]),ncol=1))
      l.c.vmsin.prop <- as.numeric(log_const_vmsin_all(prop.mat))

      llik_old <- llik_new.pi
      llik_prop <- llik_vmsin_full(data.rad, prop.mat, pi.mix.1, l.c.vmsin.prop, ncores)

      lprior_old <- MC$lprior
      lprior_prop <- sum((pmix.alpha-1) * log(pi.mix.1)) + sum(ldgamanum(prop.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((prop.mat[3,]/norm.var)^2)

      post.omg_old <- llik_old + lprior_old
      post.omg_prop <- llik_prop + lprior_prop

      if (runif(1) <  exp(post.omg_prop-post.omg_old) ) {
        k1.1 <- k1.1.prop
        k2.1 <- k2.1.prop
        k3.1 <- k3.1.prop
        accpt.kappa <- 1
        l.c.vmsin.1 <- l.c.vmsin.prop
        llik_new.omg <- llik_prop
        lprior.1 <- lprior_prop
        par.mat_new.omg <- prop.mat
      } else {
        k1.1 <- k1.1.old
        k2.1 <- k2.1.old
        k3.1 <- k3.1.old
        accpt.kappa <- 0
        l.c.vmsin.1 <- l.c.vmsin.old
        llik_new.omg <- llik_old
        lprior.1 <- lprior_old
        par.mat_new.omg <- par.mat.old
      }


      #----------------------------------------------------------------------------------
      #generating mu and nu
      #----------------------------------------------------------------------------------
      prop.mu <- prncp_reg(par.mat.old[4, ] + rnorm(ncomp,0,propscale[4]))
      prop.nu <- prncp_reg(par.mat.old[5, ] + rnorm(ncomp,0,propscale[5]))
      #----------------------------------------------------------------------------------
      prop.mat.mean <- matrix(c(par.mat_new.omg[1:3,], prop.mu,prop.nu), ncol=1)

      llik_new.prop <- llik_vmsin_full(data.rad, prop.mat.mean, pi.mix.1, l.c.vmsin.1, ncores)

      if (runif(1) <  exp(llik_new.prop-llik_new.omg) ) {
        par.mat.1 <- prop.mat.mean
        accpt.mu <- 1
        llik.1 <- llik_new.prop
      } else {
        par.mat.1 <- par.mat_new.omg
        accpt.mu <- 0
        llik.1 <- llik_new.omg
      }

      lpd.1 <- llik.1 + lprior.1

      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.vmsin" = l.c.vmsin.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.kappa" = accpt.kappa, "accpt.mu" = accpt.mu)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.kappa.all[iter] <- accpt.kappa
      accpt.mu.all[iter] <- accpt.mu

      # tuning propscale with first 20 draws
      if(autotune && iter == iter.tune && (mean(accpt.kappa.all[2:(iter.tune+1)]) < 0.6 ||
                                           mean(accpt.mu.all[2:(iter.tune+1)]) < 0.6)) {
        iter <- 2
        ntune <- ntune + 1
        propscale <- propscale/2
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter+1

    }
  }
  #******************************************************************************************

  #******************************************************************************************
  # multiple component model
  #******************************************************************************************

  if(ncomp > 1 && grepl(method, "hmc")) # using hmc
  {
    if(epsilon.random) {
      epsilon_vec <- runif(n.iter, min = 0.9*epsilon, max = 1.1*epsilon)
    }
    if(L.random) {
      L_vec <- sample(1:L, n.iter, replace = TRUE)
      # L.0 <- L
    }

    while(iter <= (n.iter+1)) {
      broken <- FALSE
      kappa.large <- FALSE
      #----------------------------------------------------------------------------------
      #generating mixture proportions
      #----------------------------------------------------------------------------------
      pi.mix.old <- MC$pi.mix
      par.mat.old <- MC$par.mat
      l.c.vmsin.old <- MC$l.c.vmsin


      # Gibbs Sampler
      {
        post.wt <- mem_p_sin(data.rad, par.mat.old, pi.mix.old, l.c.vmsin.old, ncores)
        clus.ind[ , iter] <- cID(post.wt, ncomp, runif(n.data))
        n.clus <- tabulate(clus.ind[ , iter], nbins = ncomp)
        pi.mix.1 <- as.numeric(rdirichlet(1, (pmix.alpha + n.clus))) #new mixture proportions
        llik_new.pi <- llik_vmsin_full(data.rad, par.mat.old, pi.mix.1, l.c.vmsin.old, ncores)
      }

      #----------------------------------------------------------------------------------
      #generating par.mat by HMC
      #----------------------------------------------------------------------------------

      par.mat.1 <- par.mat.old
      lprior.1 <- MC$lprior
      llik.1 <- llik_new.pi
      lpd.1 <- llik.1 + lprior.1
      l.c.vmsin.1 <- MC$l.c.vmsin
      accpt.par.mat <- 0


      current_q <- par.mat.1
      current_p <- matrix(rnorm(5*ncomp,0,1), nrow = 5)  # independent standard normal variates
      p <- current_p
      q <- current_q

      if(L.random) {
        L <- L_vec[iter-1]
        # L <- sample.int(1:L.0, n.iter, replace = TRUE)
      }
      if(epsilon.random) {
        epsilon <- epsilon_vec[iter-1]
        # epsilon <- runif(n.iter, min = 0.9*epsilon.0, max = 1.1*epsilon.0)
      }
      # Do leapfrog with L and epsilon
      {
        # Make a half step for momentum at the beginning

        p <- p - (epsilon/2) * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior
        # Alternate full steps for position and momentum

        for (i in 1:L)
        {

          # Make a full step for the position

          q <- q + epsilon * p

          if(all(!is.nan(q)) && any(abs(q[1:3, ]) >= kappa_upper)) {
            kappa.large <- TRUE
            break
          }



          if(any(is.nan(c(q, p)))) {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          }


          # Make sure the components of q1 are in the proper ranges
          {
            q1 <- q; p1 <- p

            for(j in 1:ncomp) {

              while(q1[1,j] <= 0) {
                q1[1,j] <- -q1[1,j]; p1[1,j] <- -p1[1,j]
              }

              while(q1[2,j] <= 0) {
                q1[2,j] <- -q1[2,j]; p1[2,j] <- -p1[2,j]
              }

              # q3 is unbounded in vmsin

              while(q1[4,j] < 0 || q1[4,j] >= 2*pi) {
                if(q1[4,j] < 0) {
                  q1[4,j] <- - q1[4,j]; p1[4,j] <- -p1[4,j]
                } else {
                  q1[4,j] <- 4*pi - q1[4,j]; p1[4,j] <- -p1[4,j]
                }
              }
              while(q1[5,j] < 0 || q1[5,j] >= 2*pi) {
                if(q1[5,j] < 0) {
                  q1[5,j] <- - q1[5,j]; p1[5,j] <- -p1[5,j]
                } else {
                  q1[5,j] <- 4*pi - q1[5,j]; p1[5,j] <- -p1[5,j]
                }
              }
            }


            p <- p1; q <- q1
          }
          # Make a full step for the momentum, except at end of trajectory

          if(any(is.nan(c(p, q)))) {
            broken <- TRUE
            #stop("Algorithm breaks. Try a smaller epsilon.")
            break
          }  else if(i!=L)  {
            p <- p - epsilon * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior
          }
        }

        # Make a half step for momentum at the end.
        if(all(!broken, !kappa.large)) {
          if(any(is.nan(c(p, q)))) {
            broken <- TRUE
          } else {
            p <- p - (epsilon/2) * (- grad_vmsin_all_comp(data.rad, q, pi.mix.1, ncores)
                                    + rbind((1/gam.scale + (1- 1/gam.scale)/q[1:2,]), q[3,], matrix(0, nrow = 2, ncol = ncomp)) ) # the second term in the bracket arises from prior
          }
        }

        # Negate momentum at end of trajectory to make the proposal symmetric

        if(any(is.nan(c(p, q)))) {
          broken <- TRUE
        } else {
          p <-  -p
        }

      }



      if (iter > 100 && mean(accpt.par.mat.all[1:iter]) < 0.05) {
        broken <- TRUE
      }

      if (broken) {
        print("Acceptance rate too low. Automatically restarting with a smaller \'epsilon\'.")
        iter <- 2
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }


        par.mat.all[,,iter] <- par.mat.1
        pi.mix.all[,iter] <- pi.mix.old
        llik.all[iter] <- llik.1
        lprior.all[iter] <- lprior.1
        lpd.all[iter] <- lpd.1
        accpt.par.mat.all[iter] <- accpt.par.mat


        next

      }



      # Evaluate potential and kinetic energies at start and end of trajectory

      current_U <- -lpd.1
      current_K <- sum(current_p^2) / 2

      if(kappa.large) {
        proposed_U <- proposed_K <- Inf
      } else {
        par.mat.prop <- q
        l.c.vmsin.prop <- log_const_vmsin_all(par.mat.prop)

        lprior.prop <- sum((pmix.alpha-1) * log(pi.mix.1)) + sum(ldgamanum(q[1:2,], gam.loc, gam.scale)) - 0.5*sum((q[3,]/norm.var)^2)

        llik.prop <- llik_vmsin_full(data.rad, q, pi.mix.1, l.c.vmsin.prop, ncores)

        proposed_U <- -(llik.prop + lprior.prop)
        proposed_K <- sum(p^2) / 2
      }

      exp(current_U-proposed_U+current_K-proposed_K)
      # Accept or reject the state at end of trajectory, returning either
      # the position at the end of the trajectory or the initial position

      if (runif(1) < exp(current_U-proposed_U+current_K-proposed_K))
      {
        # return (q)  # accept
        # accpt = 1
        par.mat.1 <- signif(par.mat.prop, 8)
        lprior.1 <- signif(lprior.prop, 8)
        llik.1 <- signif(llik.prop, 8)
        lpd.1 <- signif(-proposed_U, 8)
        accpt.par.mat <- 1
        l.c.vmsin.1 <- signif(l.c.vmsin.prop, 8)
      }


      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.vmsin" = l.c.vmsin.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.par.mat" = accpt.par.mat)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.par.mat.all[iter] <- accpt.par.mat


      # tuning epsilon with first 20 draws
      if(autotune && iter == iter.tune && mean(accpt.par.mat.all[2:(iter.tune+1)]) < 0.6) {
        iter <- 2
        ntune <- ntune + 1
        if(epsilon.random) {
          epsilon_vec <- epsilon_vec/2
        } else {
          epsilon <- epsilon/2
        }
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter + 1

    }
  }

  if(ncomp > 1 && grepl(method, "rwmh")) # using rwmh
  {
    while(iter <= (n.iter+1)) {
      #----------------------------------------------------------------------------------
      #generating mixture proportions
      #----------------------------------------------------------------------------------
      pi.mix.old <- MC$pi.mix
      par.mat.old <- MC$par.mat
      l.c.vmsin.old <- MC$l.c.vmsin


      # Gibbs Sampler
      {
        post.wt <- mem_p_sin(data.rad, par.mat.old, pi.mix.old, l.c.vmsin.old, ncores)
        clus.ind[ , iter] <- cID(post.wt, ncomp, runif(n.data))
        n.clus <- tabulate(clus.ind[ , iter], nbins = ncomp)
        pi.mix.1 <- as.numeric(rdirichlet(1, (pmix.alpha + n.clus))) #new mixture proportions
        llik_new.pi <- llik_vmsin_full(data.rad, par.mat.old, pi.mix.1, l.c.vmsin.old, ncores)
      }


      #----------------------------------------------------------------------------------
      #generating presicion parameters
      #----------------------------------------------------------------------------------

      k1.1.old <- par.mat.old[1, ]
      k2.1.old <- par.mat.old[2, ]
      k3.1.old <- par.mat.old[3, ]

      k1.1.prop <- pmax(k1.1.old + rnorm(ncomp,0,propscale[1]), 1e-6)
      k2.1.prop <- pmax(k2.1.old + rnorm(ncomp,0,propscale[2]), 1e-6)
      k3.1.prop <- k3.1.old + rnorm(ncomp,0,propscale[3])

      prop.mat <- unname(rbind(k1.1.prop,k2.1.prop,k3.1.prop, par.mat.old[4:5, ]))
      l.c.vmsin.prop <- as.numeric(log_const_vmsin_all(prop.mat))

      llik_old <- llik_new.pi
      llik_prop <- llik_vmsin_full(data.rad, prop.mat, pi.mix.1, l.c.vmsin.prop, ncores)

      lprior_old <- MC$lprior
      lprior_prop <- sum((pmix.alpha-1) * log(pi.mix.1)) + sum(ldgamanum(prop.mat[1:2,], gam.loc, gam.scale)) - 0.5*sum((prop.mat[3,]/norm.var)^2)

      post.omg_old <- llik_old + lprior_old
      post.omg_prop <- llik_prop + lprior_prop

      if (runif(1) <  exp(post.omg_prop-post.omg_old) ) {
        k1.1 <- k1.1.prop
        k2.1 <- k2.1.prop
        k3.1 <- k3.1.prop
        accpt.kappa <- 1
        l.c.vmsin.1 <- signif(l.c.vmsin.prop, 8)
        llik_new.omg <- signif(llik_prop, 8)
        lprior.1 <- signif(lprior_prop, 8)
        par.mat_new.omg <- signif(prop.mat, 8)
      } else {
        k1.1 <- k1.1.old
        k2.1 <- k2.1.old
        k3.1 <- k3.1.old
        accpt.kappa <- 0
        l.c.vmsin.1 <- l.c.vmsin.old
        llik_new.omg <- llik_old
        lprior.1 <- lprior_old
        par.mat_new.omg <- par.mat.old
      }


      #----------------------------------------------------------------------------------
      #generating mu and nu
      #----------------------------------------------------------------------------------
      prop.mu <- prncp_reg(par.mat.old[4, ] + rnorm(ncomp,0,propscale[4]))
      prop.nu <- prncp_reg(par.mat.old[5, ] + rnorm(ncomp,0,propscale[5]))
      #----------------------------------------------------------------------------------
      prop.mat.mean <- unname(rbind(par.mat_new.omg[1:3,], prop.mu,prop.nu))

      llik_new.prop <- llik_vmsin_full(data.rad, prop.mat.mean, pi.mix.1, l.c.vmsin.1, ncores)

      if (runif(1) <  exp(llik_new.prop-llik_new.omg) ) {
        par.mat.1 <- signif(prop.mat.mean, 8)
        accpt.mu <- 1
        llik.1 <- llik_new.prop
      } else {
        par.mat.1 <- par.mat_new.omg
        accpt.mu <- 0
        llik.1 <- llik_new.omg
      }

      lpd.1 <- llik.1 + lprior.1

      MC <- list("par.mat" = par.mat.1, "pi.mix" = pi.mix.1,
                 "l.c.vmsin" = l.c.vmsin.1, "llik" = llik.1, "lprior" = lprior.1, "lpd" = lpd.1,
                 "accpt.kappa" = accpt.kappa, "accpt.mu" = accpt.mu)

      par.mat.all[,,iter] <- MC$par.mat
      pi.mix.all[,iter] <- MC$pi.mix
      llik.all[iter] <- llik.1
      lprior.all[iter] <- lprior.1
      lpd.all[iter] <- lpd.1
      accpt.kappa.all[iter] <- accpt.kappa
      accpt.mu.all[iter] <- accpt.mu

      # tuning propscale with first 20 draws
      if(autotune && iter == iter.tune && (mean(accpt.kappa.all[2:(iter.tune+1)]) < 0.6 ||
                                           mean(accpt.mu.all[2:(iter.tune+1)]) < 0.6)) {
        iter <- 2
        ntune <- ntune + 1
        propscale <- propscale/2
      }

      if(show.progress && ((iter-1) %% 25 == 0 || iter == n.iter + 1))
        utils::setTxtProgressBar(pb, iter)

      iter <- iter+1

    }
  }
  #******************************************************************************************
  if(grepl(method, "hmc")) {
    if(epsilon.random) {
      epsilon_ave <- mean(epsilon_vec)
    } else{
      epsilon_ave <- epsilon
    }
    if(L.random) {
      L_ave <- mean(L_vec)
    } else{
      L_ave <- L
    }
  }


  if(grepl(method, "rwmh")) {
    propscale_final <- propscale
  }

  if(show.progress) cat("\n")

  allpar_val <- array(1, dim = c(6, ncomp, n.iter+1))
  allpar_val[1, , ] <- pi.mix.all
  allpar_val[2:6, , ] <- par.mat.all

  allpar_name <- c("pmix", modelpar.names)
  dimnames(allpar_val)[[1]] <- c("pmix", modelpar.names)

  result <- list("par.value" = allpar_val, "par.name" = allpar_name, "llik" = llik.all,
                 "accpt.modelpar" = accpt.par.mat.all,
                 "accpt.kappa" = accpt.kappa.all, "accpt.mu" = accpt.mu.all,
                 "lpd" = lpd.all, "model" = curr.model, "method" = method, "clus.ind" = clus.ind,
                 "epsilon.random" = epsilon.random, "epsilon" = epsilon_ave,
                 "L.random" = L.random, "L" = L_ave,  "type" = "bi",
                 "propscale.final" = propscale_final, "data" = data.rad,
                 "gam.loc" = gam.loc, "gam.scale" = gam.scale, "pmix.alpha" = pmix.alpha, "norm.var" = norm.var,
                 "n.data" = n.data, "ncomp" = ncomp, "n.iter" = n.iter)
  class(result) <- "angmcmc"

  return(result)
}

vmsin_var_cor_singlepar_large <- function(kappa1, kappa2, kappa3, N) {
  # N <- 1e4
  dat <- rvmsin(N, kappa1, kappa2, kappa3, 0, 0)

  ave_sin1sin2 <- sum(sin(dat[, 1]) * sin(dat[, 2]))/N
  ave_cos1cos2 <- sum(cos(dat[, 1]) * cos(dat[, 2]))/N

  ave_sin1sq <- sum(sin(dat[, 1])^2)/N
  ave_cos1sq <- 1-ave_sin1sq
  ave_cos1 <- sum(cos(dat[, 1]))/N

  ave_sin2sq <- sum(sin(dat[, 2])^2)/N
  ave_cos2sq <- 1-ave_sin2sq
  ave_cos2 <- sum(cos(dat[, 2]))/N

  rho_js <- ave_sin1sin2/sqrt(ave_sin1sq * ave_sin2sq)
  # ifelse(ave_sin1sin2 >= 0, 1, -1) *
  # min(abs(ave_sin1sin2)/sqrt(ave_sin1sq * ave_sin2sq), 1)

  rho_fl <- rho_js *
    ave_cos1cos2/sqrt(ave_cos1sq * ave_cos2sq)
  # ifelse(ave_cos1cos2 >= 0, 1, -1) *
  # min(abs(ave_cos1cos2)/sqrt(ave_cos1sq * ave_cos2sq), 1)

  var1 <- min(1 - ave_cos1, 1)
  var2 <- min(1 - ave_cos2, 1)

  list(var1 = var1, var2 = var2, rho_fl = rho_fl, rho_js = rho_js)
}


vmsin_var_cor_singlepar <- function(kappa1, kappa2, kappa3, N) {
  if (kappa1 > 150 | kappa2 > 150 | abs(kappa3) > 150) {
    vmsin_var_cor_singlepar_large(kappa1, kappa2, kappa3, N)
  } else {
    sobol_grid <- sobol_2d_1e4_from_seed_1
    vmsin_var_cor_singlepar_cpp(kappa1, kappa2, kappa3,
                                sobol_grid, 1)
  }

}
