% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count.R
\name{count}
\alias{count}
\alias{count_resistant}
\alias{count_susceptible}
\alias{count_S}
\alias{count_SI}
\alias{count_I}
\alias{count_IR}
\alias{count_R}
\alias{count_all}
\alias{n_sir}
\alias{count_df}
\title{Count Available Isolates}
\usage{
count_resistant(..., only_all_tested = FALSE)

count_susceptible(..., only_all_tested = FALSE)

count_S(..., only_all_tested = FALSE)

count_SI(..., only_all_tested = FALSE)

count_I(..., only_all_tested = FALSE)

count_IR(..., only_all_tested = FALSE)

count_R(..., only_all_tested = FALSE)

count_all(..., only_all_tested = FALSE)

n_sir(..., only_all_tested = FALSE)

count_df(data, translate_ab = "name", language = get_AMR_locale(),
  combine_SI = TRUE)
}
\arguments{
\item{...}{One or more vectors (or columns) with antibiotic interpretations. They will be transformed internally with \code{\link[=as.sir]{as.sir()}} if needed.}

\item{only_all_tested}{(for combination therapies, i.e. using more than one variable for \code{...}): a \link{logical} to indicate that isolates must be tested for all antimicrobials, see section \emph{Combination Therapy} below.}

\item{data}{A \link{data.frame} containing columns with class \code{\link{sir}} (see \code{\link[=as.sir]{as.sir()}}).}

\item{translate_ab}{A column name of the \link{antimicrobials} data set to translate the antibiotic abbreviations to, using \code{\link[=ab_property]{ab_property()}}.}

\item{language}{Language of the returned text - the default is the current system language (see \code{\link[=get_AMR_locale]{get_AMR_locale()}}) and can also be set with the package option \code{\link[=AMR-options]{AMR_locale}}. Use \code{language = NULL} or \code{language = ""} to prevent translation.}

\item{combine_SI}{A \link{logical} to indicate whether all values of S, SDD, and I must be merged into one, so the output only consists of S+SDD+I vs. R (susceptible vs. resistant) - the default is \code{TRUE}.}
}
\value{
An \link{integer}
}
\description{
These functions can be used to count resistant/susceptible microbial isolates. All functions support quasiquotation with pipes, can be used in \code{summarise()} from the \code{dplyr} package and also support grouped variables, see \emph{Examples}.

\code{\link[=count_resistant]{count_resistant()}} should be used to count resistant isolates, \code{\link[=count_susceptible]{count_susceptible()}} should be used to count susceptible isolates.
}
\details{
These functions are meant to count isolates. Use the \code{\link[=resistance]{resistance()}}/\code{\link[=susceptibility]{susceptibility()}} functions to calculate microbial resistance/susceptibility.

The function \code{\link[=count_resistant]{count_resistant()}} is equal to the function \code{\link[=count_R]{count_R()}}. The function \code{\link[=count_susceptible]{count_susceptible()}} is equal to the function \code{\link[=count_SI]{count_SI()}}.

The function \code{\link[=n_sir]{n_sir()}} is an alias of \code{\link[=count_all]{count_all()}}. They can be used to count all available isolates, i.e. where all input antimicrobials have an available result (S, I or R). Their use is equal to \code{n_distinct()}. Their function is equal to \code{count_susceptible(...) + count_resistant(...)}.

The function \code{\link[=count_df]{count_df()}} takes any variable from \code{data} that has an \code{\link{sir}} class (created with \code{\link[=as.sir]{as.sir()}}) and counts the number of S's, I's and R's. It also supports grouped variables. The function \code{\link[=sir_df]{sir_df()}} works exactly like \code{\link[=count_df]{count_df()}}, but adds the percentage of S, I and R.
}
\section{Interpretation of SIR}{

In 2019, the European Committee on Antimicrobial Susceptibility Testing (EUCAST) has decided to change the definitions of susceptibility testing categories S, I, and R (\url{https://www.eucast.org/newsiandr}).

This AMR package follows insight; use \code{\link[=susceptibility]{susceptibility()}} (equal to \code{\link[=proportion_SI]{proportion_SI()}}) to determine antimicrobial susceptibility and \code{\link[=count_susceptible]{count_susceptible()}} (equal to \code{\link[=count_SI]{count_SI()}}) to count susceptible isolates.
}

\section{Combination Therapy}{

When using more than one variable for \code{...} (= combination therapy), use \code{only_all_tested} to only count isolates that are tested for all antimicrobials/variables that you test them for. See this example for two antimicrobials, Drug A and Drug B, about how \code{\link[=susceptibility]{susceptibility()}} works to calculate the \%SI:

\if{html}{\out{<div class="sourceCode">}}\preformatted{--------------------------------------------------------------------
                    only_all_tested = FALSE  only_all_tested = TRUE
                    -----------------------  -----------------------
 Drug A    Drug B   considered   considered  considered   considered
                    susceptible    tested    susceptible    tested
--------  --------  -----------  ----------  -----------  ----------
 S or I    S or I        X            X           X            X
   R       S or I        X            X           X            X
  <NA>     S or I        X            X           -            -
 S or I      R           X            X           X            X
   R         R           -            X           -            X
  <NA>       R           -            -           -            -
 S or I     <NA>         X            X           -            -
   R        <NA>         -            -           -            -
  <NA>      <NA>         -            -           -            -
--------------------------------------------------------------------
}\if{html}{\out{</div>}}

Please note that, in combination therapies, for \code{only_all_tested = TRUE} applies that:

\if{html}{\out{<div class="sourceCode">}}\preformatted{    count_S()    +   count_I()    +   count_R()    = count_all()
  proportion_S() + proportion_I() + proportion_R() = 1
}\if{html}{\out{</div>}}

and that, in combination therapies, for \code{only_all_tested = FALSE} applies that:

\if{html}{\out{<div class="sourceCode">}}\preformatted{    count_S()    +   count_I()    +   count_R()    >= count_all()
  proportion_S() + proportion_I() + proportion_R() >= 1
}\if{html}{\out{</div>}}

Using \code{only_all_tested} has no impact when only using one antibiotic as input.
}

\examples{
# example_isolates is a data set available in the AMR package.
# run ?example_isolates for more info.

# base R ------------------------------------------------------------
count_resistant(example_isolates$AMX) # counts "R"
count_susceptible(example_isolates$AMX) # counts "S" and "I"
count_all(example_isolates$AMX) # counts "S", "I" and "R"

# be more specific
count_S(example_isolates$AMX)
count_SI(example_isolates$AMX)
count_I(example_isolates$AMX)
count_IR(example_isolates$AMX)
count_R(example_isolates$AMX)

# Count all available isolates
count_all(example_isolates$AMX)
n_sir(example_isolates$AMX)

# n_sir() is an alias of count_all().
# Since it counts all available isolates, you can
# calculate back to count e.g. susceptible isolates.
# These results are the same:
count_susceptible(example_isolates$AMX)
susceptibility(example_isolates$AMX) * n_sir(example_isolates$AMX)

# dplyr -------------------------------------------------------------
\donttest{
if (require("dplyr")) {
  example_isolates \%>\%
    group_by(ward) \%>\%
    summarise(
      R = count_R(CIP),
      I = count_I(CIP),
      S = count_S(CIP),
      n1 = count_all(CIP), # the actual total; sum of all three
      n2 = n_sir(CIP), # same - analogous to n_distinct
      total = n()
    ) # NOT the number of tested isolates!

  # Number of available isolates for a whole antibiotic class
  # (i.e., in this data set columns GEN, TOB, AMK, KAN)
  example_isolates \%>\%
    group_by(ward) \%>\%
    summarise(across(aminoglycosides(), n_sir))

  # Count co-resistance between amoxicillin/clav acid and gentamicin,
  # so we can see that combination therapy does a lot more than mono therapy.
  # Please mind that `susceptibility()` calculates percentages right away instead.
  example_isolates \%>\% count_susceptible(AMC) # 1433
  example_isolates \%>\% count_all(AMC) # 1879

  example_isolates \%>\% count_susceptible(GEN) # 1399
  example_isolates \%>\% count_all(GEN) # 1855

  example_isolates \%>\% count_susceptible(AMC, GEN) # 1764
  example_isolates \%>\% count_all(AMC, GEN) # 1936

  # Get number of S+I vs. R immediately of selected columns
  example_isolates \%>\%
    select(AMX, CIP) \%>\%
    count_df(translate = FALSE)

  # It also supports grouping variables
  example_isolates \%>\%
    select(ward, AMX, CIP) \%>\%
    group_by(ward) \%>\%
    count_df(translate = FALSE)
}
}
}
\seealso{
\code{\link[=proportion]{proportion_*}} to calculate microbial resistance and susceptibility.
}
