




























#![doc(
    html_logo_url = "https://www.rust-lang.org/logos/rust-logo-128x128-blk.png",
    html_favicon_url = "https://www.rust-lang.org/favicon.ico",
    html_root_url = "https://rust-random.github.io/rand/"
)]
#![deny(missing_docs)]
#![deny(missing_debug_implementations)]
#![doc(test(attr(allow(unused_variables), deny(warnings))))]
#![cfg_attr(doc_cfg, feature(doc_cfg))]
#![no_std]

use core::convert::AsMut;
use core::default::Default;

#[cfg(feature = "std")] extern crate std;
#[cfg(feature = "alloc")] extern crate alloc;
#[cfg(feature = "alloc")] use alloc::boxed::Box;

pub use error::Error;
#[cfg(feature = "getrandom")] pub use os::OsRng;


pub mod block;
mod error;
pub mod impls;
pub mod le;
#[cfg(feature = "getrandom")] mod os;


















































/// # Example




/// #![allow(dead_code)]





























pub trait RngCore {
    
    
    
    
    
    fn next_u32(&mut self) -> u32;

    
    
    
    
    
    fn next_u64(&mut self) -> u64;

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn fill_bytes(&mut self, dest: &mut [u8]);

    
    
    
    
    
    
    
    
    
    
    
    fn try_fill_bytes(&mut self, dest: &mut [u8]) -> Result<(), Error>;
}























pub trait CryptoRng {}







/// # Example




/// #[allow(unused)]






pub trait CryptoRngCore: CryptoRng + RngCore {
    
    fn as_rngcore(&mut self) -> &mut dyn RngCore;
}

impl<T: CryptoRng + RngCore> CryptoRngCore for T {
    fn as_rngcore(&mut self) -> &mut dyn RngCore {
        self
    }
}







pub trait SeedableRng: Sized {
    
    
    
    
    
    
    
    
    
    
    /// # Implementing `SeedableRng` for RNGs with large seeds
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    type Seed: Sized + Default + AsMut<[u8]>;

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn from_seed(seed: Self::Seed) -> Self;

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn seed_from_u64(mut state: u64) -> Self {
        
        fn pcg32(state: &mut u64) -> [u8; 4] {
            const MUL: u64 = 6364136223846793005;
            const INC: u64 = 11634580027462260723;

            
            
            *state = state.wrapping_mul(MUL).wrapping_add(INC);
            let state = *state;

            
            let xorshifted = (((state >> 18) ^ state) >> 27) as u32;
            let rot = (state >> 59) as u32;
            let x = xorshifted.rotate_right(rot);
            x.to_le_bytes()
        }

        let mut seed = Self::Seed::default();
        let mut iter = seed.as_mut().chunks_exact_mut(4);
        for chunk in &mut iter {
            chunk.copy_from_slice(&pcg32(&mut state));
        }
        let rem = iter.into_remainder();
        if !rem.is_empty() {
            rem.copy_from_slice(&pcg32(&mut state)[..rem.len()]);
        }

        Self::from_seed(seed)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    fn from_rng<R: RngCore>(mut rng: R) -> Result<Self, Error> {
        let mut seed = Self::Seed::default();
        rng.try_fill_bytes(seed.as_mut())?;
        Ok(Self::from_seed(seed))
    }

    
    
    
    
    
    
    
    
    
    /// # Panics
    
    
    
    
    #[cfg(feature = "getrandom")]
    #[cfg_attr(doc_cfg, doc(cfg(feature = "getrandom")))]
    fn from_entropy() -> Self {
        let mut seed = Self::Seed::default();
        if let Err(err) = getrandom::getrandom(seed.as_mut()) {
            panic!("from_entropy failed: {}", err);
        }
        Self::from_seed(seed)
    }
}




impl<'a, R: RngCore + ?Sized> RngCore for &'a mut R {
    #[inline(always)]
    fn next_u32(&mut self) -> u32 {
        (**self).next_u32()
    }

    #[inline(always)]
    fn next_u64(&mut self) -> u64 {
        (**self).next_u64()
    }

    #[inline(always)]
    fn fill_bytes(&mut self, dest: &mut [u8]) {
        (**self).fill_bytes(dest)
    }

    #[inline(always)]
    fn try_fill_bytes(&mut self, dest: &mut [u8]) -> Result<(), Error> {
        (**self).try_fill_bytes(dest)
    }
}




#[cfg(feature = "alloc")]
impl<R: RngCore + ?Sized> RngCore for Box<R> {
    #[inline(always)]
    fn next_u32(&mut self) -> u32 {
        (**self).next_u32()
    }

    #[inline(always)]
    fn next_u64(&mut self) -> u64 {
        (**self).next_u64()
    }

    #[inline(always)]
    fn fill_bytes(&mut self, dest: &mut [u8]) {
        (**self).fill_bytes(dest)
    }

    #[inline(always)]
    fn try_fill_bytes(&mut self, dest: &mut [u8]) -> Result<(), Error> {
        (**self).try_fill_bytes(dest)
    }
}

#[cfg(feature = "std")]
impl std::io::Read for dyn RngCore {
    fn read(&mut self, buf: &mut [u8]) -> Result<usize, std::io::Error> {
        self.try_fill_bytes(buf)?;
        Ok(buf.len())
    }
}


impl<'a, R: CryptoRng + ?Sized> CryptoRng for &'a mut R {}


#[cfg(feature = "alloc")]
impl<R: CryptoRng + ?Sized> CryptoRng for Box<R> {}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_seed_from_u64() {
        struct SeedableNum(u64);
        impl SeedableRng for SeedableNum {
            type Seed = [u8; 8];

            fn from_seed(seed: Self::Seed) -> Self {
                let mut x = [0u64; 1];
                le::read_u64_into(&seed, &mut x);
                SeedableNum(x[0])
            }
        }

        const N: usize = 8;
        const SEEDS: [u64; N] = [0u64, 1, 2, 3, 4, 8, 16, -1i64 as u64];
        let mut results = [0u64; N];
        for (i, seed) in SEEDS.iter().enumerate() {
            let SeedableNum(x) = SeedableNum::seed_from_u64(*seed);
            results[i] = x;
        }

        for (i1, r1) in results.iter().enumerate() {
            let weight = r1.count_ones();
            
            
            
            assert!((20..=44).contains(&weight));

            for (i2, r2) in results.iter().enumerate() {
                if i1 == i2 {
                    continue;
                }
                let diff_weight = (r1 ^ r2).count_ones();
                assert!(diff_weight >= 20);
            }
        }

        
        assert_eq!(results[0], 5029875928683246316);
    }
}
