% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xtvfreg.R
\name{xtvfreg}
\alias{xtvfreg}
\title{Variance Function Panel Regression}
\usage{
xtvfreg(
  formula,
  data,
  group,
  panel_id,
  mean_vars,
  var_vars,
  weights = NULL,
  subset = NULL,
  converge = 1e-06,
  max_iter = 100,
  verbose = TRUE
)
}
\arguments{
\item{formula}{A formula specifying the dependent variable}

\item{data}{A data frame containing the variables}

\item{group}{A character string naming the grouping variable}

\item{panel_id}{A character string naming the panel identifier}

\item{mean_vars}{A character vector of variable names for the mean equation}

\item{var_vars}{A character vector of variable names for the variance equation}

\item{weights}{Optional character string naming the weight variable}

\item{subset}{Optional logical vector for subsetting}

\item{converge}{Convergence tolerance (default: 1e-6)}

\item{max_iter}{Maximum iterations (default: 100)}

\item{verbose}{Logical; print iteration history? (default: TRUE)}
}
\value{
An object of class "xtvfreg" containing:
\item{results}{List of results for each group}
\item{groups}{Vector of group values}
\item{call}{The matched call}
\item{convergence}{Convergence information for each group}
\item{variance_decomp}{Variance decomposition for each group}
\item{depvar}{Name of dependent variable}
\item{panel_id}{Name of panel identifier}
\item{group_var}{Name of grouping variable}
\item{mean_vars}{Variables in mean equation}
\item{var_vars}{Variables in variance equation}
}
\description{
Implements an iterative mean-variance panel regression estimator that allows
both the mean and variance of the dependent variable to be functions of
covariates. Based on Mooi-Reci and Liao (2025).
}
\examples{
# Example using nlswork subset data
data(nlswork_subset)

# Prepare the data
# Keep only observations with complete wage data and white/black race
analysis_data <- subset(nlswork_subset, 
                        !is.na(ln_wage) & 
                        !is.na(tenure) & 
                        race \%in\% 1:2)

# Create race grouping variable
analysis_data$race_group <- factor(analysis_data$race,
                                   levels = 1:2,
                                   labels = c("white", "black"))

# Create within and between components for tenure
analysis_data$m_tenure <- ave(analysis_data$tenure,
                              analysis_data$idcode,
                              FUN = function(x) mean(x, na.rm = TRUE))
analysis_data$d_tenure <- analysis_data$tenure - analysis_data$m_tenure

# Estimate varying effects model
result <- xtvfreg(
  formula = ln_wage ~ 1,
  data = analysis_data,
  group = "race_group",
  panel_id = "idcode",
  mean_vars = c("m_tenure", "d_tenure", "age"),
  var_vars = c("m_tenure"),
  verbose = FALSE
)

# View a summary of results
summary(result)

# Extract coefficients for white group if needed
coef(result, equation = "mean", group = "white")

}
\references{
Mooi-Reci, I., and Liao, T. F. (2025). Unemployment: a hidden source of wage
inequality? European Sociological Review, 41(3), 382-401.
\doi{10.1093/esr/jcae052}
}
