#' @noRd
.normalize_sex <- function(x, sex_map) {
  if (is.factor(x))  x <- as.character(x)
  if (is.character(x)) x <- tolower(x)
  if (is.logical(x))   x <- ifelse(x, "male", "female")
  male_vals   <- unlist(sex_map$male,   use.names = FALSE)
  female_vals <- unlist(sex_map$female, use.names = FALSE)
  out <- ifelse(x %in% male_vals, "male",
                ifelse(x %in% female_vals, "female", NA_character_))
  factor(out, levels = c("male","female"))
}

#' @noRd
.validate_exphaz_scalars <- function(ageDiag, time, data, max_age = 115L) {
  if (!is.numeric(ageDiag)) stop("`ageDiag` must be numeric (years).", call. = FALSE)
  if (!is.numeric(time))    stop("`time` must be numeric (years).", call. = FALSE)
  if (!is.data.frame(data)) stop("`data` must be a data.frame.", call. = FALSE)

  n <- NROW(data)
  if (length(time)    != n) stop("Length mismatch: `time` != nrow(data).", call. = FALSE)
  if (length(ageDiag) != n) stop("Length mismatch: `ageDiag` != nrow(data).", call. = FALSE)

  if (anyNA(ageDiag))           stop("`ageDiag` has NA.", call. = FALSE)
  if (anyNA(time))              stop("`time` has NA.", call. = FALSE)
  if (!all(is.finite(ageDiag))) stop("`ageDiag` has non-finite values.", call. = FALSE)
  if (!all(is.finite(time)))    stop("`time` has non-finite values.", call. = FALSE)
  if (any(time < 0))            stop("`time` contains negatives.", call. = FALSE)

  list(max_age = as.integer(max_age))
}


#' @noRd
.ensure_dimid <- function(ratetable) {
  dn <- attr(ratetable, "dimnames")
  if (is.null(dn))
    stop("`ratetable` lacks `dimnames` and cannot be auto-fixed.")
  # If names(dn) exist but dimid is missing, set it from names(dn)
  if (is.null(attr(ratetable, "dimid"))) {
    nm <- names(dn)
    if (is.null(nm) || any(!nzchar(nm)))
      stop("`ratetable` has `dimnames` but they are unnamed; cannot infer `dimid`.")
    attr(ratetable, "dimid") <- nm
  }
  ratetable
}


#' @noRd
.validate_rmap_dims <- function(rmap, data, ratetable,
                                required = c("age","sex","year"),
                                unknown_dim = c("warn","error","drop"),
                                coerce_year_to_date = TRUE,
                                sex_map = list(
                                  male   = c("male","m",1,"1",TRUE),
                                  female = c("female","f",0,"0",FALSE)
                                )) {
  unknown_dim <- match.arg(unknown_dim)

  if (!is.list(rmap) || is.null(names(rmap)) || any(!nzchar(names(rmap))))
    stop("`rmap` must be a named list mapping ratetable dimensions to column names in `data`.", call. = FALSE)

  RTdimid  <- attr(ratetable, "dimid")
  RTdimnm  <- attr(ratetable, "dimnames")
  if (is.null(RTdimid) || is.null(RTdimnm))
    stop("`ratetable` must carry 'dimid' and 'dimnames' attributes.", call. = FALSE)

  extra <- setdiff(names(rmap), RTdimid)
  if (length(extra)) {
    msg <- paste0("`rmap` has dims not present in ratetable: ", paste(extra, collapse = ", "), ".")
    switch(unknown_dim,
           warn  = warning(msg, call. = FALSE),
           error = stop(msg, call. = FALSE),
           drop  = { rmap[extra] <- NULL }
    )
  }


  miss_req <- setdiff(required, names(rmap))
  if (length(miss_req)) stop("`rmap` must include: ", paste(required, collapse = ", "), ".", call. = FALSE)


  cols <- unlist(rmap, use.names = FALSE)
  miss_cols <- setdiff(cols, names(data))
  if (length(miss_cols))
    stop("`data` is missing columns referenced by `rmap`: ",
         paste(miss_cols, collapse = ", "), ".", call. = FALSE)

  if ("year" %in% names(rmap)) {
    yv <- data[[ rmap$year ]]
    if (!inherits(yv, "Date")) {
      if (is.numeric(yv) && isTRUE(coerce_year_to_date)) {
        yy <- as.integer(yv)
        bad <- is.na(yy) | yy < 1800 | yy > 2200
        if (any(bad)) stop("`rmap$year` numeric but implausible; convert to Date yourself.", call. = FALSE)
        warning("Coercing numeric `rmap$year` to Date using January 1.", call. = FALSE)
        data[[ rmap$year ]] <- as.Date(sprintf("%d-01-01", yy))
      } else {
        stop("`rmap$year` must be a Date column (diagnosis date).", call. = FALSE)
      }
    }
  }

  if ("sex" %in% names(rmap)) {
    sx_raw <- data[[ rmap$sex ]]
    sx     <- .normalize_sex(sx_raw, sex_map)
    if (anyNA(sx)) {
      u <- utils::head(unique(sx_raw[is.na(sx)]), 5L)
      stop("`rmap$sex` contains unmapped values, e.g.: ", paste(u, collapse = ", "),
           ". Provide `sex_map` or recode.", call. = FALSE)
    }
    data$.__sex_norm <- sx
  }

  # Level checks for all categorical extra dims (not age/year/sex)
  known <- intersect(names(rmap), RTdimid)
  for (nm in known) {
    if (nm %in% c("age","year","sex")) next
    col <- rmap[[nm]]
    vals <- as.character(data[[col]])
    ix <- which(RTdimid == nm)
    rt_levels <- as.character(RTdimnm[[ix]])
    if (!is.null(rt_levels)) {
      bad <- setdiff(unique(vals[!is.na(vals)]), rt_levels)
      if (length(bad)) {
        b <- utils::head(bad, 10L)
        stop("Values of dim '", nm, "' in `data$", col, "` not in ratetable: ",
             paste(b, collapse = ", "), if (length(bad) > 10) " ...", ".", call. = FALSE)
      }
    }
    if (anyNA(vals)) stop("`data$", col, "` (dim '", nm, "') contains NA.", call. = FALSE)
  }

  list(rmap = rmap, data = data, RTdimid = RTdimid, RTdimnm = RTdimnm)
}

#' @noRd
exphaz_years <- function(ageDiag,
                         time,
                         data,
                         rmap,
                         ratetable,
                         ratedata = NULL,
                         add.rmap = NULL,
                         varlist = NULL,
                         temp01  = NULL,
                         scale = 365.241,
                         pophaz = c("classic","rescaled","corrected"),
                         only_ehazard = FALSE,
                         # internal controls
                         coerce_year_to_date = TRUE,
                         max_age = 115L,
                         clamp_to_table = TRUE,
                         unknown_dim = c("warn","error","drop"),
                         sex_map = list(
                           male   = c("male","m",1,"1",TRUE),
                           female = c("female","f",0,"0",FALSE)
                         )) {

  pophaz      <- match.arg(pophaz)
  unknown_dim <- match.arg(unknown_dim)

  ratetable <- .ensure_dimid(ratetable)
  scal <- .validate_exphaz_scalars(ageDiag, time, data, max_age)
  max_age <- scal$max_age

  if (missing(ratetable)) {
    if (is.null(ratedata))
      stop("Provide either `ratetable` or `ratedata`.", call. = FALSE)
    if (!is.matrix(ratedata) || ncol(ratedata) < 2L)
      stop("`ratedata` must be a matrix with >=2 columns (male, female).", call. = FALSE)
    if (is.null(rmap$sex))
      stop("With `ratedata`, `rmap$sex` must be provided (to choose male/female column).", call. = FALSE)

    sx <- .normalize_sex(data[[rmap$sex]], sex_map)
    if (anyNA(sx)) {
      u <- utils::head(unique(data[[rmap$sex]][is.na(sx)]), 5L)
      stop("`rmap$sex` has unmapped values, e.g.: ", paste(u, collapse = ", "), ".", call. = FALSE)
    }
    male  <- sx == "male"
    ageDC <- pmax(0, pmin(ageDiag + floor(time), nrow(ratedata) - 1L))

    ehazard <- ifelse(male,
                      ratedata[ageDC + 1L, 1L],
                      ratedata[ageDC + 1L, 2L])

    ehazardInt <- NULL
    if (!is.null(add.rmap)) {
      ratedataInt <- apply(ratedata, 2L, cumsum)
      ageDiagY    <- trunc(ageDiag)
      up <- ifelse(male,
                   (ageDC >= 1) * ratedataInt[ageDC, 1L] +
                     ratedata[ageDC + 1L, 1L] * (ageDC - trunc(ageDC)),
                   (ageDC >= 1) * ratedataInt[ageDC, 2L] +
                     ratedata[ageDC + 1L, 2L] * (ageDC - trunc(ageDC)))
      lo <- ifelse(male,
                   (ageDiagY >= 1) * ratedataInt[ageDiagY, 1L] +
                     ratedata[ageDiagY + 1L, 1L] * (ageDiag - trunc(ageDiag)),
                   (ageDiagY >= 1) * ratedataInt[ageDiagY, 2L] +
                     ratedata[ageDiagY + 1L, 2L] * (ageDiag - trunc(ageDiag)))
      ehazardInt <- up - lo
    }

    return(list(ehazard = ehazard, ehazardInt = ehazardInt))
  }

  chk <- .validate_rmap_dims(
    rmap, data, ratetable,
    required = c("age","sex","year"),
    unknown_dim = unknown_dim,
    coerce_year_to_date = coerce_year_to_date,
    sex_map = sex_map
  )
  rmap    <- chk$rmap
  data    <- chk$data
  RTdimid <- chk$RTdimid
  RTdimnm <- chk$RTdimnm

  ageDC <- ageDiag + time
  if (max(ageDC, na.rm = TRUE) > max_age)
    stop("Resulting age at follow-up exceeds ", max_age,
         " years. Check that `time` is in years.", call. = FALSE)

  year_end <- as.integer(format(data[[ rmap$year ]] + as.difftime(time * scale, units = "days"), "%Y"))

  newdata01 <- setNames(vector("list", length(RTdimid)), RTdimid)
  for (nm in RTdimid) {
    if      (nm == "age")  newdata01[[nm]] <- trunc(ageDC)
    else if (nm == "year") newdata01[[nm]] <- year_end
    else if (nm == "sex")  newdata01[[nm]] <- data$.__sex_norm
    else if (nm %in% names(rmap)) newdata01[[nm]] <- data[[ rmap[[nm]] ]]
    else stop("Ratetable expects dimension '", nm, "' but `rmap` did not provide it.", call. = FALSE)
  }
  newdata01 <- as.data.frame(newdata01, optional = TRUE, stringsAsFactors = FALSE)

  ageIx  <- which(RTdimid == "age")
  yearIx <- which(RTdimid == "year")
  maxAgeTab  <- suppressWarnings(max(as.integer(RTdimnm[[ageIx]])))
  maxYearTab <- suppressWarnings(max(as.integer(RTdimnm[[yearIx]])))
  if (isTRUE(clamp_to_table)) {
    overA <- newdata01$age  > maxAgeTab
    overY <- newdata01$year > maxYearTab
    if (any(overA, na.rm = TRUE)) {
      newdata01$age[overA] <- maxAgeTab
      message("Some age+time exceeded ratetable max age; clamped to ", maxAgeTab, ".")
    }
    if (any(overY, na.rm = TRUE)) {
      newdata01$year[overY] <- maxYearTab
      message("Some year+time exceeded ratetable max year; clamped to ", maxYearTab, ".")
    }
  } else {
    if (max(newdata01$age,  na.rm = TRUE) > maxAgeTab)
      stop("Computed age exceeds ratetable max age (", maxAgeTab, ").", call. = FALSE)
    if (max(newdata01$year, na.rm = TRUE) > maxYearTab)
      stop("Computed year exceeds ratetable max year (", maxYearTab, ").", call. = FALSE)
  }

  idx_list <- Map(function(x, lev) match(as.character(x), as.character(lev)),
                  newdata01, RTdimnm)
  Indic <- do.call(cbind, idx_list)
  if (anyNA(Indic)) {
    bad <- which(is.na(Indic), arr.ind = TRUE)[1, , drop = FALSE]
    stop("Ratetable lookup failed for some rows. Example: row ",
         bad[1, "row"], ", dimension '", colnames(Indic)[bad[1, "col"]],
         "'. Check levels and rmap.", call. = FALSE)
  }


  ehazard <- vapply(seq_len(nrow(Indic)), function(i) {
    ratetable[matrix(Indic[i, ], nrow = 1L)]
  }, numeric(1)) * scale

  dateDiag <- as.Date(data[[ rmap$year ]])
  if (isTRUE(only_ehazard))
    return(list(ehazard = ehazard, dateDiag = dateDiag))

  .data_exp <- data
  .data_exp$.__time_days <- as.numeric(time) * scale
  .data_exp$.__age_days  <- as.integer(ageDiag * scale)
  .data_exp$.__year_date <- as.Date(.data_exp[[rmap$year]])
  sx <- .data_exp[[rmap$sex]]
  if (is.character(sx)) {
    sx <- factor(tolower(sx), levels = c("male","female"))
  }
  .data_exp$.__sex <- sx


  rt_dims   <- attr(ratetable, "dimid") %||% names(attr(ratetable, "dimnames"))
  have_dept <- !is.null(rt_dims) && any(rt_dims == "dept")


rmap_list_call <- if (have_dept && !is.null(rmap$dept)) {
  substitute(list(age = .__age_days,
                  sex = .__sex,
                  year = .__year_date,
                  dept = .__dept))
} else {
  substitute(list(age = .__age_days,
                  sex = .__sex,
                  year = .__year_date))
}

survexp_call <- substitute(
  survival::survexp(
    Surv(.__time_days) ~ 1,
    data      = .data_exp,
    ratetable = ratetable,
    rmap      = RMAP,
    method    = "individual.h"
  ),
  list(RMAP = rmap_list_call)
)

ehazardInt <- eval(survexp_call, envir = environment())


  list(ehazard = ehazard, ehazardInt = ehazardInt, dateDiag = dateDiag)
}
