#' Extract population totals
#'
#' Calculate weights using three methods: IPW, Calibration, and Calibration+IPW
#'
#' Calculates weights intended to reduce the sampling bias present in All of Us. Three versions
#' of weights are calculated from different reweighting strategies: IPW, Calibration, and 
#' Calibration+IPW.
#'
#' @param sample   data.frame with representative sample
#' @param vars character vector with names of calibration variables
#' @param weight character vector with name of the weight variable
#'
#' @return list of data.frame with added (or replaced) weight columns and survey designs
#' @import dplyr
#' @importFrom stats reformulate
#' @importFrom purrr map_dfr
#' @importFrom stringr str_detect
#' @importFrom survey svytotal svydesign
#' @importFrom nonprobsvy nonprob control_sel
extract_totals <- function(sample, vars, weight) {
  
  Sample <- sample %>% select(all_of(vars), all_of(weight))
  
  # change variables to factors for svytotals
  Sample <- Sample %>%  
    mutate(across(all_of(vars), as.factor))
  
  ## define the survey design
  survey_design <- svydesign(
    ids = ~1,
    strata = NULL,
    weights = reformulate(weight),
    data = Sample,
    nest = TRUE
  )
  
  ### Calibration weights
  # Loop through calibration variables and compute totals
  cal_totals <- map_dfr(vars, ~{
    estm <- as.data.frame(svytotal(as.formula(paste0("~", .x)), survey_design))
    data.frame(
      VAR = .x,
      VARNAME = rownames(estm),
      CALTOT = estm$total,
      stringsAsFactors = FALSE
    )
  }) %>%
    left_join(tibble(VAR = vars), by = "VAR")
  
  # finalize cal_totals so it includes calibration totals only
  cal_totals <- cal_totals %>%
    filter(!str_detect(VARNAME, "0$")) %>%
    filter(!str_detect(VAR, "1$"))
  
  return(cal_totals)

  
  # # finalize cal_totals so it includes calibration totals only
  # cal_totals <- cal_totals %>%
  #   filter(!str_detect(VARNAME, "0$")) %>%
  #   filter(!str_detect(VAR, "1$"))%>% ## drop first category from each auxiliary variable
  #   select(-VARNAME)
  # 
  # pop_totals_A <- c(`(Intercept)` = cal_totals_grouped$CALTOT[1], setNames(cal_totals$CALTOT, cal_totals$VAR))
  # model_formula_A <- reformulate(cal_totals$VAR)
  # 
  # # Append 1 to the end of variables in pop_totals_A. Since they are factors, these are
  # # added when converting from model_formula_A into the model.matrix. Exclude intercept.
  # idx <- !grepl('Intercept', names(pop_totals_A))
  # names(pop_totals_A)[idx] <- paste0(names(pop_totals_A[idx]), '1')
  
  
    # pop_totals <- c(`(Intercept)` = cal_totals_grouped$CALTOT[1], setNames(cal_totals$CALTOT, cal_totals$VAR))
    # 
    # # Append 1 to the end of variables in pop_totals. Since they are factors, these are
    # # added when converting from model_formula_A into the model.matrix. Exclude intercept.
    # idx <- !grepl('Intercept', names(pop_totals))
    # names(pop_totals)[idx] <- paste0(names(pop_totals[idx]), '1')
    # 
    # model_formula <- reformulate(cal_totals$VAR)
    # 

    # return(list(
    #   pop_totals = pop_totals,
    #   model_formula = model_formula
    # ))
}
