% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/line_integral2d.R
\name{line_integral2d}
\alias{line_integral2d}
\title{Line integral of a scalar field along a planar curve, with optional 3D visualization}
\usage{
line_integral2d(
  f,
  r,
  a,
  b,
  plot = TRUE,
  n_curve = 400,
  n_curtain_v = 40,
  n_surf_x = 80,
  n_surf_y = 80,
  colorscale = "Viridis",
  surface_opacity = 0.65,
  show_surface_grid = TRUE,
  surface_grid_color = "rgba(80,80,80,0.25)",
  surface_grid_width = 1,
  curtain = 0.4,
  curtain_colorscale = c("white", "#2a9d8f"),
  curve_color = "black",
  curve_width = 3,
  lifted_color = "red",
  lifted_width = 2,
  h = NULL,
  method = c("adaptive", "simpson"),
  n_simpson = 1000,
  scene = list(aspectmode = "data", xaxis = list(title = "x"), yaxis = list(title = "y"),
    zaxis = list(title = "z")),
  bg = list(paper = "white", plot = "white")
)
}
\arguments{
\item{f}{A scalar field, given as \code{function(x, y)} returning a
numeric value.}

\item{r}{A parametric curve, given as \code{function(t)} that returns a
numeric vector of length two, interpreted as \code{c(x, y)}.}

\item{a, b}{Numeric scalars giving the parameter limits, with \code{b > a}.}

\item{plot}{Logical; if \code{TRUE}, produces a visualization using
\pkg{plotly}.}

\item{n_curve}{Number of sample points along the curve for plotting.}

\item{n_curtain_v}{Number of subdivisions in the vertical direction for
the curtain.}

\item{n_surf_x, n_surf_y}{Grid resolution for sampling the surface
\code{z = f(x, y)}.}

\item{colorscale}{Color scale for the surface. It may be a \pkg{plotly}
scale name, a single color, or a vector of colors defining a gradient.}

\item{surface_opacity}{Numeric opacity for the surface, between 0 and 1.}

\item{show_surface_grid}{Logical; if \code{TRUE}, overlays grid lines on
the surface.}

\item{surface_grid_color}{Character string giving the color of surface
grid lines.}

\item{surface_grid_width}{Numeric width of surface grid lines.}

\item{curtain}{Numeric value between 0 and 1 giving the opacity of the
vertical curtain. A value of zero disables the curtain.}

\item{curtain_colorscale}{Color scale for the curtain, using the same
formats accepted by \code{colorscale}.}

\item{curve_color}{Color for the curve drawn at height zero.}

\item{curve_width}{Width of the curve drawn at height zero.}

\item{lifted_color}{Color for the lifted curve whose height is
\code{f(x(t), y(t))}.}

\item{lifted_width}{Width of the lifted curve.}

\item{h}{Step size used for centered finite differences when computing
the derivative of \code{r(t)}. If \code{NULL}, a default is chosen
automatically.}

\item{method}{Integration method; may be \code{"adaptive"} (using
\code{stats::integrate}) or \code{"simpson"} (using a composite Simpson
rule).}

\item{n_simpson}{Number of subintervals for the Simpson method. This
value is automatically adjusted to be even.}

\item{scene}{List configuring the 3D scene in \pkg{plotly}.}

\item{bg}{List with background color settings for the figure, with entries
such as \code{paper} and \code{plot}.}
}
\value{
A list with:
\itemize{
\item \code{value}: the numeric value of the line integral,
\item \code{fig}: a \pkg{plotly} figure when \code{plot = TRUE}, or
\code{NULL} otherwise.
}
}
\description{
Computes a numerical line integral of a scalar field along a parametric
curve in the plane. The function integrates the quantity
\code{f(r(t)) * speed(t)}, where \code{r(t)} is the parametric curve
and \code{speed(t)} is the length of its derivative. Optionally, it
produces a 3D visualization that includes a surface representing
\code{z = f(x, y)}, the curve itself in the plane, a lifted version of
the curve showing \code{z = f(x(t), y(t))}, and a vertical curtain
over the curve.
}
\details{
The function evaluates the scalar field along the curve and computes an
approximation of the derivative of \code{r(t)} using centered finite
differences. The integral can be computed either through a built-in
adaptive integration routine or via a composite Simpson rule with a user
specified number of subintervals.

For visualization, the function:
\itemize{
\item builds a rectangular surface \code{z = f(x, y)} using only the
endpoints of the curve,
\item plots the curve in the plane,
\item plots a lifted copy of the curve where the height is given by
the scalar field,
\item optionally constructs a vertical curtain over the curve by
extruding the height values.
}
}
\examples{
f <- function(x, y) x^2 + y^2
r <- function(t) c(t*cos(t), t*sin(3*t))
line_integral2d(f, r, a = 0, b = 2*pi, plot = FALSE)

}
