#' uniVal
#'
#' @description
#' A function for unifying the correlations of multiple tests with a criterion (i.e., the validity)
#'
#' @param val Vector of validities of the tests
#' @param b Vector of weightings for the validities; must match order of `val`. Default: 1
#' @param corm Correlation matrix of the tests; must be square with 1s on the diagonal
#'
#' @return
#' A single value of the class `numeric` representing the unified validity estimate across multiple tests
#'
#' @examples
#' val = c(0.20, 0.40, 0.30)
#' b = c(1, 1, 1)
#' corm = matrix(c(1, 0.50, 0.35, 0.50, 1, 0.75, 0.35, 0.70, 1), nrow = 3, ncol = 3)
#' uniVal(val, b, corm)
#'
#' @importFrom utils combn
#'
#' @export
uniVal <- function(val, b = NULL, corm) {
  num_val = length(val)
  if (is.null(b)) b = rep(1, num_val)
  if (length(b) != num_val) {
    stop("val and b must have the same length.")
  }
  if (!is.matrix(corm) || nrow(corm) != num_val || ncol(corm) != num_val) {
    stop("corm must be a square matrix with dimensions matching the length of validities.")
  }
  numerator = sum(b * val)
  combs = combn(c(1:length(val)), 2, simplify = FALSE)
  combs = do.call(rbind, lapply(combs, function(x) data.frame(v1 = x[1], v2 = x[2])))
  weighted_r_sum = 0
  for(i in seq_along(combs$v1)) {
    weighted_r_sum = weighted_r_sum + 2 * b[combs[i, 1]] * b[combs[i, 2]] * corm[combs[i, 1], combs[i, 2]]
  }
  denominator = sqrt(sum(b^2) + weighted_r_sum)
  validity = numerator / denominator
  return(validity)
}
