#' Get point estimates from model
#'
#' @description
#' Generate point estimates from a dose response curve.
#'
#' @param dataset A dataframe used to generate `model`.
#' @param model A drm model, generated by `modelcomp()` or `drm()`.
#' @param EDx Numeric. The effective dose level to estimate. Defaults
#'  to 0.5.
#' @param interval Character. Method for calculating confidence intervals
#'  of EDx. Choices: `"tfls"`, `"fls"`, `"delta"`, `"none"`. Defaults
#'  to "tfls". See `drc::ED()` for more information.
#' @param level Numeric. Confidence level for the interval calculation.
#'  Defaults to 0.95.
#' @param type Character. Indicates if EDx is `"absolute"` or
#'  `"relative"` to the curve. Defaults to absolute.
#' @param list_obj Optional. List object used for integration with
#'  [runtoxdrc()].
#' @param quiet Logical. Indicates if results should be hidden. Defaults
#'  to FALSE.
#' @param EDargs.supplement List. Optional user-supplied list of additional arguments
#'  compatible with `drc::ED()`.
#'
#' @returns A dataframe of the point estimates. If `list_obj` is provided, returns
#'  this within a list as `list_obj$effectmeasure`.
#'
#' @importFrom drc ED
#'
#' @export

getECx <- function(
  dataset,
  model,
  EDx = 0.5,
  interval = c("tfls", "fls", "delta", "none"),
  level = 0.95,
  type = c("absolute", "relative"),
  quiet = FALSE,
  EDargs.supplement = list(),
  list_obj = NULL
) {
  interval <- match.arg(interval)
  type <- match.arg(type)

  # Build argument list for ED()
  EDargs <- c(
    list(
      object = model,
      respLev = EDx,
      interval = interval,
      level = level,
      type = type,
      display = !quiet
    ),
    EDargs.supplement
  )

  # Safely run ED() and capture output
  EDvals <- tryCatch(
    suppressWarnings(do.call(ED, EDargs)),
    error = function(e) return(NULL)
  )

  # Handle failed fit or NA result
  if (is.null(EDvals)) {
    n_ecx <- length(EDx)
    return(
      data.frame(
        `Effect Measure` = paste0("EC", EDx * 100),
        Estimate = rep(NA, n_ecx),
        `Std. Error` = rep(NA, n_ecx),
        Lower = rep(NA, n_ecx),
        Upper = rep(NA, n_ecx)
      )
    )
  }

  ## --- Clean and format output --- ##

  # Convert to data frame, preserving numeric types
  EDdf <- as.data.frame(EDvals, stringsAsFactors = FALSE)

  # Add ECx labels from rownames
  EDdf$`Effect Measure` <- paste0(
    "EC",
    gsub("\\.0+$", "", as.numeric(sub(".*:", "", rownames(EDdf))) * 100)
  )

  # Reorder columns (Effect Measure first)
  EDdf <- EDdf[, c(
    "Effect Measure",
    names(EDdf)[names(EDdf) != "Effect Measure"]
  )]

  # Ensure missing columns (Lower/Upper) exist if interval = "none"
  expected_cols <- c("Estimate", "Std. Error", "Lower", "Upper")
  missing_cols <- setdiff(expected_cols, names(EDdf))
  if (length(missing_cols) > 0) {
    EDdf[missing_cols] <- NA
  }

  rownames(EDdf) <- NULL

  # Print if requested
  if (!quiet) {
    print(EDdf)
  }

  ## --- Combine or return --- ##
  if (is.null(list_obj)) {
    return(EDdf)
  } else {
    if (!is.list(list_obj)) {
      stop("Provided list_obj must be a list.")
    }
    list_obj$effectmeasure <- EDdf
  }
  return(list_obj)
}
