################################################################################
# Performs two one-sided tests for Ho: distribution of paired differences 
# symmetrically distributed and centered on zero 
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9 
# Date: Feb 06, 2026

equivalence.types <- c("delta", "epsilon")

tost.sign.rank <- function(
    x,
    y,
    eqv.type   = equivalence.types,
    eqv.level  = 1,
    upper      = NA,
    ccontinuity= FALSE,
    conf.level = 0.95, 
    x.name     = "",
    y.name     = "",
    relevance  = TRUE) {
  # Confirm variable names
  if (x.name=="") {
    x.name <- gsub(".*\\$","",deparse(substitute(x)))
    }
  if (y.name=="") {
    y.name <- gsub(".*\\$","",deparse(substitute(y)))
    }
  # Validate x and y same length
  x <- x[complete.cases(x)]
  y <- y[complete.cases(y)]
  if (length(x) != length(y)) {
    rlang::abort(message="x and y must have equal numbers of complete values")
    }
  # Validate and sanitize eqv.type
  if (length(eqv.type) > 1) {
    eqv.type <- "epsilon"
    }
  eqv.type <- tolower(eqv.type)
  if ( eqv.type != "delta" & eqv.type != "epsilon" ) {
    rlang::abort(message="option eqv.type() must be either delta or epsilon")
    }
  if (eqv.type == "delta") {
    eqv.type.display <- "\U0394"
    }
  if (eqv.type == "epsilon") {
    eqv.type.display <- "\U03B5"
    }
  # Validate eqv.level and upper
  if ( eqv.level <= 0 && is.na(upper) ) {
    rlang::abort(message="eqv.level incorrectly specified\n the tolerance must be a positive real value")
    }
  if ( eqv.level <= 0 && !is.na(upper) && upper>0 && eqv.type=="epsilon" ) {
    lower <- abs(eqv.level)
    }
  # Validate upper
  if (!is.na(upper) && upper<0) {
    rlang::abort(message="upper incorrectly specified\n the tolerance must be a positive real value for asymmatric equivalence intervals")
    }
  if (is.na(upper) || upper == abs(eqv.level)) {
    upper <- abs(eqv.level)
    lower <- abs(eqv.level)
    if (eqv.type == "delta") {
      eqv.level.display <- sprintf("%-7.0f",eqv.level)
      }
    if (eqv.type == "epsilon") {
      eqv.level.display <- sprintf("%-8.4f",eqv.level)
      }
    }
  if (!is.na(upper) && upper>0) {
    upper <- abs(upper)
    lower <- abs(eqv.level)
    if (eqv.type == "delta") {
      eqv.type.lower.display <- "\U0394l"
      eqv.type.upper.display <- "\U0394u"
      }
    if (eqv.type == "epsilon") {
      eqv.type.lower.display <- "\U03B5l"
      eqv.type.upper.display <- "\U03B5u"
      }
    upper.display <- trimws(sprintf("%#-8.5g",upper))
    lower.display <- trimws(sprintf("%#-8.5g",-1*lower))
    }
  # Validate conf.level and create alpha
  alpha <- 1 - conf.level
  if (alpha <= 0 | alpha >= 1) {
    rlang::abort(message="conf.level must be >0 and <1")
    }
  # Create conclusion strings
  pos.decision <- "Reject"
  neg.decision <- "Reject"
  # Create table.title, mid.bar, and bottom.bar
  table.title <- paste0(
                   pad.spaces(8),
                   "Sign \U2502",
                   pad.spaces(6),
                   "Obs",
                   pad.spaces(4),
                   "Sum ranks",
                   pad.spaces(4),
                   "Expected", collapse="")
  mid.bar     <- paste0(pad.horizontal(13),"\U253C",pad.horizontal(33), collapse="")
  bottom.bar  <- paste0(pad.spaces(21),pad.horizontal(10), collapse="")
  cont.cor.notice <- "\n"
  if (ccontinuity) {
    cont.cor.notice <- "\nUsing continuity correction\n"
    }
  # Generate various statistics, format some for display
  d <- x - y
  n <- length(x)
  n.display <- sprintf("%8.0f",n)
  sigma.sq.T <- n*(n+1)*(2*n+1)/24
  sigma.sq.T.display <- sprintf("%10.2f",sigma.sq.T)
  ranks.d <- rank(abs(d),ties.method="average",na.last=NA)
  sign.ranks.d <- sign(d)*ranks.d
  n.positive.ranks <- length(d[d>0])
  n.positive.ranks.display <- sprintf("%8.0f",n.positive.ranks)
  n.negative.ranks <- length(d[d<0])
  n.negative.ranks.display <- sprintf("%8.0f",n.negative.ranks)
  n.zero.ranks <- length(d[d==0])
  n.zero.ranks.display <- sprintf("%8.0f",n.zero.ranks)
  s.sq <- sum((ranks.d[d!=0])^2)/4
  s.sq.display <- pad.left(sprintf("%.8g",s.sq),10)
  se <- 2 * sqrt(s.sq)
  zero.adjust <- -n.zero.ranks*(n.zero.ranks+1)*(2*n.zero.ranks+1)/24
  zero.adjust.display <- pad.left(sprintf("%.4g",zero.adjust),10)
  ties.adjust <- s.sq - sigma.sq.T - zero.adjust
  ties.adjust.display <- pad.left(sprintf("%.4g",ties.adjust),10)
  sum.positive.ranks <- sum(ranks.d[d>0])
  sum.positive.ranks.display <- sprintf("%10.1f",sum.positive.ranks)
  sum.negative.ranks <- sum(ranks.d[d<0])
  sum.negative.ranks.display <- sprintf("%10.1f",sum.negative.ranks)
  sum.zero.ranks <- n.zero.ranks*(n.zero.ranks+1)/2
  sum.zero.ranks.display <- sprintf("%10.0f",sum.zero.ranks)
  sum.all.ranks <- sum(c(1:n))
  sum.all.ranks.display <- sprintf("%10.1f",sum.all.ranks)
  expected.ranks <- (sum.positive.ranks+sum.negative.ranks)/2
  expected.ranks.display <- sprintf("%10.1f",expected.ranks)
  T_stat <- min(sum.positive.ranks,sum.negative.ranks)
  cont.cor <- 0
    if (ccontinuity) {
    cont.cor <- 0.5
    }
  z.pos <- (sign(sum.positive.ranks - sum.negative.ranks) * (abs(sum.positive.ranks - sum.negative.ranks)-cont.cor))/se
  z.pos.display <- sprintf("%-7.3f", z.pos)
  p.pos <- 2*pnorm(abs(z.pos), lower.tail=FALSE)
  p.pos.display <- format.extreme.p.vals(p.pos)
  # Rejection decision for two-sided test
  if ( 2*pnorm(abs(z.pos), lower.tail = FALSE) > alpha ) {
    pos.decision <- "Fail to reject"
    }
  # Negativist test statistics
  if ( eqv.type == "delta" ) {
    z1 <- (upper - (sign(sum.positive.ranks - sum.negative.ranks) * (abs(sum.positive.ranks - sum.negative.ranks) - cont.cor)))/se
    z2 <- ((sign(sum.positive.ranks - sum.negative.ranks) * (abs(sum.positive.ranks - sum.negative.ranks) - cont.cor)) + lower)/se
    }
  if ( eqv.type == "epsilon" ) {
    z1 <- upper - ( (sign(sum.positive.ranks-sum.negative.ranks)*(abs(sum.positive.ranks-sum.negative.ranks) - cont.cor))/se)
    z2 <- ( (sign(sum.positive.ranks-sum.negative.ranks)*(abs(sum.positive.ranks-sum.negative.ranks) - cont.cor))/se) + lower
    }
  z1.display <- sprintf("%-8.4g",z1)
  z2.display <- sprintf("%-8.4g",z2)
  neg.z.stats.out <- paste0(pad.spaces(8), "z1 = ", z1.display, pad.spaces(16), "z2 = ", z2.display, collapse="")
  # p values and rejection decision for two one-sided tests
  p1 <- pnorm(z1, lower.tail = FALSE)
  p2 <- pnorm(z2, lower.tail = FALSE)    
  p1.display <- format.extreme.p.vals(p1)
  p2.display <- format.extreme.p.vals(p2)
  neg.p.vals.out <- paste0(pad.spaces(3), "Pr(Z > z1) ", p1.display, pad.spaces(10), "Pr(Z > z2) ", p2.display, collapse="")
  if (p1 > alpha | p2 > alpha) {
    neg.decision <- "Fail to reject"
    }
  if (relevance) {
    rlang::inform(message="\nRelevance signed-rank test\n")
    rlang::inform(message="Signed-rank test for the distribution of differences in paired data being")
    rlang::inform(message="different from one that is symmetrical & centered on zero\n")
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(pad.spaces(4),"Positive \U2502 ",n.positive.ranks.display,pad.spaces(3),sum.positive.ranks.display,pad.spaces(1),expected.ranks.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(4),"Negative \U2502 ",n.negative.ranks.display,pad.spaces(3),sum.negative.ranks.display,pad.spaces(1),expected.ranks.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(8),"Zero \U2502 ",n.zero.ranks.display,pad.spaces(3),sum.zero.ranks.display,pad.spaces(1),sum.zero.ranks.display,sep=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(pad.spaces(9),"All \U2502 ",n.display,pad.spaces(3),sum.all.ranks.display,pad.spaces(1),sum.all.ranks.display,"\n",sep=""))
    rlang::inform(message=paste0("Unadjusted variance  ",sigma.sq.T.display,sep=""))
    rlang::inform(message=paste0("Adjustment for ties  ",ties.adjust.display,sep=""))
    rlang::inform(message=paste0("Adjustment for zeros ",zero.adjust.display,sep=""))
    rlang::inform(message=bottom.bar)
    rlang::inform(message=paste0("Adjusted variance",pad.spaces(4),s.sq.display,"\n",sep=""))
    rlang::inform(message=paste0("Ho: ",x.name," - ",y.name," has a symmetric distribution centered on zero",sep=""))
    rlang::inform(message=paste0("Ha: ",x.name," - ",y.name," has an asymmetric distribution,\n    a distribution not centered on zero, or\n    has a distribution which is asymmetric and not centered on zero",sep=""))
    rlang::inform(message=cont.cor.notice)
    rlang::inform(message=paste0(pad.spaces(11),"z = ",z.pos.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(1),"Pr(|Z|>|z|) ",trimws(p.pos.display),"\n",sep=""))
    }
  rlang::inform(message=paste0("\nSigned-rank test for the distribution of paired or matched data being"))
  rlang::inform(message=paste0("equivalent to one that is symmetrical & centered on zero\n"))
  rlang::inform(message=table.title)
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(pad.spaces(4),"Positive \U2502 ",n.positive.ranks.display,pad.spaces(3),sum.positive.ranks.display,pad.spaces(1),expected.ranks.display,sep=""))
  rlang::inform(message=paste0(pad.spaces(4),"Negative \U2502 ",n.negative.ranks.display,pad.spaces(3),sum.negative.ranks.display,pad.spaces(1),expected.ranks.display,sep=""))
  rlang::inform(message=paste0(pad.spaces(8),"Zero \U2502 ",n.zero.ranks.display,pad.spaces(3),sum.zero.ranks.display,pad.spaces(1),sum.zero.ranks.display,sep=""))
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(pad.spaces(9),"All \U2502 ",n.display,pad.spaces(3),sum.all.ranks.display,pad.spaces(1),sum.all.ranks.display,"\n",sep=""))
  rlang::inform(message=paste0("Unadjusted variance",pad.spaces(2),sigma.sq.T.display,sep=""))
  rlang::inform(message=paste0("Adjustment for ties",pad.spaces(2),ties.adjust.display,sep=""))
  rlang::inform(message=paste0("Adjustment for zeros ",zero.adjust.display,sep=""))
  rlang::inform(message=bottom.bar)
  rlang::inform(message=paste0("Adjusted variance    ",s.sq.display,"\n",sep=""))
  if (eqv.type == "delta") {
    if (upper == lower) {
      rlang::inform(message=paste0(pad.spaces(9),eqv.type.display," = ",pad.right(upper.display,8),eqv.type.display," expressed in units of signed ranks (T)",sep=""))
      }
    if (upper != lower) {
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.lower.display," = ",pad.right(lower.display,8),eqv.type.lower.display," expressed in units of signed ranks (T)",sep=""))
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.upper.display," =  ",pad.right(upper.display,7),eqv.type.upper.display," expressed in units of signed ranks (T)",sep=""))
      }
    critical.value <- (se * qnorm(alpha,lower.tail=FALSE))
    critical.value.display <- sprintf("%-6.4g",critical.value)
    if (upper == lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if ",eqv.type.display," \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper != lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |",eqv.type.lower.display,"| \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper != lower & upper <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if ",eqv.type.upper.display," \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper == lower) {
      rlang::inform(message=paste0("\nHo: |T-E(T)| \U2265 \U0394:")) 
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-[T-E(T)] \U2264 0",pad.spaces(10),"Ho2: [T-E(T)]+\U0394 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-[T-E(T)] > 0",pad.spaces(10),"Ha2: [T-E(T)]+\U0394 > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    if (upper != lower) {
      rlang::inform(message=paste0("\nHo: [T-E(T)] \U2264 \U0394l, or [T-E(T)] \U2265 \U0394u:"))
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-[T-E(T)] \U2264 0",pad.spaces(9),"Ho2: [T-E(T)]-\U0394l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-[T-E(T)] > 0",pad.spaces(9),"Ha2: [T-E(T)]-\U0394l > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }
  if (eqv.type == "epsilon") {
    if (upper == lower) {
      rlang::inform(message=paste0(pad.spaces(9),"\U03B5 = ",pad.right(upper.display,8),eqv.type.display," expressed in units of the z distribution",sep=""))
      }
    if (upper != lower) {
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.lower.display," = ",pad.right(lower.display,8),eqv.type.lower.display," expressed in units of the z distribution",sep=""))
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.upper.display," =  ",pad.right(upper.display,7),eqv.type.upper.display," expressed in units of the z distribution",sep=""))
      }
    critical.value <- qnorm(alpha, lower.tail=FALSE)
    critical.value.display <- sprintf("%-6.4g",critical.value)
    if ( (upper == lower) & (lower <= critical.value)) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if ",eqv.type.display," \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper != lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |",eqv.type.lower.display,"| \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper != lower & upper <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if ",eqv.type.upper.display," \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.sign.rank).",sep=""))
      }
    if (upper == lower) {
      rlang::inform(message=paste0("\nHo: |Z| \U2265 \U03B5:"))
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5-Z \U2264 0",pad.spaces(17),"Ho2: Z+\U03B5 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5-Z > 0",pad.spaces(17),"Ha2: Z+\U03B5 > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      } 
    if (upper != lower) {
      rlang::inform(message=paste0("\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:"))
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u-Z \U2264 0",pad.spaces(16),"Ho2: Z-\U03B5l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u-Z > 0",pad.spaces(16),"Ha2: Z-\U03B5l > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }
  # Output combined tests results if relevance test is requested
  if (relevance) {
    # Format Delta or epsilon to remove trailing zeros
    if (upper == lower) {
      if (eqv.type == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",alpha,", and \U0394 = ",trimws(upper.display),":",sep=""))
        }
      if (eqv.type == "epsilon") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",alpha,", and \U03B5 = ",trimws(upper.display),":",sep=""))
        }
      }
    if (upper != lower) {
      if (eqv.type == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",alpha," \U0394l = ",trimws(lower.display),", and \U0394u = ",trimws(upper.display),sep=""))
        }
      if (eqv.type == "epsilon") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",alpha," \U03B5l = ",trimws(lower.display),", and \U03B5u = ",trimws(upper.display),sep=""))
        }
      }
    rlang::inform(message=paste0("  Ho test for difference from symmetric & centered on 0: ",pos.decision,sep=""))
    rlang::inform(message=paste0("  Ho test for equivalence to symmetric & centered on 0:  ",neg.decision,sep=""))
    if ((pos.decision == "Reject") & (neg.decision == "Reject")) {
      rel.conclusion <- "Trivial difference (overpowered test)"
      }
    if ((pos.decision == "Reject") & (neg.decision == "Fail to reject")) {
      rel.conclusion <- "Relevant difference"
      }
    if ((pos.decision == "Fail to reject") & (neg.decision == "Reject")) {
      rel.conclusion <- "Equivalence"
      }
    if ((pos.decision == "Fail to reject") & (neg.decision == "Fail to reject")) {
      rel.conclusion <- "Indeterminate (underpowered test)"
      }
    rlang::inform(message=paste0("\nConclusion from combined tests: ",rel.conclusion,sep=""))
    }
###############################################################################
# Program end. Close up shop and return things.                               #
###############################################################################

  out <- list() 
  # Prepare return stuff for two-sample test
  if (!relevance) {
    out$statistics <- c(z1,z2)
    names(out$statistics) <- c("z1","z2")
    out$p.values <- c(p1,p2)
    names(out$p.values) <- c("p1","p2")
    }
   else {
    out$statistics <- c(z1,z2,z.pos)
    names(out$statistics) <- c("z1","z2","z")
    out$p.values <- c(p1,p2,p.pos)
    names(out$p.values) <- c("p1","p2","p")
    }
  out$signed_rank_sums <- c(sum.positive.ranks, abs(sum.negative.ranks), expected.ranks)
  names(out$signed_rank_sums) <- c("pos. rank sum","abs(neg. rank sum)","expected under Ho+")
  out$sample_size <- n
  names(out$sample_size) <- c("n")
  out$counts <- c(n.negative.ranks, n.positive.ranks, n.zero.ranks)
  names(out$counts) <- c("N_neg","N_pos","N_tie")
  out$var_adj <- s.sq
  names(out$var_adj) <- "adjusted \U03C3\U00B2"
  if (eqv.type=="delta") {
    if (upper==abs(eqv.level)) {
      out$threshold <- lower
      names(out$threshold) <- "\U394"
      }
     else {
       out$threshold <- c(upper, lower)
       names(out$threshold) <- c("\U0394u", "\U0394l")
      }
    }
   else {
    if (upper==abs(eqv.level)) {
      out$threshold <- lower
      names(out$threshold) <- "\U3B5"
      }
     else {
       out$threshold <- c(upper, lower)
       names(out$threshold) <- c("\U03B5u", "\U03B5l")
      }
    }
  if(relevance) {
    out$conclusion <- rel.conclusion
    names(out$conclusion) <- "relevance conclusion"
    }

  invisible(out)
  }
