################################################################################
# Performs two one-sided tests for proportion equivalence
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9
# Date: Feb 06, 2026

equivalence.types <- c("delta", "epsilon")

continuity.correction.methods <- c("none", "yates", "ha")

tost.pri <- function(
    n1=NA, 
    obs1=NA, 
    n2=NA, 
    obs2=NA, 
    count=FALSE, 
    eqv.type = equivalence.types, 
    eqv.level=1, 
    upper=NA, 
    ccontinuity=continuity.correction.methods,
    conf.level = 0.95, 
    x.name="x",
    y.name="y",
    relevance=TRUE) {
  # Calculate alpha
  alpha <- (1 - conf.level)
  # default eqv.type is "delta" if unspecified
  if (length(eqv.type)>1) {
    eqv.type <- "delta"
    }
  # make lower case eqv.type and ccontinuity options
  eqv.type <- tolower(eqv.type)
  ccontinuity <- tolower(ccontinuity)
  # default ccontinuity is "none" if unspecified
  if (length(ccontinuity) > 1) {
    ccontinuity <- "none"
    }
  # Sanitize the upper option if unspecified (for later tests)
  if (is.na(upper)) {
    upper <- abs(eqv.level)
    }
  # Validate number of numerical arguments
  if ( !(is.numeric(n1) & is.numeric(obs1) & is.numeric(obs2)) & !(is.numeric(n1) & is.numeric(obs1) & is.numeric(n2) & is.numeric(obs2)) ) {
    rlang::abort(message="tost.pri must specify either n1, obs1, and obs2 (for one-sample tests), or\nn1, obs1, n2, and obs2 (for two-sample tests)")
    }
  # Validate that n1 is a positive integer
  if (n1%%1 != 0 | n1 < 1) {
    rlang::abort(message="n1 must be a positive integer")
    }
  # Validate that obs1 is a proportion or a count
  if (!count & (obs1 < 0 | obs1 > 1)) {
    rlang::abort(message="obs1 must be a proportion (0\u2264obs1\u22641) if count=FALSE")
    }
  if (count & (obs1%%1 != 0 | obs1<0)) {
    rlang::abort(message="obs1 must be a non-negative integer if count=TRUE")
    }
  # Validate that n2 is a positive integer
  if (!is.na(n2) & (n2%%1 != 0 | n2 < 1)) {
    rlang::abort(message="n2 must be a positive integer")
    }
  # Validate that obs2 is a proportion or a count (must be a proportion in a one-sample test)
  if (!count & (obs2 < 0 | obs2 > 1)) {
    rlang::abort(message="obs2 must be a proportion (0\u2264obs2\u22641) if count=FALSE")
    }
  if (count & !is.na(n2) & (obs2%%1 != 0 | obs2<0)) {
    rlang::abort(message="obs2 must be a non-negative integer if count=TRUE")
    }
  if (count & is.na(n2) & (obs2%%1==0 & obs2 != 0 & obs2 != 1)) {
    rlang::abort(message="obs2 must be a proportion (0\u2264obs2\u22641) in a one-sample test")
    }
  if (count & !is.na(n2) & obs2 > n2) {
    rlang::abort(message="count obs2 cannot exceed n2 in a two-sample test")
    }
  if (!(tolower(ccontinuity) %in% c("none", "yates", "ha"))) {
    rlang::warn(message=paste0("Ignoring ccontinuity = \"", ccontinuity, "\",\n  ccontinuity must be one of c(\"none\", \"yates\", \"ha\")\n", sep=""))
    }
   # Correct user-erased variable names
  if (trimws(x.name)=="") {
    x.name <- "var 1"
    }
  # Format x.name for output
  x.long.name <- pad.left(paste0(gsub(".*\\$","",x.name)),48)
  x.name <- pad.left(paste0(gsub(".*\\$","",x.name)),12)
  if (trimws(y.name)=="") {
    y.name <- "var 2"
    }
  # Format y.name for output
  y.long.name <- pad.left(paste0(gsub(".*\\$","",y.name)),48)
  y.name <- pad.left(paste0(gsub(".*\\$","",y.name)),12)
  # Create default x.label
  x.label <- paste0(x.name,"=1")
  # Create default y.label
  y.label <- paste0(y.name,"=1")
  # Validate continuity correction selection
  if (length(ccontinuity) > 1) {
    rlang::warn(message="Using ccontinuity = \"none\"")
    ccontinuity <- "none"
    }
  # Continuity correction notice function
  cont.cor.notice <- function(ccontinuity) {
    if (ccontinuity == "yates") {
      rlang::inform(message="\nUsing the Yates continuity correction")
      } 
    if (ccontinuity == "ha" & !is.na(n2)) {
      rlang::inform(message="\nUsing the Hauck-Anderson continuity correction")
      } 
    if (ccontinuity == "ha" & is.na(n2)) {
      rlang::warn(message=paste0("Ignoring ccontinuity argument,\n  Hauck-Anderson continuity correction not relevant in one-sample tests"))
      } 
    }
  # Set up formatting for confidence interval column heading
  cl.width <- nchar(toString(100*conf.level))
  cl.width.pad <- paste0(rep(" ",times=(6-cl.width)),collapse="")
  conf.level.display <- paste0(cl.width.pad,"[",toString(100*conf.level),"% Conf. Interval]",sep="")
  if (cl.width > 5) {
    conf.level.display <- paste0("     [",toString(100*conf.level),"% CI]")
    }
  # Create top bar, mid bar, and bottom bar
  top.bar      <- paste0(
    paste0(
      rep("\U2500",13),
      collapse=""),
    "\U252C",
    paste0(
      rep("\U2500",66),
      collapse=""),
    collapse="")
  table.title  <- paste0(
    pad.spaces(4),
    "Variable \U2502",
    pad.spaces(7),
    "Mean",
    pad.spaces(3),
    "Std. err.",
    pad.spaces(19),
    conf.level.display,
    sep="")
  mid.bar      <- paste0(
    paste0(
      rep("\U2500",13),
      collapse=""),
    "\U253C",
    paste0(
      rep("\U2500",66),
      collapse=""),
    collapse="")
  bottom.bar   <- paste0(
    paste0(
      rep("\U2500",13),
      collapse=""),
    "\U2534",
    paste0(
      rep("\U2500",66),
      collapse=""),
    collapse="")
  # One-sample test
  if (is.na(n2)) {
    if (ccontinuity == "ha") {
      cont.cor.notice(ccontinuity)
      ccontinuity <- "none"
      }
    # Get n, accounting for missing values of x
    n <- n1
    # Format n for very large sample sizes
    n.display <- pad.left(toString(n),14)
    # Calculate the various sample statistics for this test
    if (!count) {
      count.x <- obs1 * n
      p.hat <- obs1
      }
     else {
      count.x <- obs1
      p.hat <- count.x/n
      }
    p0 <- obs2
    p.hat.display <- pad.left(toString(signif(p.hat,8)),11)
    estimate <- p.hat
    theta <- p.hat - p0
    theta.display <- pad.left(toString(signif(theta,8)),11)
    sd.p.hat <- sqrt(p.hat*(1-p.hat))
    sd.p.hat.display <- pad.left(toString(signif(sd.p.hat,7)),11)
    sd.p <- sqrt(p0*(1-p0))
    sd.p.display <- pad.left(toString(signif(sd.p,7)),11)
    se.p.hat <- sd.p.hat/sqrt(n)
    se.p.hat.display <- pad.left(toString(signif(se.p.hat,6)),11)
    se.p <- sd.p/sqrt(n)
    se.p.display <- pad.left(toString(round(se.p,7)),11)
    # Calculate the confidence interval for p, and format for output
    z.crit <- qnorm(p=alpha/2, lower.tail=FALSE)
    ci.p.lb <- p.hat - z.crit*se.p.hat
    ci.p.ub <- p.hat + z.crit*se.p.hat
    ci.p.lb.display <- pad.left(toString(signif(ci.p.lb,7)),11)
    ci.p.ub.display <- pad.left(toString(signif(ci.p.ub,7)),11)
    ci.upper.diff.lb <- upper - theta - z.crit*se.p
    ci.upper.diff.ub <- upper - theta + z.crit*se.p
    ci.upper.diff.lb.display <- pad.left(toString(signif(ci.upper.diff.lb,7)),11)
    ci.upper.diff.ub.display <- pad.left(toString(signif(ci.upper.diff.ub,7)),11)
    ci.lower.diff.lb <- theta + abs(eqv.level) - z.crit*se.p
    ci.lower.diff.ub <- theta + abs(eqv.level) + z.crit*se.p
    ci.lower.diff.lb.display <- pad.left(toString(signif(ci.lower.diff.lb,7)),11)
    ci.lower.diff.ub.display <- pad.left(toString(signif(ci.lower.diff.ub,7)),11)
    # No continuity correction
    if (ccontinuity != "yates") {
      cont.cor <- 0
      }
    # Yates continuity correction for one-sample test
    if (ccontinuity == "yates") {
      cont.cor <- 0.5*(1/n)
      }
    # Calculate and format positivist test statistics
    if (theta >= 0) {        # Sign of theta affects direction of correction
      z.pos <- (theta - cont.cor) / se.p
      }
     else {
      z.pos <- (theta + cont.cor) / se.p
      }
    z.pos.display <- sprintf("%-8.4f",z.pos)
    se.crit <- se.p
    p.pos <- 2*pnorm(q=abs(z.pos), lower.tail=FALSE)
    p.pos.display <- format.extreme.p.vals(p.pos)
    lower <- -1*abs(eqv.level)
    if (eqv.type == "delta") {
      z1 <- (upper - (theta  + cont.cor) ) / se.p
      z2 <- ((theta - cont.cor) + abs(eqv.level)) / se.p
      }
    if (eqv.type == "epsilon") {
      z1 <- upper - z.pos
      z2 <- z.pos + abs(eqv.level)
      }
    z1.display <- sprintf("%-8.4g",z1)
    z2.display <- sprintf("%-8.4g",z2)
    p1 <- pnorm(q=z1, lower.tail=FALSE)
    p1.display <- format.extreme.p.vals(p1)
    p2 <- pnorm(q=z2, lower.tail=FALSE)
    p2.display <- format.extreme.p.vals(p2)
    if (z1 <= -10) {
      z.pad <- pad.spaces(14)
      }
    if (z1 < 0) {
      z.pad <- pad.spaces(15)
      }
    if (z1 >= 0) {
      z.pad <- pad.spaces(16)
      }
    display.width <- 80
    n.obs.pad <- display.width - nchar(trimws(x.long.name)) - nchar(": Number of obs = ") - nchar(n.display)
    if (relevance) {
      pos.title <- "One-sample z test for proportion difference\n"
      z.equals.and.z <- 11
      z.pos.pad <- pad.spaces(display.width - nchar(paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),sep="")) - z.equals.and.z)
      rlang::inform(message=paste0("\nOne-sample relevance z test of sample proportions"))
      rlang::inform(message=paste0("\n",pos.title,sep=""))
      rlang::inform(message=paste0(pad.spaces(n.obs.pad),trimws(x.long.name),": Number of obs = ",n.display,sep=""))
      rlang::inform(message=top.bar)
      rlang::inform(message=table.title)
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(x.name," \U2502",p.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
      rlang::inform(message=bottom.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),z.pos.pad,"z = ",z.pos.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(7),"Ho: \U03B8 = 0",sep=""))
      cont.cor.notice(ccontinuity)
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(29),"Pr(|Z| > |z|) ",p.pos.display,"\n",sep=""))
      }
    neg.title <- "One-sample z test for proportion equivalence\n"
    rlang::inform(message=paste0("\n",neg.title,pad.spaces(n.obs.pad),trimws(x.long.name),": Number of obs = ",n.display,sep=""))
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(x.name," \U2502",p.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
    if (eqv.type == "delta") {
      rlang::inform(message=mid.bar)
      if (upper == abs(eqv.level)) {
        upper.diff <- abs(eqv.level) - theta
        upper.diff.display <- pad.left(signif(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(8),"\U0394-\U03B8\U02C6")
        lower.diff <- theta + abs(eqv.level)
        lower.diff.display <- pad.left(signif(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(8),"\U03B8\U02C6+\U0394")
        delta.display <- trimws(trim0(sprintf("%.4f",signif(abs(eqv.level),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U0394 = ",delta.display,pad.spaces(9 - nchar(delta.display)),"\U0394 expressed in same units as prop(",trimws(x.name),"=1)",sep=""))
        criticalvalue <- se.p*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-5.3g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394 \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: |\U03B8| \U2265 \U0394:")
        }
      if (upper != abs(eqv.level)) {
        upper.diff <- upper - theta
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(7),"\U0394u-\U03B8\U02C6")
        lower.diff <- theta + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(7),"\U03B8\U02C6-\U0394l")
        delta.lower.display <- trimws(trim0(sprintf("%.4f",(signif(-1*abs(eqv.level),5)))))
        delta.upper.display <- trimws(trim0(sprintf("%.4f",signif(abs(upper),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U0394l = ",delta.lower.display,pad.spaces(9 - nchar(delta.lower.display)),"\U0394l expressed in same units as prop(",trimws(x.label),")",sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U0394u =  ",delta.upper.display,pad.spaces(8 - nchar(delta.upper.display)),"\U0394u expressed in same units as prop(",trimws(x.label),")",sep=""))
        criticalvalue <- se.p*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U0394l| \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394u \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:")
        }
      }
    if (eqv.type=="epsilon") {
      if (upper == abs(eqv.level)) {
        epsilon.display <- trimws(trim0(sprintf("%5.3f",signif(abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U03B5 = ",epsilon.display,pad.spaces(9 - nchar(epsilon.display)),"\U03B5 expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5 \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
        }
      if (upper != abs(eqv.level)) {
        epsilon.upper.display <- trimws(trim0(sprintf("%5.3f",signif(upper,5))))
        epsilon.lower.display <- trimws(trim0(sprintf("%5.3f",signif(-1*abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") - ",p0," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5l = ",epsilon.lower.display,pad.spaces(9 - nchar(epsilon.lower.display)),"\U03B5l expressed in units of the z distribution",sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5u =  ",epsilon.upper.display,pad.spaces(8 - nchar(epsilon.upper.display)),"\U03B5u expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U03B5l| \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5u \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
        }
      }
    cont.cor.notice(ccontinuity)
    rlang::inform(message="")
    rlang::inform(message=paste0(pad.spaces(8), "z1 = ", z1.display, z.pad, "z2 = ", z2.display, "\n", sep=""))
    if (upper != abs(eqv.level)) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-\U03B8 \U2264 0",pad.spaces(16),"Ho2: \U03B8-\U0394l \U2264 0"))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-\U03B8 > 0",pad.spaces(16),"Ha2: \U03B8-\U0394l > 0"))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u\U2013Z \U2264 0",pad.spaces(16),"Ho2: Z-\U03B5l \U2264 0"))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u\U2013Z > 0",pad.spaces(16),"Ha2: Z-\U03B5l > 0"))
        }
      }
     else {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-\U03B8 \U2264 0",pad.spaces(17),"Ho2: \U03B8+\U0394 \U2264 0"))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-\U03B8 > 0",pad.spaces(17),"Ha2: \U03B8+\U0394 > 0"))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5\U2013Z \U2264 0",pad.spaces(17),"Ho2: Z+\U03B5 \U2264 0"))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5\U2013Z > 0",pad.spaces(17),"Ha2: Z+\U03B5 > 0"))
        }
      }
    } 
  # Two-sample test
  if (!is.na(n2)) {
    # Get n1 and n2, accounting for missing values of x
    if(!count) {
      px.hat  <- obs1
      nx      <- n1
      count.x <- px.hat*nx
      py.hat  <- obs2
      ny      <- n2
      count.y <- py.hat*ny
      count   <- count.x + count.y
      n       <- nx + ny
      p.hat   <- count/n
      }
     else {
      nx      <- n1
      count.x <- obs1
      px.hat  <- count.x/nx
      ny      <- n2
      count.y <- obs2
      py.hat  <- count.y/ny
      count   <- count.x + count.y
      n       <- nx + ny
      p.hat   <- count/n
      }
    # Format stats for output
    px.hat.display <- pad.left(toString(round(px.hat,7)),11)
    py.hat.display <- pad.left(toString(round(py.hat,7)),11)
    p.hat.display <- pad.left(toString(round(p.hat,7)),11)
    nx.display <- pad.left(toString(round(nx,14)),14)
    ny.display <- pad.left(toString(round(ny,14)),14)
    n.display  <- pad.left(toString(round(n,14)),14)
    estimate1 <- px.hat
    estimate2 <- py.hat
    theta.hat <- px.hat - py.hat
    theta.hat.display <- pad.left(toString(round(theta.hat,7)),11)
    se.px.hat <- sqrt((px.hat * (1 - px.hat)) / nx)
    se.px.hat.display <- pad.left(toString(round(se.px.hat,7)),11)
    se.py.hat <- sqrt((py.hat * (1 - py.hat)) / ny)
    se.py.hat.display <- pad.left(toString(round(se.py.hat,7)),11)
    se.p.hat <- sqrt(p.hat * (1 - p.hat) * ((1/nx) + (1/ny)))
    se.p.hat.display <- pad.left(toString(round(se.p.hat,7)),11)
    se.theta.hat <- sqrt(se.px.hat^2 + se.py.hat^2)
    se.theta.hat.display <- pad.left(toString(round(se.theta.hat,7)),11)
    # No continuity correction
    if (ccontinuity != "yates") {
      cont.cor <- 0
      }
    # Yates continuity correction
    if (ccontinuity == "yates") {
      cont.cor <- 0.5*(1.0/nx + 1.0/ny)
      }
    # Hauck-Anderson continuity correction
    if (ccontinuity == "ha") {
      cont.cor <- (1.0/(2.0 * min(nx,ny)))
      se.p.hat <- sqrt( ((px.hat * (1 - px.hat)) / (nx - 1)) + ((py.hat * (1 - py.hat)) / (ny - 1)) )
      se.p.hat.display <- pad.left(toString(round(se.p.hat,7)),11)
      }
    # Calculate the confidence intervals for x, y, and theta, and format for output
    z.crit <- qnorm(p=alpha/2, lower.tail=FALSE)
    ci.p1.lb <- px.hat - z.crit*se.px.hat
    ci.p1.ub <- px.hat + z.crit*se.px.hat
    ci.p1.lb.display <- pad.left(sprintf("%011.7g",ci.p1.lb),11)
    ci.p1.ub.display <- pad.left(sprintf("%011.7g",ci.p1.ub),11)
    ci.p2.lb <- py.hat - z.crit*se.py.hat
    ci.p2.ub <- py.hat + z.crit*se.py.hat
    ci.p2.lb.display <- pad.left(sprintf("%011.7g",ci.p2.lb),11)
    ci.p2.ub.display <- pad.left(sprintf("%011.7g",ci.p2.ub),11)
    ci.p.lb <- theta.hat - z.crit * se.p.hat
    ci.p.ub <- theta.hat + z.crit * se.p.hat
    ci.p.lb.display <- pad.left(toString(round(ci.p.lb,7)),11)
    ci.p.ub.display <- pad.left(toString(round(ci.p.ub,7)),11)
    ci.theta.lb <- theta.hat - z.crit * sqrt(se.px.hat^2 + se.py.hat^2)
    ci.theta.ub <- theta.hat + z.crit * sqrt(se.px.hat^2 + se.py.hat^2)
    ci.theta.lb.display <- pad.left(toString(round(ci.theta.lb,7)),11)
    ci.theta.ub.display <- pad.left(toString(round(ci.theta.ub,7)),11)
    ci.upper.diff.lb <- upper - theta.hat - z.crit*se.p.hat
    ci.upper.diff.ub <- upper - theta.hat + z.crit*se.p.hat
    ci.upper.diff.lb.display <- pad.left(toString(round(ci.upper.diff.lb,7)),11)
    ci.upper.diff.ub.display <- pad.left(toString(round(ci.upper.diff.ub,7)),11)
    ci.lower.diff.lb <- theta.hat + abs(eqv.level) - z.crit*se.p.hat
    ci.lower.diff.ub <- theta.hat + abs(eqv.level) + z.crit*se.p.hat
    ci.lower.diff.lb.display <- pad.left(toString(round(ci.lower.diff.lb,7)),11)
    ci.lower.diff.ub.display <- pad.left(toString(round(ci.lower.diff.ub,7)),11)
    # Calculate and format positivist test statistics
    if (theta.hat >= 0) {        # Sign of theta affects direction of correction
      z.pos <- (theta.hat - cont.cor) / se.p.hat
      }
     else {
      z.pos <- (theta.hat + cont.cor) / se.p.hat
      }
    z.pos.display <- sprintf("%-8.4f",z.pos)
    z.pos.table   <- sprintf("%-8.2f",z.pos)
    se.crit <- se.p.hat
    p.pos <- 2*pnorm(q=abs(z.pos), lower.tail=FALSE)
    p.pos.display <- format.extreme.p.vals(p.pos)
    lower <- -1*abs(eqv.level)
    if (eqv.type == "delta") {
      z1 <- (upper - (theta.hat  + cont.cor) ) / se.p.hat
      z2 <- ((theta.hat - cont.cor) + abs(eqv.level)) / se.p.hat
      }
    if (eqv.type == "epsilon") {
      z1 <- upper - z.pos
      z2 <- z.pos + abs(eqv.level)
      }
    z1.display <- sprintf("%-8.4g",z1)
    z2.display <- sprintf("%-8.4g",z2)
    p1 <- pnorm(q=z1, lower.tail=FALSE)
    p1.display <- format.extreme.p.vals(p1)
    p2 <- pnorm(q=z2, lower.tail=FALSE)
    p2.display <- format.extreme.p.vals(p2)
    if (z1 <= -10) {
      z.pad <- pad.spaces(16)
      }
    if (z1 < 0) {
      z.pad <- pad.spaces(17)
      }
    if (z1 >= 0) {
      z.pad <- pad.spaces(16)
    }
    nx.obs.pad <- 48 - nchar(trimws(x.long.name))
    ny.obs.pad <- 48 - nchar(trimws(y.long.name))
    z.pos.pad <- 65 - nchar(paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.name),") - prop(",trimws(y.name),")","= ",trimws(theta.hat.display)))
    if (z.pos.pad < 0 ) {
      z.pos.pad <- paste0("\n",pad.spaces(11))
      }
     else {
      z.pos.pad <- pad.spaces(z.pos.pad)
      }
    if (relevance) {
      pos.title <- "Two-sample z test for proportion difference\n"
      rlang::inform(message="\nTwo-sample relevance z test of sample proportions")
      rlang::inform(message=paste0("\n",pos.title,sep=""))
      rlang::inform(message=paste0(pad.spaces(nx.obs.pad),trimws(x.name),": Number of obs = ",nx.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(ny.obs.pad),trimws(y.name),": Number of obs = ",ny.display,sep=""))
      rlang::inform(message=top.bar)
      rlang::inform(message=table.title)
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(x.name," \U2502",px.hat.display,se.px.hat.display,pad.spaces(22),ci.p1.lb.display,ci.p1.ub.display,sep=""))
      rlang::inform(message=paste0(y.name," \U2502",py.hat.display,se.py.hat.display,pad.spaces(22),ci.p2.lb.display,ci.p2.ub.display,sep=""))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 |",theta.hat.display,se.theta.hat.display,pad.spaces(22),ci.theta.lb.display,ci.theta.ub.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(13),"\U2502  under Ho:",se.p.hat.display,pad.spaces(3),z.pos.table,pad.spaces(3),trimws(sprintf("%8.4f",p.pos)),sep=""))
      rlang::inform(message=bottom.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.name),"=1) - prop(",trimws(y.name),"=1)","= ",trimws(theta.hat.display),z.pos.pad,"z = ",z.pos.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(7), "Ho: \U03B8 = 0",sep=""))
      cont.cor.notice(ccontinuity)
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(29),"Pr(|Z| > |z|) ",p.pos.display,"\n",sep=""))
      }
    neg.title <- "Two-sample z test for proportion equivalence\n"
    rlang::inform(message=paste0("\n",neg.title,sep=""))
    rlang::inform(message=paste0(pad.spaces(nx.obs.pad),trimws(x.long.name),": Number of obs = ",nx.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(ny.obs.pad),trimws(y.long.name),": Number of obs = ",ny.display,sep=""))
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(x.name," \U2502",px.hat.display,se.px.hat.display,pad.spaces(22),ci.p1.lb.display,ci.p1.ub.display,sep=""))
    rlang::inform(message=paste0(y.name," \U2502",py.hat.display,se.py.hat.display,pad.spaces(22),ci.p2.lb.display,ci.p2.ub.display,sep=""))
    if (eqv.type == "delta") {
      rlang::inform(message=mid.bar)
      if (upper == abs(eqv.level)) {
        upper.diff <- abs(eqv.level) - theta.hat
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(8),"\U0394-\U03B8\U02C6")
        lower.diff <- theta.hat + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(8),"\U03B8\U02C6+\U0394")
        delta.display <- trimws(trim0(sprintf("%.4f",round(abs(eqv.level),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.hat.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.hat.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.name),"=1) - prop(",trimws(y.name),"=1) = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U0394 = ",delta.display,pad.spaces(9 - nchar(delta.display)),"\U0394 expressed in same units as prop(",trimws(x.name),"=1)",sep=""))
        criticalvalue <- se.p.hat*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394 \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: |\U03B8| \U2265 \U0394:")
        }
      if (upper != abs(eqv.level)) {
        upper.diff <- upper - theta.hat
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(7),"\U0394u-\U03B8\U02C6")
        lower.diff <- theta.hat + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(7),"\U03B8\U02C6-\U0394l")
        delta.lower.display <- trimws(trim0(sprintf("%.4f",(round(-1*abs(eqv.level),5)))))
        delta.upper.display <- trimws(trim0(sprintf("%.4f",round(abs(upper),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.hat.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.hat.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(8),"\U03B8\U02C6 = prop(",trimws(x.name),") - prop(",trimws(y.name),") = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U0394l = ",delta.lower.display,pad.spaces(11 - nchar(delta.lower.display)),"\U0394l expressed in same units as prop(",trimws(x.name),"=1)",sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U0394u =  ",delta.upper.display,pad.spaces(10 - nchar(delta.upper.display)),"\U0394u expressed in same units as prop(",trimws(x.name),"=1)",sep=""))
        criticalvalue <- se.p.hat*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U0394l| \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394u \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:")
        }
      }
    if (eqv.type=="epsilon") {
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 \U2502",theta.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
      if (upper == abs(eqv.level)) {
        epsilon.display <- trimws(trim0(sprintf("%5.3f",signif(abs(eqv.level),5))))
        rlang::inform(message=top.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.name),") - prop(",trimws(y.name),") = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U03B5 = ",epsilon.display,pad.spaces(9 - nchar(epsilon.display)),"\U03B5 expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5 \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
        }
      if (upper != abs(eqv.level)) {
        epsilon.upper.display <- trimws(trim0(sprintf("%5.3f",signif(upper,5))))
        epsilon.lower.display <- trimws(trim0(sprintf("%5.3f",signif(-1*abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.name),") - prop(",trimws(y.name),") = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U03B5l = ",epsilon.lower.display,pad.spaces(11 - nchar(epsilon.lower.display)),"\U03B5l expressed in units of the z distribution",sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U03B5u =  ",epsilon.upper.display,pad.spaces(10 - nchar(epsilon.upper.display)),"\U03B5u expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U03B5l| \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5u \U2264 z-crit (", criticalvalue.display, "). See help(tost.pri).",sep=""))
          }
        rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
        }
      }
    cont.cor.notice(ccontinuity)
    rlang::inform(message="")
    rlang::inform(message=paste0(pad.spaces(8), "z1 = ", z1.display, z.pad, "z2 = ", z2.display, "\n", sep=""))
    if (upper != abs(eqv.level)) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-\U03B8 \U2264 0",pad.spaces(16),"Ho2: \U03B8-\U0394l \U2264 0",sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-\U03B8 > 0",pad.spaces(16),"Ha2: \U03B8-\U0394l > 0", sep=""))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u\U2013Z \U2264 0",pad.spaces(16),"Ho2: T-\U03B5l \U2264 0",sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u\U2013Z > 0",pad.spaces(16),"Ha2: T-\U03B5l > 0",sep=""))
        }
      }
     else {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-\U03B8 \U2264 0",pad.spaces(17),"Ho2: \U03B8+\U0394 \U2264 0",sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-\U03B8 > 0",pad.spaces(17),"Ha2: \U03B8+\U0394 > 0",sep=""))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5\U2013Z \U2264 0",pad.spaces(17),"Ho2: Z+\U03B5 \U2264 0",sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5\U2013Z > 0",pad.spaces(17),"Ha2: Z+\U03B5 > 0",sep=""))
        }
      }
    }
  rlang::inform(message=paste0(pad.spaces(3),"Pr(Z > z1) ",p1.display,pad.spaces(18 - nchar(p1.display)),"Pr(Z > z2) ",p2.display))
  if (relevance) {
    if (upper != abs(eqv.level)) {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U0394l = ", trim0(delta.lower.display),", and \U0394u = ", trim0(delta.upper.display),":",sep=""))
        }
       else {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U03B5l = ", trim0(epsilon.lower.display),", and \U03B5u = ", trim0(epsilon.upper.display),":",sep=""))
        }
      }
     else {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", and \U0394 = ", trim0(delta.display),":",sep=""))
        }
       else {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", and \U03B5 = ",trim0(epsilon.display),":",sep=""))
        }
      }
    if (p.pos <= alpha) {
      pos.decision <- "Reject"
      }
     else {
      pos.decision <- "Fail to reject"
      }
    if (p1 <= alpha & p2 <= alpha) {
      neg.decision <- "Reject"
      }
     else {
      neg.decision <- "Fail to reject"
      }
    rel.conclusion <- "Indeterminate (underpowered test)"
    if (pos.decision == "Reject" & neg.decision == "Fail to reject") {
      rel.conclusion <- "Relevant difference"
      }
     else {
      if (pos.decision == "Fail to reject" & neg.decision == "Reject") {
        rel.conclusion = "Equivalence"
        }
       else {
        if (pos.decision == "Reject" & neg.decision == "Reject") {
          rel.conclusion <- "Trivial difference (overpowered test)"
          }
        }
      }
    rlang::inform(message=paste0("  Ho test for difference:  ",pos.decision))
    rlang::inform(message=paste0("  Ho test for equivalence: ",neg.decision,"\n"))
    rlang::inform(message=paste0("Conclusion from combined tests: ",rel.conclusion,"\n"))
    }

  out <- list() 
  # Prepare return stuff for one-sample test
  if (is.na(n2)) {
    if (!relevance) {
      out$statistics <- c(z1,z2)
      names(out$statistics) <- c("z1","z2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
    else {
      out$statistics <- c(z1,z2,z.pos)
      names(out$statistics) <- c("z1","z2","z")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
     }
    out$proportion <- p.hat
    names(out$proportion) <- paste0("prop(",trimws(x.name),"=1)",collapse="")
    out$sample_size <- n
    names(out$sample_size) <- "n"
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  # Prepare return stuff for two-sample test
  if (!is.na(n2)) {
    if (!relevance) {
      out$statistics <- c(z1,z2)
      names(out$statistics) <- c("z1","z2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
     else {
      out$statistics <- c(z1,z2,z.pos)
      names(out$statistics) <- c("z1","z2","z")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
      }
    out$proportion <- c(px.hat, py.hat, p.hat)
    names(out$proportion) <- c(paste0("prop(",trimws(x.name),"=1)",collapse=""), paste0("prop(",trimws(y.name),"=1)",collapse=""), paste0("prop(",trimws(x.name),"=1 + ",trimws(y.name),"=1)", collapse=""))
    out$sample_size <- c(nx,ny,nx+ny)
    names(out$sample_size) <- c("nx","ny","nx+ny")
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  invisible(out)
  }
