################################################################################
# Performs immediate paired z test for equivalence binary data
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9 
# Date: February 6, 2026

equivalence.types <- c("delta", "epsilon")

continuity.correction.methods <- c("none", "yates", "edwards")

tost.mcci <- function(
    a=NA, b=NA, c=NA, d=NA,
    eqv.type=equivalence.types,
    eqv.level=1,
    upper=NA,
    ccontinuity=continuity.correction.methods,
    conf.level=0.95,
    relevance=TRUE) {
  # Validate that a through d are positive integers
  if (a%%1 != 0 | a < 1) {
    stop("a must be a positive integer")
    }
  if (b%%1 != 0 | b < 1) {
    stop("b must be a positive integer")
    }
  if (c%%1 != 0 | c < 1) {
    stop("c must be a positive integer")
    }
  if (d%%1 != 0 | d < 1) {
    stop("d must be a positive integer")
    }

  # Validate conf.level and create alpha
  alpha <- 1 - conf.level
  if (alpha <= 0 | alpha >= 1) {
    stop("conf.level must be >0 and <1")
    }
  alpha.display <- sub("0+$", "", as.character(alpha))
  # Validate and sanitize eqv.type
  if (length(eqv.type) > 1) {
    eqv.type <- "delta"
    }
  eqv.type <- tolower(eqv.type)
  if ( eqv.type != "delta" & eqv.type != "epsilon" ) {
    stop("option eqv.type() must be either delta or epsilon")
    }

  # Validate eqv.level and upper
  if (!is.na(upper) & upper<0) {
    stop("option upper must be a positive real value")
    }
  if (is.na(upper)) {
    upper <- 0
    }
  if (upper==0 | upper == abs(eqv.level)) {
    upper <- abs(eqv.level)
    lower <- abs(eqv.level)
    }
  if (upper>0) {
    lower <- abs(eqv.level)
    }
  lower.display <- sprintf("%6.4f",lower)
  upper.display <- sprintf("%6.4f",upper)

  # Validate and sanitize ccontinuity
  if (length(ccontinuity) > 1) {
    ccontinuity <- "none"
    }
  ccontinuity <- tolower(ccontinuity)

  # Set default decision values
  pos.decision <- "Reject"
  neg.decision <- "Reject"

  # deal with continuity corrections
  if (length(ccontinuity)>1) {
    ccontinuity <- "none"
    }
  if (!(ccontinuity %in% continuity.correction.methods)) {
    stop("ccontinuity must be one of continuity.correction.methods")
    }
  cont.cor = 0
  cont.cor.notice <- "\n"
  if (ccontinuity=="yates") {
    cont.cor <- 0.5
    cont.cor.notice <- "\nUsing the Yates continuity correction\n\n"
    }
  if (ccontinuity=="edwards") {
    cont.cor <- 1
    cont.cor.notice <- "\nUsing the Edwards continuity correction\n\n"
    }
  # Create top bar, table.title, mid bar, and bottom bar for output
  top.bar     <- paste0(
                   paste0(rep("\U2500",17),collapse=""),
                   "\U252C",
                   paste0(rep("\U2500",24),collapse=""),
                   "\U252C",
                   paste0(rep("\U2500",11),collapse=""), collapse="")
  table.title <- paste0(
                   pad.spaces(17),
                   "\U2502",
                   pad.spaces(8),
                   "Controls",
                   pad.spaces(8),
                   "\U2502\nCases",
                   pad.spaces(12),
                   "\U2502",
                   pad.spaces(3),
                   "Exposed",
                   pad.spaces(3),
                   "Unexposed",
                   pad.spaces(2),
                   "\U2502",
                   pad.spaces(6),
                   "Total", collapse="") 
  mid.bar     <- paste0(
                   eval(paste0(rep("\U2500",17),collapse="")),
                   "\U253C",
                   eval(paste0(rep("\U2500",24),collapse="")),
                   "\U253C",
                   eval(paste0(rep("\U2500",11),collapse="")), collapse="")
  bottom.bar  <- paste0(
                   paste0(rep("\U2500",17),collapse=""),
                   "\U2534",
                   paste0(rep("\U2500",24),collapse=""),
                   "\U2534",
                   paste0(rep("\U2500",11),collapse=""), collapse="")
  chisq.pos   <- ((abs(b - c) - cont.cor)^2)/(b + c)  #McNemar's chisq test statistic 
  p.pos = pchisq(chisq.pos,df=1,lower.tail=FALSE)     # p value for same
  if (p.pos > alpha) {
    pos.decision <- "Fail to reject"
    }
  # Conduct positivist test as requested 
  if (relevance) {
    rlang::inform(message="\nRelevance test for paired binary data")
    rlang::inform(message="\nMcNemar\U2019s test for difference in paired binary data")
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(
          pad.left("Exposed",16),
          " \U2502 ", 
          pad.left(a,9),
          pad.spaces(3),
          pad.left(b,9),
          "  \U2502 ",
          pad.left(a+b,9), collapse=""))
    rlang::inform(message=paste0(
          pad.left("Unexposed",16),
          " \U2502 ", 
          pad.left(c,9),
          pad.spaces(3),
          pad.left(d,9),
          "  \U2502 ",
          pad.left(c+d,9), collapse=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(
          pad.left("Total",16),
          " \U2502 ", 
          pad.left(a+c,9),
          pad.spaces(3),
          pad.left(b+d,9),
          "  \U2502 ",
          pad.left(a+b+c+d,9), collapse=""))
    rlang::inform(message=bottom.bar)
    chisq.pos.display <- pad.left(sprintf("%8.4f",chisq.pos),11)
    p.pos.display <- format.extreme.p.vals(p.pos)
    rlang::inform(message=paste0("McNemar\U2019s \U03C7\U00B2(1) = ", chisq.pos.display, pad.spaces(6),"Pr(X > \U03C7\U00B2) ", p.pos.display,"", sep=""))
    rlang::inform(message="Ho: Pr(b) = Pr(c)")
    rlang::inform(message="Ha: Pr(b) \U2260 Pr(c)")
    rlang::inform(message=cont.cor.notice)
    }
  rlang::inform(message="\nWald test for equivalence in paired binary data")
  rlang::inform(message=top.bar)
  rlang::inform(message=table.title)
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(
        pad.left("Exposed",16),
        " \U2502 ", pad.left(a,9),
        pad.spaces(3),
        pad.left(b,9),
        "  \U2502 ",
        pad.left(a+b,9), collapse=""))
  rlang::inform(message=paste0(
        pad.left("Unexposed",16),
        " \U2502 ", pad.left(c,9),
        pad.spaces(3),
        pad.left(d,9),
        "  \U2502 ",
        pad.left(c+d,9), collapse=""))
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(
        pad.left("Total",16),
        " \U2502 ", pad.left(a+c,9),
        pad.spaces(3),
        pad.left(b+d,9),
        "  \U2502 ",
        pad.left(a+b+c+d,9), collapse=""))
  rlang::inform(message=bottom.bar)
  n     <- a + b + c + d
  d     <- b + c
  low   <- min(b,c)
  theta <- b/n - c/n
  diff  <- b - c
  var.theta <- (d - (n * theta^2))
  se.theta <- sqrt(var.theta)
  se.theta.display <- trimws(toString(sprintf("%8.4f", se.theta)))
  z.pos <- sign(diff)*(abs(diff) - cont.cor) / se.theta
  if (eqv.type=="delta") {
    z1 <- ((n*upper) - (diff - cont.cor))/ se.theta 
    z2 <- ((diff + cont.cor) + (n*lower))/ se.theta
    }
  if (eqv.type=="epsilon") {
    z1 <- upper - sign(diff)*(abs(diff) - cont.cor) / se.theta
    z2 <- sign(diff)*(abs(diff) + cont.cor) / se.theta + lower
    }
  z1.display <- trimws(toString(sprintf("%-8.4g",z1)))
  z2.display <- trimws(toString(sprintf("%-8.4g",z2)))
  neg.z.stats.out <- paste0(pad.spaces(8), "z1 = ", z1.display, pad.spaces(19), "z2 = ", z2.display, collapse="")
  p1 <- pnorm(z1, lower.tail=FALSE)
  p2 <- pnorm(z2, lower.tail=FALSE)
  p1.display <- format.extreme.p.vals(p1)
  p2.display <- format.extreme.p.vals(p2)
  neg.p.vals.out <- paste0(pad.spaces(3), "Pr(Z > z1) ", p1.display, pad.spaces(10), "Pr(Z > z2) ", p2.display, collapse="")
  if (p1>alpha | p2>alpha) {
    neg.decision <- "Fail to reject"
    }
  rlang::inform(message=paste0(pad.spaces(6), "diff = ", b, " - ", c," = ", diff, sep=""))
  rlang::inform(message=paste0(pad.spaces(4), "s.e. \U03B8 = ", se.theta.display, sep=""))
  if (eqv.type=="delta") {
    if (upper==lower) {
      rlang::inform(message=paste0(pad.spaces(9),"\U0394 = ", lower.display, pad.spaces(3), "\U0394 expressed in units of probability", sep=""))
      }
     else {
      rlang::inform(message=paste0(pad.spaces(8),"\U0394l = -", lower.display, pad.spaces(2), "\U0394l expressed in units of probability", sep=""))
      rlang::inform(message=paste0(pad.spaces(8),"\U0394u =  ", upper.display, pad.spaces(2), "\U0394u expressed in units of probability", sep=""))
      }
    criticalvalue <- (se.theta * qnorm(1-alpha))/n
    criticalvalue.display <- trimws(toString(sprintf("%-7.4f",criticalvalue)))
    if (upper==lower & lower<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U0394 \U2264 z-crit*s.e/n (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper!=lower & lower<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |\U0394l| \U2264 z-crit*s.e/n (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper!=lower & upper<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U0394u \U2264 z-crit*s.e/n (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper==lower) {
      rlang::inform(message="\nHo: |\U03B8| \U2264 \U0394:") 
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3), "Ho1: \U0394-\U03B8 \U2264 0", pad.spaces(17), "Ho2: \U03B8+\U0394 \U2264 0", sep=""))
      rlang::inform(message=paste0(pad.spaces(3), "Ha1: \U0394-\U03B8 > 0", pad.spaces(17), "Ha2: \U03B8+\U0394 > 0", sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    if (upper!=lower) {
      rlang::inform(message="\nHo: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:") 
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3), "Ho1: \U0394u-\U03B8 \U2264 0", pad.spaces(16), "Ho2: \U03B8-\U0394l \U2264 0", sep=""))
      rlang::inform(message=paste0(pad.spaces(3), "Ha1: \U0394u-\U03B8 > 0", pad.spaces(16), "Ha2: \U03B8-\U0394l > 0", sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }
  if (eqv.type=="epsilon") {
    if (upper==lower) {
      rlang::inform(message=paste0(pad.spaces(10), "\U03B5 = ", lower.display, pad.spaces(3), "\U03B5 expressed in units of the z distribution", sep=""))
      }
    if (upper!=lower) {
      rlang::inform(message=paste0(pad.spaces(9), "\U03B5l = -", lower.display, pad.spaces(2), "\U03B5l expressed in units of the z distribution", sep=""))
      rlang::inform(message=paste0(pad.spaces(9), "\U03B5u =  ", upper.display, pad.spaces(2), "\U03B5u expressed in units of the z distribution", sep=""))
      }
    criticalvalue = qnorm(conf.level)
    criticalvalue.display <- trimws(toString(sprintf("%-5.3f",criticalvalue)))
    if (upper==lower & lower<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U03B5 \U2264 z-crit (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper!=lower & lower<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |\U03B5l| \U2264 z-crit (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper!=lower & upper<=criticalvalue) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U03B5u \U2264 z-crit (", criticalvalue.display, "). See help(tost.mcci).", sep=""))
      }
    if (upper==lower) {
      rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3), "Ho1: \U03B5-Z \U2264 0", pad.spaces(17), "Ho2: Z+\U03B5 \U2264 0", sep=""))
      rlang::inform(message=paste0(pad.spaces(3), "Ha1: \U03B5-Z > 0", pad.spaces(17), "Ha2: Z+\U03B5 > 0", sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    if (upper!=lower) {
      rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
      rlang::inform(message=cont.cor.notice)
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3), "Ho1: \U03B5u-Z \U2264 0", pad.spaces(17), "Ho2: Z-\U03B5l \U2264 0", sep=""))
      rlang::inform(message=paste0(pad.spaces(3), "Ha1: \U03B5u-Z > 0", pad.spaces(17), "Ha2: Z-\U03B5l > 0", sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }  
  if (relevance) {
    lower.display <- sub("0+$", "", as.character(lower.display))
    upper.display <- sub("0+$", "", as.character(upper.display))
    if (upper==lower) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display,", and \U0394 = ", lower.display, ":", sep=""))
       }
      if (eqv.type=="epsilon") {
       rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display,", and \U03B5 = ", upper.display, ":", sep=""))
       }
      }
    if (upper!=lower) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display, ", \U0394l = -", lower.display, ", and \U0394u = ", upper.display, ":", sep=""))
        }
      if (eqv.type=="epsilon") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ", alpha.display, ", \U03B5l = -", lower.display, ", and \U03B5u = ", upper.display, ":", sep=""))
        }
      }
    rlang::inform(message=paste0("  Ho test for difference:  ", pos.decision, sep=""))
    rlang::inform(message=paste0("  Ho test for equivalence: ", neg.decision, sep=""))
    if (pos.decision == "Reject" & neg.decision == "Reject") {
      relevance.conclusion <- "Trivial difference (overpowered test)"
      }
    if (pos.decision == "Reject" & neg.decision == "Fail to reject") {
      relevance.conclusion <- "Relevant difference"
      }
    if (pos.decision == "Fail to reject" & neg.decision == "Reject") {
      relevance.conclusion <- "Equivalence"
      }
    if (pos.decision == "Fail to reject" & neg.decision == "Fail to reject") {
      relevance.conclusion <- "Indeterminate (underpowered test)"
     }
     rlang::inform(message=paste0("\nConclusion from combined tests: ",relevance.conclusion,"", sep=""))
    }

  out <- list() 
  # Prepare return values
  if (!relevance) {
    out$statistics <- c(z1,z2)
    names(out$statistics) <- c("z1","z2")
    out$p.values <- c(p1,p2)
    names(out$p.values) <- c("p1","p2")
    }
  else {
    out$statistics <- c(z1,z2,z.pos)
    names(out$statistics) <- c("z1","z2","z")
    out$p.values <- c(p1,p2,p.pos)
    names(out$p.values) <- c("p1","p2","p")
    }
  out$estimate <- diff/n
  if (eqv.type=="delta") {
    if (upper==abs(eqv.level)) {
      out$threshold <- lower
      names(out$threshold) <- "\U394"
      }
     else {
      out$threshold <- c(upper, lower)
      names(out$threshold) <- c("\U0394u", "\U0394l")
      }
     }
    else {
     if (upper==abs(eqv.level)) {
       out$threshold <- lower
       names(out$threshold) <- "\U3B5"
       }
      else {
        out$threshold <- c(upper, lower)
        names(out$threshold) <- c("\U03B5u", "\U03B5l")
       }
     }
  if(relevance) {
    out$conclusion <- relevance.conclusion
    names(out$conclusion) <- "relevance conclusion"
    }

  invisible(out)
  }
