library(testthat)
library(survival)
library(dplyr)

# Cox model for testing
cox_fit <- coxph(Surv(AVAL, EVENT) ~ TRT01P, ties = "exact", data = codebreak200)

test_that("tipping_point_model_free returns correct structure for random sampling", {
  expect_warning(res <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "sotorasib",
    J = 2,
    tipping_range = c(20, 40),
    cox_fit = cox_fit,
    method = "random sampling",
    seed = 123
  ), "Tipping point not found, please check 'tipping_range'.")

  expect_s3_class(res, "tipse")
  expect_true(all(c(
    "original_data", "original_HR", "reason_to_impute", "arm_to_impute",
    "method_to_impute", "imputation_results", "imputation_data"
  ) %in% names(res)))

  # Check imputation_results structure
  expect_true(all(c("HR", "HR_upperCI", "HR_lowerCI", "parameter", "tipping_point") %in% names(res$imputation_results)))
})


test_that("tipping_point_model_free throws error for invalid cox_fit", {
  expect_error(
    tipping_point_model_free(
      dat = codebreak200,
      reason = "Early dropout",
      impute = "sotorasib",
      J = 2,
      tipping_range = c(20, 40),
      cox_fit = "not_a_coxph",
      method = "random sampling"
    ),
    "Argument 'cox_fit' must be a valid cox model object"
  )
})

test_that("tipping_point_model_free throws error for invalid impute arm", {
  expect_error(
    tipping_point_model_free(
      dat = codebreak200,
      reason = "Early dropout",
      impute = "invalid_arm",
      J = 2,
      tipping_range = c(20, 40),
      cox_fit = cox_fit,
      method = "random sampling"
    ),
    "Argument 'impute' must be one of the arms"
  )
})

test_that("tipping_point_model_free throws error when reason is empty", {
  expect_error(
    tipping_point_model_free(
      dat = codebreak200,
      reason = character(0),
      impute = "sotorasib",
      J = 2,
      tipping_range = c(20, 40),
      cox_fit = cox_fit,
      method = "random sampling"
    ),
    "Argument 'reason' must specify at least one censoring reason"
  )
})

test_that("tipping_point_model_free throws error when percentile_range is missing for random sampling", {
  expect_error(
    tipping_point_model_free(
      dat = codebreak200,
      reason = "Early dropout",
      impute = "sotorasib",
      J = 2,
      tipping_range = NULL,
      cox_fit = cox_fit,
      method = "random sampling"
    ),
    "tipping_range"
  )
})

test_that("tipping_point_model_free throws error when npts_range is missing for deterministic sampling", {
  expect_error(
    tipping_point_model_free(
      dat = codebreak200,
      reason = "Early dropout",
      impute = "sotorasib",
      J = 2,
      tipping_range = NULL,
      cox_fit = cox_fit,
      method = "deterministic sampling"
    ),
    "tipping_range"
  )
})

test_that("tipping_point_model_free sets tipping_point flag correctly", {
  res <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "docetaxel",
    J = 2,
    tipping_range = seq(10, 90, by = 10),
    cox_fit = cox_fit,
    method = "random sampling",
    seed = 123
  )

  expect_true("tipping_point" %in% names(res$imputation_results))
  expect_true(any(res$imputation_results$tipping_point %in% c(TRUE, FALSE)))
})

test_that("tipping_point_model_free respects seed for reproducibility", {
  res1 <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "docetaxel",
    J = 2,
    tipping_range = seq(10, 90, by = 10),
    cox_fit = cox_fit,
    method = "random sampling",
    seed = 123
  )

  res2 <- tipping_point_model_free(
    dat = codebreak200,
    reason = "Early dropout",
    impute = "docetaxel",
    J = 2,
    tipping_range = seq(10, 90, by = 10),
    cox_fit = cox_fit,
    method = "random sampling",
    seed = 123
  )

  expect_equal(res1$imputation_results, res2$imputation_results)
})
